/*************************************************************************
 *
 *   sfiUserLib.c  - Library of routines for the user to write for 
 *                   Aysyncronous readout and buffering of events
 *                   from FASTBUS. Use RAM LIST PROGRAMMING
 *
 *
 *
 */


#include "sfiLib.h"
#include "dmaPList.h"
#include "../dcpu.h"

#define SFI_ADDR 0xe00000
#define IRQ_SOURCE  0x10

#define ADC_SLOT 15
#define SCANMASK 0x8000
#define RAM_ADDR 0x100

#define MAX_NUM_EVENTS    400
#define MAX_SIZE_EVENTS   2048      /* Size in Bytes */


/* 0: External Mode   1:Trigger Supervisor Mode */
#define TS_MODE  0

/* Input and Output Partions for SFI Readout */
ROL_MEM_ID sfiIN, sfiOUT;

/* Dual CPU variables needed to access */
extern DCPU_CONTROL *dcpuCntlPtr;
extern DCPU_CONTROL *dcpuRemCntlPtr;
extern int dcpuReady;
extern int dcpuSize;   /* Buffer size in Bytes */

/* system clock */
extern int vxTicks;

void 
SFIRamLoad () 
{
  SFI_RAM_LOAD_ENABLE(RAM_ADDR); /* Setup Sequencer for RAM List Loading at RAM_ADDR */

  /* Load the List */
  FBWC(ADC_SLOT,0,0x400);        /* Address ADC CSR0 Write 0x400 and release (Load Next Event) */
  FBAD(ADC_SLOT,0);              /* Address ADC  DSR0 */
  FBBR(100);                     /* Start Block Read (And Clear Word Counter) */
  FBREL;                         /* Release the Bus */

  FBSWC;                         /* Store Final Word Count in FIFO */

  FBEND;                         /* End of List */
  SFI_WAIT;                      /* Wait a little */
  SFI_RAM_LOAD_DISABLE;          /* Re-enable Sequencer */
 
}

/* Initialize Memory partitions for managing events */
void
sfiUserInit()
{
  unsigned int res, laddr;

  /* Setup SFI pointers to Enable triggers, and establish
     DMA address offsets */
  res = (unsigned long) sysBusToLocalAdrs(0x39,(char *)SFI_ADDR,(char **)&laddr);
  if (res != 0) {
     printf("sfiUserInit: Error Initializing SFI res=%d \n",res);
  } else {
     printf("sfiUserInit: Calling InitSFI() routine with laddr=0x%x.\n",laddr);
     InitSFI(laddr);
  }


  /* Setup Buffer memory to store events */
  dmaPFreeAll();
  sfiIN  = dmaPCreate("sfiIN",MAX_SIZE_EVENTS,MAX_NUM_EVENTS,0);
  sfiOUT = dmaPCreate("sfiOUT",0,0,0);

  dmaPStatsAll();
}


void
sfiUserDownload()
{

  /* Reinitialize the Buffer memory */
  dmaPReInitAll();

  /* Initialize AUX Card */
  sfiAuxInit();

  /* Initialize SFI Interrupt interface - use defaults */
  sfiIntInit(0,0,0,0);

  /* Connect User Trigger Routine */
  sfiIntConnect(sfiUserTrigger,IRQ_SOURCE);


  printf("sfiUserDownload: User Download Executed\n");

}

void
sfiUserPrestart()
{
  unsigned int adc_id;

  SFI_ENABLE_SEQUENCER;

  /* PRogram ADC */
  fpwc(ADC_SLOT,0,0x40000000);
  fpwc(ADC_SLOT,0,0x104);
  fpwc(ADC_SLOT,1,0x40);
  fprc(ADC_SLOT,0,&adc_id);

  printf("ADC id = 0x%x\n",adc_id);
  
  /* Load the Ram List */
  SFIRamLoad();

  /* Initialize SFI Interrupt variables */
  sfiIntClearCount();

}

void
sfiUserGo()
{
  /* Enable Interrupts */
  sfiIntEnable(IRQ_SOURCE,TS_MODE);
}

void
sfiUserEnd()
{
  int status, dma, count;

  /* Disable Interrupts */
  sfiIntDisable(IRQ_SOURCE,TS_MODE);

  /* Check on last Transfer */
  if(sfiIntCount > 0) {
    status = sfiSeqDone();
    if(status != 0) {
      logMsg("ERROR: in Last Transfer Event NUMBER %d, status = 0x%x\n",sfiIntCount,status,0,0,0,0);
      sfi_error_decode(0);
       *dma_dabufp++ = 0xfb000bad;
    }else{
      dma = sfiReadSeqFifo();
      count = dma&0xffffff;
      printf("Last DMA status = 0x%x\n",dma);
      dma_dabufp += count;
    }
    PUTEVENT(sfiOUT);
  }

  /*Reset AUX Port */
  sfiAuxWrite(0x8000);

  printf("sfiUserEnd: Ended after %d events\n",sfiIntCount);
  
}

void
sfiUserTrigger(int arg)
{
  int ii, status, dma, count;
  unsigned int datascan, auxval;


  /* check on Last Transfer */
  if (sfiIntCount > 1) {
    status = sfiSeqDone();
    if(status != 0) {
      logMsg("ERROR: in Last Transfer Event NUMBER %d, status = 0x%x\n",sfiIntCount,status,0,0,0,0);
      sfi_error_decode(0);
      *dma_dabufp++ = 0xfb000bad;
    }else{
      dma = sfiReadSeqFifo();
      count = dma&0xffffff;
      dma_dabufp += count;
    }
    PUTEVENT(sfiOUT);
  }

  /*Get Aux data (event type and Sync) check for data and start transfer*/
  auxval = sfiAuxRead();
  SFI_ENABLE_SEQUENCER;

  ii=0;
  datascan = 0;
  while ((ii<20) && (SCANMASK != (datascan&SCANMASK))) {
    datascan = 0;
    fb_frcm_1(9,0,&datascan,1,0,1,0,0,0);
    ii++;
  }

 
  if(ii<20) {
    GETEVENT(sfiIN,sfiIntCount);
    *dma_dabufp++ = 0xda0000da;   /* Place holder for the length */
    *dma_dabufp++ = auxval;       /* Send AUX port info for Event type and Sync info */
    sfiRamStart(RAM_ADDR,dma_dabufp);
  }

  if(dmaPEmpty(sfiIN)) {
    sfiNeedAck = 1;
    return;
  } else {
    sfiIntAck(IRQ_SOURCE,TS_MODE);
  }
}

void
sfiUserStatus()
{

}

void
sfiUserTest()
{
  int lock_key;
  DANODE *outEvent;
  
  while(1) {
    if(dmaPEmpty(sfiOUT))
      continue;
    
    /* get event - then free it */
    lock_key = intLock();
    outEvent = dmaPGetItem(sfiOUT);

    dmaPFreeItem(outEvent);
    intUnlock(lock_key);
  }


}

void
dcpuUserPoll(int bufsize, int to)
{
  int lock_key;
  int ii, jj, mt, len;
  int csr, cmd, state, flush=0;
  int tcount, ecount, eMax=0;
  DANODE *outEvent;

  if (bufsize <= 0) 
    bufsize = (dcpuSize>>3);  /* Bufsize default in words - half the total buffer space */

  if (to <= 0) 
    to = 2*(sysClkRateGet());  /* Time out default in ticks - every 2 seconds */

  /* Wait in IDLE loop until DCPU library has been initialized */
  while(dcpuReady != 1) {
    taskDelay(60);
  }


  /* Init Buffer memory control words */
  dcpuCntlPtr->nevents = 0;
  dcpuCntlPtr->nwords = 0;
  dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);
  dcpuCntlPtr->clear = 0;
  dcpuCntlPtr->full  = 0;
  dcpuCntlPtr->first = 0;
  dcpuCntlPtr->last  = 0;
  ii=0;

  state = DC_IDLE;  /*Initial State is Idle */

  /* Enter control loop - check for commands from Destination CPU */
  printf("dcpuUserPoll: Enabled...\n");
  tcount = vxTicks;

  while(1) {
    cmd = (dcpuCntlPtr->csr)&DC_CMD_MASK;

    switch (cmd) {
      case DC_DOWNLOAD:
	sfiUserDownload();
	eMax = dcpuCntlPtr->user[0]; /* Get Max Event count */
	printf("dcpuUserPoll: Max Event count = %d, Max buffer size %d words\n",
	       eMax,bufsize);
	state = DC_DOWNLOADED;
	dcpuCntlPtr->csr = state;
	break;
      case DC_PRESTART:
	sfiUserPrestart();
	dcpuCntlPtr->nevents = 0;
	dcpuCntlPtr->nwords = 0;
	dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);
	dcpuCntlPtr->clear = 0;
	dcpuCntlPtr->full  = 0;
	dcpuCntlPtr->first = 0;
	dcpuCntlPtr->last  = 0;
	state = DC_PRESTARTED;
	dcpuCntlPtr->csr = state;
        break;
      case DC_GO:
	sfiUserGo();
	flush = 0;
	tcount = vxTicks;
	state = DC_ACTIVE;
	dcpuCntlPtr->csr = state;
        break;
      case DC_END:
	sfiUserEnd();
	flush = 1;
	dcpuCntlPtr->nevents = sfiIntCount; /* Set final event count */
	dcpuCntlPtr->csr = DC_ACTIVE;       /*Until we flush all events */
	dcpuRemCntlPtr->csr = DC_ENDING;    /* Notify Dest Board Triggers are disabled */
        break;
      default:
	jj=0;
	mt=0;
	while((state == DC_ACTIVE)&&(jj++<MAX_NUM_EVENTS)) {
	  if(dmaPEmpty(sfiOUT))
	    mt=1;
    
	  if(((dcpuCntlPtr->full) == 0)&&((dcpuCntlPtr->clear)==0)) {
	    if(mt==0) {
	      /* get event off the list*/
	      lock_key = intLock();
	      outEvent = dmaPGetItem(sfiOUT);
	      if(outEvent != 0) {
		len = outEvent->length;

		/* Copy event to dcpuMemory */
		dcpuCntlPtr->nevents = sfiIntCount;
		if((dcpuCntlPtr->nwords)==0) /*Empty Buffer*/
		  dcpuCntlPtr->first = outEvent->nevent;
		dcpuCntlPtr->last = outEvent->nevent;
	    
      
		*(dcpuCntlPtr->local)++ = len;
		for(ii=0;ii<len;ii++) {
		  *(dcpuCntlPtr->local)++ = outEvent->data[ii];
		}
		dcpuCntlPtr->nwords += (len+1); 
		ecount = dcpuCntlPtr->last - dcpuCntlPtr->first + 1;

		/* buffer is full, too many events, or there is a timeout then flag to send */
		if((dcpuCntlPtr->nwords > bufsize)||(ecount >= eMax)||(vxTicks > (tcount + to))) {
		  sfi_pulse(1,0x100,1);   /* Set Nim out 1 */
		  dcpuCntlPtr->clear = 0;
		  dcpuCntlPtr->full  = 1;
		}

	      /* Free the event and renable interrupts */
		dmaPFreeItem(outEvent);
		intUnlock(lock_key);
	    
		if(sfiNeedAck > 0) {
		  sfiIntAck(IRQ_SOURCE,0);
		}
	      }else{
		intUnlock(lock_key);
		logMsg("Error: Could not get Event from List\n",0,0,0,0,0,0);
	      }
	    }

	  } else {

	    /*We are full - check to see if the transfer is complete */
	    if(dcpuCntlPtr->clear) {
      	      dcpuCntlPtr->nwords = 0;
	      dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);
	      dcpuCntlPtr->clear = 0;
	      dcpuCntlPtr->full  = 0;
	      tcount = vxTicks;
	      sfi_pulse(2,0x100,1);   /* Clear Nim out 1 */
	    }
	  }
	  
	  /* check to see if we are Ending - flush what we have
	     Two Conditions - Ending and Last event in buffer, */
	  if((flush)&&(sfiIntCount == dcpuCntlPtr->last)) {
	    dcpuCntlPtr->full  = 1;
	    while(dcpuCntlPtr->clear == 0) {
	      sysUsDelay(50);
	    }

	    dcpuCntlPtr->nwords = 0;
	    dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);
	    dcpuCntlPtr->clear = 0;
	    dcpuCntlPtr->full  = 0;

	    flush=0;
	    state = DC_ENDED;
	    dcpuCntlPtr->csr = state;
	  }
	} /* End active While loop */
	
    } /* End switch */
  }
}
