/******************************************************************************
*
*  sis3801Lib.c  -  Driver library for readout of a Struck 3801 Multi-scaler version
*                   5 or later firmware using a VxWorks 5.3.1  or later based 
*                   Single Board computer. This library can use the interrupt 
*                   capabilities of the scaler to drive the readout. The user has 
*                   the option of attaching his own interrupt service routine.
*
*  Author: David Abbott 
*          Jefferson Lab Data Acquisition Group
*          January 1999
*
*/

#include "vxWorks.h"
#include "stdio.h"
#include "string.h"
#include "intLib.h"
#include "iv.h"
#include "logLib.h"
#include "taskLib.h"
#include "vxLib.h"

/* Include scaler definitions */
#include "sis3801Lib.h"

IMPORT  STATUS sysBusToLocalAdrs(int, char *, char **);
IMPORT  STATUS intDisconnect(int);
IMPORT  STATUS sysIntEnable(int);


LOCAL BOOL        s3801IntRunning  = FALSE;	              /* running flag */
LOCAL VOIDFUNCPTR s3801IntRoutine  = NULL;	              /* user interrupt service routine */
LOCAL int         s3801IntArg      = NULL;	              /* arg to user routine */
LOCAL UINT32      s3801IntLevel    = S3801_VME_INT_LEVEL;   /* default VME interrupt level */
LOCAL UINT32      s3801IntVec      = S3801_INT_VEC;         /* default interrupt Vector */

/* Define global variables */
int Ns3801 = 0;
volatile struct s3801_struct *s3801p[20];    /* pointer to Scaler memory map */
volatile UINT32 scalData[32][20];            /* storage for Scaler data Sums */


/*******************************************************************************
*
* scalInt - default interrupt handler
*
* This rountine handles the timer interrupt.  A user routine is
* called, if one was connected by s3801Connect().
*
* RETURNS: N/A
*
* SEE ALSO: s3801IntConnect()
*/

LOCAL void 
scalInt (int id)
{
      int mask;

      mask = (s3801p[id]->csr)&S3801_ENABLE_IRQ_MASK; /* Get Scaler IRQ mask */
      s3801p[id]->csr = S3801_DISABLE_IRQ_MASK;       /* disable all interrupt sources */
      
      if (s3801IntRoutine != NULL)	/* call user routine */
        (*s3801IntRoutine) (s3801IntArg);

      s3801p[id]->csr = mask;   /* Re-enable interrupts */

}

/*******************************************************************************
*
* s3801IntConnect - connect a user routine to the scaler interrupt
*
* This routine specifies the user interrupt routine to be called at each
* interrupt
*
* RETURNS: OK, or ERROR if s3801() interrupt handler is not used.
*/

STATUS 
s3801IntConnect (VOIDFUNCPTR routine, int arg)
    {
      s3801IntRoutine = routine;
      s3801IntArg = arg;
      
      return (OK);
    }

/*******************************************************************************
*
* s3801Init - initialize scaler for interrupts
*
*
* RETURNS: OK, or ERROR if the scaler address is invalid.
*/

STATUS 
s3801Init (UINT32 addr, UINT32 addr_inc, int nscal)
{
  int ii, res, rdata, errFlag;
  unsigned int boardID, laddr;


  /* Check for valid address */
  if(addr==0) {
    printf("scalInit: ERROR: Must specify local VME (A24) or CPU Based (A32) address for scaler\n");
    return(ERROR);
  }else if (addr > 0x00ffffff) {
    laddr = addr;
  }else{
    /* get the scaler's address */
    res = sysBusToLocalAdrs(0x39,(char *)addr,(char **)&laddr);
    if (res != 0) {
      printf("s3801Init: ERROR in sysBusToLocalAdrs(0x39,0x%x,&laddr) \n",addr);
      return(ERROR);
    }
  }

  if((addr_inc==0)||(nscal==0))
    nscal = 1; /* assume only one SCALER to initialize */


  Ns3801 = 0;
  for (ii=0;ii<nscal;ii++) {
    s3801p[ii] = (struct s3801_struct *)(laddr + ii*addr_inc);
    /* Check if Board exists at that address */
    res = vxMemProbe((char *) &(s3801p[ii]->irq),VX_READ,4,(char *)&rdata);
    if(res < 0) {
      printf("s3801Init: ERROR: No addressable board at addr=0x%x\n",(UINT32) s3801p[ii]);
      s3801p[ii] = NULL;
      errFlag = 1;
      break;
    } else {
      /* Check if this is a Model 3300 */
      boardID =  rdata&S3801_BOARD_ID_MASK;
      if(boardID != S3801_BOARD_ID) {
        printf(" ERROR: Board ID does not match: %d \n",boardID);
        return(ERROR);
      }
    }
    Ns3801++;
    printf("Initialized SCALER ID %d at address 0x%08x \n",ii,(UINT32) s3801p[ii]);
  }


  /* Zero out scaler Data array */
  bzero((char *)scalData,sizeof(scalData));
      
  return(OK);
}


/* Key Address Functions */
void
s3801Enable(int id)
{
  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801Enable: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3801p[id]->enable = 1;
  s3801p[id]->next = 1;
}

void
s3801Disable(int id)
{
  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801Disable: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3801p[id]->disable = 1;
}

void
s3801Clear(int id)
{
  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801Clear: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3801p[id]->clear = 1;
}

void
s3801Next(int id)
{
  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801Next: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3801p[id]->next = 1;
}

void
s3801Reset(int id)
{
  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801Reset: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3801p[id]->reset = 1;
}

void
s3801EnablePulser(int id)
{
  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801EnablePulser: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3801p[id]->enablerefpulser = 1;
}

void
s3801DisablePulser(int id)
{
  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801DisablePulser: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return;
  }
  s3801p[id]->disablerefpulser = 1;
}


/*******************************************************************************
*
* s3801IntEnable - enable scaler interrupts
*
* This routine takes an interrupt source mask and enables interrupts
* fo the initialized scaler.
*     mask:    bit 0  IRQ on start of CIP     
*    (1-15)    bit 1  IRQ on FIFO full
*              bit 2  IRO on FIFO half full
*              bit 3  IRO on FIFO almost full
*
* RETURNS: OK, or ERROR if not intialized.
*/

STATUS 
s3801IntEnable (int id, UINT32 mask)
{

  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801IntEnable: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }

      
  if((mask==0) || (mask > 0xf)) {
    printf("VME Interrupt mask=0x%x is out of range\n",mask);
    return(ERROR);
  }else{
    if(s3801p[id]) {
      /* Program scaler to generate interrupts */
      s3801p[id]->csr = S3801_DISABLE_IRQ_MASK;   /* disable all interrupt sources */
      s3801p[id]->irq = (0x800) | (s3801IntLevel<<8) | s3801IntVec;
      s3801p[id]->csr = (mask<<20);
    }else{
      printf("s3801 not Initialized. Call s3801Init()!\n");
      return(ERROR);
    }
  }
  return (OK);
}


/*******************************************************************************
*
* s3801IntDisable - disable the scaler interrupts
*
* RETURNS: OK, or ERROR if not initialized
*/

STATUS 
s3801IntDisable (int id)
{

  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801IntDisable: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return(ERROR);
  }


  if (s3801p[id]) {
    s3801p[id]->csr = S3801_DISABLE_IRQ_MASK;
    s3801IntRunning = FALSE;
  } else {
    printf("s3801 not Initialized. Call s3801Init()!\n");
    return(ERROR);
  }
      
  return (OK);
}


/*******************************************************************************
*
* s3801IntUser - default user interrupt routine
*
*  This routine can be modified to add the code the User wishes to execute
*  upon recieving a scaler interrupt
*
*  RETURNS: N/A
*
*/
void s3801IntUser (int arg)
{
  int ii;

  /* Check if FIFO is empty and read N scaler values out of the fifo. The argument
     to the ISR defines the number of scaler reads (or active scaler channels)
   */

  if((s3801p[0]->csr)&S3801_STATUS_FIFO_EMPTY) {
    /* FIFO is empty */
    logMsg("s3801IntUser: FIFO Empty\n",0,0,0,0,0,0);
  } else {
    /* logMsg("s3801IntUser: Read %d scalers\n",arg,0,0,0,0,0); */
    for(ii=0;ii<arg;ii++) {
      scalData[ii][0] = scalData[ii][0] + s3801p[0]->fifo[0];
    }
  }
}

/*******************************************************************************
*
* scalTest - Program Scaler to test interrupts
*            mode input specifies optional test modes
*            Refer to User manual on bit enable definitions
*            for the various scaler operation modes.
*
*  RETURNS: N/A
*
*/
void 
s3801Test(int id, UINT32 mode)
{
  mode &= 0xff;  /* Mask the mode input */

  if(s3801p[id]) {
    printf("Reset, Clear FIFO, and Enable scaler.\n");
    s3801p[id]->reset = 1;
    s3801p[id]->clear = 1;
    s3801p[id]->enable = 1;
    if(mode) {
      s3801p[id]->csr = mode;
      printf("Program CSR for mode: 0x%x\n",mode);
    }
    s3801p[id]->next = 1;
  }else{
    printf("Call s3801Init() first!\n");
  }
}



/*******************************************************************************
*
* scalPulse - Pulse all enabled scalers (count) times
*
*  RETURNS: N/A
*
*/
void 
s3801Pulse(int id, int count)
{
  int ii;

  if(count == 0) count = 1;

  if(s3801p[id]) {
    if(s3801p[id]->csr&S3801_STATUS_TEST_MODE) {
      for(ii=0;ii<count;ii++)
	s3801p[id]->test = 1;
      /* printf("Completed %d test pulses\n",count); */
    } else {
      printf("Test mode not enabled. Call scalTest(%d, 0x20)\n",id);
    }
  }else{
    printf("Call s3801Init() first!\n");
  }
}


/*******************************************************************************
*
* scalPrint - Print scaler Values to Standard out  (For STR7200/SIS3800 only)
*
*  RETURNS: N/A
*
*/
void 
s3801Print(int id, int scount)
{
  int ii, status;

  if((scount<=0)||(scount>32)) scount = 32;

  status = (s3801p[id]->csr)&S3801_FIFO_STATUS_MASK;
  if(status == S3801_STATUS_FIFO_EMPTY) {
    printf("s3801Print: ERROR: No Data in FIFO\n");
    return;
  }

  printf("   Scaler Data :\n");

  for(ii=1;ii<=scount;ii+=4) {
    printf(" SCAL%02d: %10d  SCAL%02d: %10d  SCAL%02d: %10d  SCAL%02d: %10d \n",
	   ii,  s3801p[id]->fifo[ii-1],
	   ii+1,s3801p[id]->fifo[ii],
	   ii+2,s3801p[id]->fifo[ii+1],
	   ii+3,s3801p[id]->fifo[ii+2]);
  }

}

/*******************************************************************************
*
* s3801Read - Read Scaler data to a Buffer
*
*  RETURNS: N/A
*
*/
int
s3801Read(int id, UINT32 *data, int scount)
{
  int ii, status;

  if((scount<=0)||(scount>32)) scount = 32;

  status = (s3801p[id]->csr)&S3801_FIFO_STATUS_MASK;
  if(status == S3801_STATUS_FIFO_EMPTY) {
    logMsg("s3801Read: ERROR: No Data in FIFO\n",0,0,0,0,0,0);
    return(ERROR);
  }

  for(ii=0;ii<scount;ii++) {
     data[ii] = s3801p[id]->fifo[ii];
  }

  return(scount);
}


/*******************************************************************************
*
* scalStatus - Print status of scaler in readable form
*
*  RETURNS: N/A
*
*/
void 
s3801Status(int id, int sflag)
{
  int CSR, IRQ;
  int input_mode;
  char irq_mask;
  char irq_mask_pending;
  char next_enabled[9];
  char irq_enabled[9];
  char test_enabled[9];
  char fifo_status[13];

  if((id<0) || (s3801p[id] == NULL)) {
    logMsg("s3801Status: ERROR : SCALER id %d not initialized \n",id,0,0,0,0,0);
    return;
  }

  if(s3801p[id]) {
    CSR = s3801p[id]->csr;
    IRQ = s3801p[id]->irq;
    if (IRQ&S3801_INT_ENABLED_MASK)
      strncpy((char *)&irq_enabled[0],"Enabled \0",9);
    else
      strncpy((char *)&irq_enabled[0],"Disabled\0",9);

    if (CSR&S3801_STATUS_ENABLE_NEXT)
      strncpy((char *)&next_enabled[0],"Enabled \0",9);
    else
      strncpy((char *)&next_enabled[0],"Disabled\0",9);

    if (CSR&S3801_STATUS_TEST_MODE)
      strncpy((char *)&test_enabled[0],"Enabled \0",9);
    else
      strncpy((char *)&test_enabled[0],"Disabled\0",9);

    irq_mask = (CSR&S3801_ENABLE_IRQ_MASK)>>20;
    irq_mask_pending = (CSR&S3801_DISABLE_IRQ_MASK)>>28;
    input_mode = (CSR&S3801_INPUT_MODE_MASK)>>2;


    switch (CSR&S3801_FIFO_STATUS_MASK) {
    case S3801_STATUS_FIFO_FULL:
      strncpy((char *)&fifo_status[0],"Full        \0",13);
      break;
    case S3801_STATUS_FIFO_ALMOST_FULL:
      strncpy((char *)&fifo_status[0],"Almost Full \0",13);
      break;
    case S3801_STATUS_FIFO_HALF_FULL:
      strncpy((char *)&fifo_status[0],"Half Full   \0",13);
      break;
    case S3801_STATUS_FIFO_ALMOST_EMPTY:
      strncpy((char *)&fifo_status[0],"Almost Empty\0",13);
      break;
    case S3801_STATUS_FIFO_EMPTY:
    case (S3801_STATUS_FIFO_EMPTY|S3801_STATUS_FIFO_ALMOST_EMPTY):
      strncpy((char *)&fifo_status[0],"Empty       \0",13);
      break;
    default:
      strncpy((char *)&fifo_status[0],"Not Empty   \0",13);
    }
    
    /* Print out results */
    printf("  Scaler (ID=%d) Status (Addr=0x%x Module id=0x%x)\n",id,(int) s3801p[id],((IRQ&S3801_BOARD_ID_MASK)>>16));
    printf("  CSR Register = 0x%08x\n",CSR);
    printf("      FIFO Status     = %s\n",fifo_status);
    printf("      Next Logic      = %s\n",next_enabled);
    printf("      Test Mode       = %s\n",test_enabled);
    printf("      Input Mode      = %d\n",input_mode);
    printf("  IRQ Register = 0x%08x\n",IRQ);
    printf("      Interrupts          = %s\n",irq_enabled);
    printf("      VME Interrupt Level = %d\n",(IRQ&S3801_INT_LEVEL_MASK)>>8);
    printf("      Interrupt Vector    = 0x%x\n",(IRQ&S3801_INT_VEC_MASK));
    printf("      IRQ Source Mask     = 0x%x  IRQs pending = 0x%x\n",irq_mask,irq_mask_pending);

  }else{
    printf("Call s3801Init() first!\n");
  }

}

