/**************************************************************************
 * Copyright (c)   2001    Southeastern Universities Research Association,
 *                         Thomas Jefferson National Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
 **************************************************************************
 *
 * Description:
 *      Main Display Part of cmlog browser
 *
 * Author:  
 *      Jie Chen
 *      Jefferson Lab HPC Group
 *
 * Revision History:
 *   $Log: Disp.java,v $
 *   Revision 1.1  2001/10/18 18:45:41  chen
 *   First version of Jcmlog
 *
 *
 *
 */
package cmlog.gui;

import java.io.*;
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.table.*;

import cmlog.*;

public class Disp
{
    /**
     * Type of this display.
     */
    public static final int UPDATING = 1024;
    public static final int QUERYING = 1025;
    public static final int IDLE     = 1026;

    /**
     * JTable object.
     */
    private JTable table_ = null;

    /**
     * ScrollPane holding this table.
     */
    private JScrollPane scrollPane_ = null;

    /**
     * Type of this display
     */
    private int    type_ = IDLE;

    /**
     * Parent window (frame) of this display
     */
    private Jcmlog  parent_ = null;

    /**
     * Internal data model.
     */
    private DataHandler dataHandler_ = null;

    /**
     * Query progress dialog.
     */
    private QueryProgDialog qpdialog_ = null;
    private boolean         firstTime_ = true;

    /**
     * Configuration object.
     */
    private Config config_ = null;


    /**
     * Set up column widths.
     */
    private void initColumnSizes(int[] colwidths)
    {
	TableColumn column = null;

        for (int i = 0; i < colwidths.length; i++) {
            column = table_.getColumnModel().getColumn(i);
            column.setPreferredWidth(colwidths[i]);
        }
    }


    /**
     * Constructor
     *
     */
    public Disp (Config config, int initnumrow, int type)
    {
	config_ = config;

	dataHandler_ = new DataHandler (config.getNumberOfColumns(),
					initnumrow,
					config.getColumnTitles(), 
					config.getColumnTags(), 
					config.getUpdateBufferSize(), type,
					this);
	type_ = type;

	table_ = new JTable (dataHandler_);
	table_.setFont(new Font("Helvetica", Font.PLAIN, 14));
	table_.setRowSelectionAllowed (false);
	table_.setColumnSelectionAllowed (false);

	initColumnSizes(config.getColumnWidths ());

	// add data handler model listener
	dataHandler_.addTableModelListener (table_);

	// create Query Progress Dialog
	qpdialog_ = new QueryProgDialog (this);
    }

    /**
     * Move table to bottom of scroll pane
     */
    public void scrollTableToBottom (int row)
    {
	// get table row height
	int rowh = table_.getRowHeight();
	// get table width and height
	int height = table_.getHeight();

	// get current y coordinate of this row
	int y = row*rowh;

	// get current view are
	Rectangle trect = scrollPane_.getViewport().getViewRect();

	if (y >= trect.y + trect.height) {
	    trect.y = y - trect.height + rowh;
	    table_.scrollRectToVisible(trect);
	}
    }

    /**
     * Return table object to upper window.
     */
    public JTable getTable ()
    {
	return table_;
    }

    /**
     * Set scroll pane
     */
    public void setScrollPane (JScrollPane pane)
    {
	scrollPane_ = pane;
    }

    /**
     * Return parent window of this display.
     */
    public Jcmlog getParentWindow ()
    {
	return parent_;
    }

    /**
     * Set parent window of this display.
     */
    public void setParentWindow (Jcmlog parent)
    {
	parent_ = parent;
    }

    /**
     * Return configuration object.
     */
    public Config getConfigObject ()
    {
	return config_;
    }


    /**
     * Display Error Message
     */
    public void popupErrorDialog (String message)
    {
	JOptionPane.showMessageDialog(parent_, message, "CMLOG Error",
				      JOptionPane.ERROR_MESSAGE); 
    }

    /**
     * Display Warning Message
     */
    public void popupWarningDialog (String message)
    {
	JOptionPane.showMessageDialog(parent_, message, "CMLOG Warning",
				      JOptionPane.WARNING_MESSAGE); 
    }


    /**
     * Display Info Message
     */
    public void popupInfoDialog (String message)
    {
	JOptionPane.showMessageDialog(parent_, message, "CMLOG Info",
				      JOptionPane.INFORMATION_MESSAGE); 
    }

    /**
     * Popup query progress dialog
     */
    public void popupQueryProgDialog (String title, double newfrom, 
				      QueryArgument qarg)
    {
	qpdialog_.updateQueryInfo (title, newfrom, qarg);

	if (firstTime_ == true) {
	    qpdialog_.pack ();
	    qpdialog_.setLocationRelativeTo (parent_);
	    firstTime_ = false;
	}
	// I have to use setVisiable instead of show to make sure
	// dialog box is poped up on top of table widget.
	qpdialog_.setVisible (true);
    }

    /**
     * Get type of this display.
     */
    public int getType ()
    {
	return type_;
    }

    /**
     * Set type of this display.
     */
    public void setType (int type)
    {
	type_ = type;
	parent_.setDisplayType (type_);
    }
       

    /**
     * Start monitoring incoming messages.
     */
    public void startUpdating (String selection)
    {
	if (dataHandler_.startUpdating (selection) == false) {
	    popupErrorDialog ("Cannot monitor cmlog messages.");
	    type_ = IDLE;
	    return;
	}
	else 
	    type_ = UPDATING;
	
	parent_.setDisplayType (type_);

	// set start/end time
	parent_.setStartTime (new Date());
	parent_.setEndTime (null);
    }

    /**
     * Stop monitoring incoming messages.
     */
    public void stopUpdating ()
    {
	if (dataHandler_.stopUpdating () == false) {
	    popupErrorDialog ("Cannot stop monitoring cmlog messages.");
	    return;
	}
	else 
	    type_ = IDLE;
	
	parent_.setDisplayType (type_);
	// set start/end time
	parent_.setEndTime (new Date());
    }

    /**
     * Doing query
     */
    public void doQuery (Date from, Date to,
			 int  numitems, boolean getall,
			 String qstr, boolean removeold)
    {
	// if we are in querying mode already, do nothing
	if (type_ == Disp.QUERYING)
	    return;

	if (dataHandler_.doQuery (from, to, numitems, getall, qstr) == false){
	    popupErrorDialog ("Cannot initiate a query process.");
	    return;
	}
	
	if (removeold == true)
	    // clean up old messages
	    dataHandler_.removeAll ();
	type_ = Disp.QUERYING;
	
	// display right menus
	parent_.setDisplayType (type_);

	// set start and end time
	parent_.setStartTime (from);
	parent_.setEndTime (null);
    }

    /**
     * Stop a query
     */
    public void stopQuery ()
    {
	// if we are in querying mode already, do nothing
	if (type_ != Disp.QUERYING)
	    return;

	dataHandler_.stopQuery ();
    }

    /**
     * Remove all items inside the table.
     */
    public void removeAll ()
    {
	parent_.setStartTime (null);
	parent_.setEndTime (null);
	dataHandler_.removeAll ();
    }

    /**
     * Perform exit action.
     */
    public void exit ()
    {
	try {
	    dataHandler_.close ();
	}catch (Exception e) {
	    ;
	}
	System.exit (0);
    }
    

    /**
     * Perform connect action.
     */
    public void connect (String host, int port)
    {
	try {
	    dataHandler_.connect (host, port);
	}catch (Exception e) {
	    popupErrorDialog ("Cannot connect to a cmlog server at " + String.valueOf (port) + "@" + host + ".\n" + "Error Message is: " + e.getMessage());
	}
	// set parent menu items to reflect connected state
	parent_.setConnectedState ();
    }

    /**
     * Performe disconnect operation.
     */
    public void disconnect ()
    {
	try {
	    dataHandler_.close ();
	}catch (Exception e) {
	    ;
	}
	// set parent menu items to reflect disconnected state
	parent_.setDisconnectedState ();
    }

    /**
     * Save current settings.
     */
    public void saveConfig ()
    {
	// get current column width
	TableColumn column = null;

        for (int i = 0; i < config_.getNumberOfColumns (); i++) {
            column = table_.getColumnModel().getColumn(i);
            config_.setColumnWidth (i, column.getWidth ());
        }

	try {
	    config_.saveConfig ();
	}catch (Exception e) {
	    popupErrorDialog ("Could not save settings to the config file.");
	}
	popupInfoDialog ("Current settings is saved to the config file.");
    }	
}




