/**************************************************************************
 * Copyright (c)   2001    Southeastern Universities Research Association,
 *                         Thomas Jefferson National Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
 **************************************************************************
 *
 * Description:
 *      Read configuration information from a file
 *
 * Author:  
 *      Jie Chen
 *      Jefferson Lab HPC Group
 *
 * Revision History:
 *   $Log: Config.java,v $
 *   Revision 1.3  2001/10/19 18:16:11  chen
 *   Change how image is loaded
 *
 *   Revision 1.2  2001/10/19 00:14:24  chen
 *   fix a bug related to empty server/port
 *
 *   Revision 1.1  2001/10/18 18:45:40  chen
 *   First version of Jcmlog
 *
 *
 *
 */
package cmlog.gui;

import java.io.*;
import java.util.*;

import cmlog.*;

public class Config
{
    /**
     * Name of tokens inside the configuration file.
     */
    private static final String COL_TOKEN = "Col";
    private static final String NAME_TOKEN = "name";
    private static final String SERVER_TOKEN = "server";
    private static final String PORT_TOKEN = "port";
    private static final String PROTOCOL_TOKEN = "protocol";
    private static final String UMSG_TOKEN = "updateMessage";
    private static final String QMSG_TOKEN = "queryMessage";
    private static final String UBUFSIZE_TOKEN = "ubufsize";
    private static final String WINW_TOKEN = "windowWidth";
    private static final String WINH_TOKEN = "windowHeight";
    private static final String CODECONV_TOKEN = "codeConversion";


    /**
     * Default titles.
     */
    private static String[] deftitles_ = {"Host", "Time", "Message"};

    /**
     * Default tags
     */
    private static String[] deftags_ = {"host", "cmlogTime", "text"};

    /**
     * Default widths of columns.
     */
    private static int[]    defwidths_ ={50, 50, 300};
    
    /**
     * Update buffer size.
     */
    private static int      updateBufsize_ = 2000;
   

    /**
     * global instance of configuation object.
     */
    private static Config configObj_ = null;

    /**
     * Name of this configuration
     */
    private String name_ = "CMLOG Java Browser";

    /**
     * Number of columns.
     */
    private int numcols_ = 0;

    /**
     * Widths of columns
     */
    private int[] colwidths_ = null;

    /**
     * Titles of these columns
     */
    private String[] coltitles_ = null;

    /**
     * Tags corresponding to these titles.
     */
    private String[] coltags_ = null;

    /**
     * Updating buffer size.
     */
    private int ubufsize_ = 2000;

    /**
     * Updating selection message.
     */
    private String umsg_ = null;

    /**
     * Possible Server Hosts and Port
     */
    private String[] servers_ = null;
    private int[]    ports_ = null;

    /**
     * Protocol version
     */
    private int      pvers_ = 2;

    /**
     * Constructor.
     *    This is a private constructor, caller has to use static method
     *    to access the global configuration object.
     */
    private Config ()
    {
	int i;

	numcols_ = 3;

	colwidths_ = new int[numcols_];
	coltitles_ = new String[numcols_];
	coltags_ = new String[numcols_];

	for (i = 0; i < numcols_; i++) {
	    colwidths_[i] = defwidths_[i];
	    coltitles_[i] = deftitles_[i];
	    coltags_[i] = deftags_[i];
	}
    }
    
    /**
     * Get the global configuration information object.
     */
    public static Config getConfig ()
    {
	if (configObj_ == null)
	    configObj_ = new Config();

	return configObj_;
    }

    /**
     * Save to default config file.
     */
    public void saveConfig ()
	throws IOException
    {
	String cfname = System.getProperty ("user.home") + File.separator + ".jcmlogrc"; 

	output (cfname);
    }

    /**
     * Output configuration information to a file.
     */
    public void output (String file)
	throws IOException
    {
	int    i;
	File   configf = new File (file);

	if (configf.exists () == true)
	    configf.delete ();

	if (configf.createNewFile () == false)
	    throw new IOException ("Cannot create file " + file);

	if (configf.canWrite () == false)
	    throw new IOException ("Cannot open configuration file " + file + " to write");

	PrintWriter w = new PrintWriter (new FileOutputStream (configf));
	
	w.println ("# Configuration file for cmlog");
	if (name_ != null) {
	    w.println ("# Name of this configuration");
	    w.println ("name " + name_);
	}

	w.println ("# Type          Title          Tag          Width");
	for (i = 0; i < numcols_; i++) {
	    w.println ("Col    " + coltitles_[i] + "     " + coltags_[i] + "     " + String.valueOf (colwidths_[i]));
	}
	
	w.println ("# Buffer size for updating the message");
	w.println ("ubufsize     " + String.valueOf (ubufsize_));

	w.println ("# Servers");
	if (servers_ != null) {
	    w.print ("server ");
	    for (i = 0; i < servers_.length; i++) {
		w.print (servers_[i]);
		w.print ("  ");
	    }
	    w.println ();
	}

	if (ports_ != null) {
	    w.print ("port ");
	    for (i = 0; i < ports_.length; i++) {
		w.print (ports_[i]);
		w.print ("   ");
	    }
	    w.println ();
	}

	w.print ("protocol ");
	w.println (pvers_);

	w.flush ();
	w.close ();
    }
	
	

    /**
     * Parse configuration file.
     * 
     * @param file configuration file name
     * @return true if configuration information are correct; otherwise false
     */
    public boolean parseFile (String file)
	throws FileNotFoundException, IOException
    {
	String line = null;
	String token0;
	String[] tokens = new String[3];
	int    num, i;
	StringTokenizer st = null;
	File   configf = new File (file);

	if (configf.exists () == false) 
	    throw new FileNotFoundException ("Configuration file " + file + " does not exist");
	
	if (configf.canRead () == false)
	    throw new IOException ("Cannot open configuration file " + file + " to read");

	BufferedReader bufr = new BufferedReader (new FileReader (configf));

	num = 0;
	while ((line = bufr.readLine()) != null) {
	    line = line.trim ();
	    if (line.length() != 0 && line.charAt (0) != '#') {
		// not a comment line
		st = new StringTokenizer (line);
		if ((token0 = st.nextToken ()) != null &&
		    token0.equals (Config.COL_TOKEN))
		    num++;
	    }
	}

	bufr.close ();
	if (num == 0)
	    return false;
	
	// set values for number of columns
	numcols_ = num;
	colwidths_ = new int[numcols_];
	coltitles_ = new String[numcols_];
	coltags_ = new String[numcols_];
	num = 0;
	
	bufr = new BufferedReader (new FileReader (configf));
	while ((line = bufr.readLine()) != null) {
	    line = line.trim();
	    if (line.length() != 0 && line.charAt (0) != '#') {
		// not a comment line
		st = new StringTokenizer (line);

		if ((token0 = st.nextToken ()) != null) {
		    if (token0.equals (Config.COL_TOKEN) == true) {
			i = 0;
			while (st.hasMoreTokens () && i < 3) {
			    tokens[i++] = st.nextToken();
			}
			// Configuration file line format is
			// Col   Title  Tag   width
			if (i == 3) {
			    coltitles_[num] = tokens[0];
			    coltags_[num] = tokens[1];
			    colwidths_[num] = Integer.valueOf(tokens[2]).intValue();
			}
			num++;
		    }
		    else if (token0.equals (Config.NAME_TOKEN) == true) {
			int    idx = -1;
			String tmp = st.nextToken ();

			if (tmp != null) {
			    idx = line.indexOf (tmp);
			    if (idx != -1)
				name_ = line.substring (idx);
			}

			if (idx == -1)
			    name_ = "CMLOG Java Browser";
		    }
		    else if (token0.equals (Config.SERVER_TOKEN) == true) {
			Vector data = new Vector ();
			Object[] tmp = null;
			while (st.hasMoreTokens ()) 
			    data.add (st.nextToken());

			tmp = data.toArray ();
			servers_ = new String[tmp.length];
			for (i = 0; i < tmp.length; i++)
			    servers_[i] = (String)tmp[i];
		    }
		    else if (token0.equals (Config.PORT_TOKEN) == true) {
			Vector data = new Vector ();
			Object[] tmp = null;
			String   nou = null;

			while (st.hasMoreTokens ()) 
			    data.add (st.nextToken());

			tmp = data.toArray ();
			ports_ = new int[tmp.length];
			for (i = 0; i < tmp.length; i++) {
			    nou = (String)tmp[i];
			    ports_[i] = Integer.valueOf(nou).intValue();
			}
		    }
		    else if (token0.equals (Config.UBUFSIZE_TOKEN) == true) {
			String tmp =  st.nextToken ();
			if (tmp == null)
			    return false;
			ubufsize_ = Integer.valueOf(tmp).intValue();
		    }
		    else if (token0.equals (Config.PROTOCOL_TOKEN) == true) {
			String tmp =  st.nextToken ();
			if (tmp == null)
			    return false;
			pvers_ = Integer.valueOf(tmp).intValue();
		    }
		    else if (token0.equals (Config.UMSG_TOKEN) == true) 
			umsg_ = parseUpdateMessageLine (line);
		}
	    }
	}
	bufr.close ();

	return true;
    }

    /**
     * Parse updating message.
     */
    private String parseUpdateMessageLine (String line)
    {
	int firstidx, lastidx;
	String tmp;

	firstidx = line.indexOf ('"');
	lastidx  = line.lastIndexOf ('"');

	if (firstidx == -1 || lastidx == -1)
	    return "none";

	tmp = line.substring (firstidx + 1, lastidx);
	
	// check whether tmp is an empty string
	tmp = tmp.trim ();

	if (tmp.length() == 0)
	    return "none";
	
	return tmp;
    }

	    
    /**
     * Get number of columns.
     */
    public int getNumberOfColumns ()
    {
	return numcols_;
    }

    /**
     * Get all column widths.
     */
    public int[] getColumnWidths ()
    {
	return colwidths_;
    }

    /**
     * Set column width.
     */
    public void setColumnWidth (int idx, int width)
    {
	colwidths_[idx] = width;
    }

    /**
     * Get total column widths.
     */
    public int getTotalColumnWidth ()
    {
	int width = 0;

	for (int i = 0; i < colwidths_.length; i++)
	    width += colwidths_[i];

	return width;
    }

    /**
     * Get all column titles.
     */
    public String[] getColumnTitles ()
    {
	return coltitles_;
    }

    /**
     * Get all column tags.
     */
    public String[] getColumnTags ()
    {
	return coltags_;
    }

    /**
     * Get updating buffer size.
     */
    public int getUpdateBufferSize ()
    {
	return ubufsize_;
    }

    /**
     * Set updating buffer size.
     */
    public void setUpdateBufferSize (int size)
    {
	ubufsize_ = size;
    }

    /**
     * Get protocol version.
     */
    public int getProtocolVersion ()
    {
	return pvers_;
    }

    /**
     * Set protocol version.
     */
    public void setProtocolVersion (int version)
    {
	pvers_ = version;
    }


    /**
     * Convert a title into a tag.
     */
    public String getTagForTitle (String title)
    {
	int i;

	for (i = 0; i < numcols_; i++) {
	    if (title.equals (coltags_[i]) == true)
		return title;
	    else if (title.equals (coltitles_[i]) == true)
		return coltags_[i];
	}
	return null;
    }

    /**
     * Convert a tag into a title.
     */
    public String getTitleForTag (String tag)
    {
	int i;

	for (i = 0; i < numcols_; i++) {
	    if (tag.equals (coltags_[i]) == true)
		return coltitles_[i];
	}
	return null;
    }

    /**
     * Return configuration name.
     */
    public String getName ()
    {
	return name_;
    }


    /**
     * Return possible server names.
     */
    public String[] getServerNames ()
    {
	return servers_;
    }

    /**
     * Return possible port numbers.
     */
    public int[] getPortNumbers ()
    {
	return ports_;
    }

    /**
     * Add a server name to server poll.
     */
    public void addServer (String server)
    {
	int i;

	if (servers_ == null) {
	    servers_ = new String[1];
	    servers_[0] = server;
	    return;
	}

	for (i = 0; i < servers_.length; i++) {
	    if (servers_[i].equals (server) == true)
		return;
	}

	// create new servers
	String[] servers = new String[servers_.length + 1];
	for (i = 0; i < servers_.length; i++) {
	    servers[i] = servers_[i];
	}
	servers[i] = server;

	// change pointer
	servers_ = servers;
    }

    /**
     * Add a new port number to the port poll.
     */
    public void addPort (int port)
    {
	int i;

	if (ports_ == null) {
	    ports_ = new int[1];
	    ports_[0] = port;
	    return;
	}

	for (i = 0; i < ports_.length; i++) {
	    if (ports_[i] == port)
		return;
	}

	// create new ports
	int[] ports = new int[ports_.length + 1];
	for (i = 0; i < ports_.length; i++) {
	    ports[i] = ports_[i];
	}
	ports[i] = port;

	// change pointer
	ports_ = ports;
    }
	

    /**
     * Get update selection String
     */
    public String getUpdateSelMessage ()
    {
	return umsg_;
    }

}
     




