//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      Tag Table for cdev data
//
// Author:  
//      Jie Chen, Chip Watson
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: cdevTagTable.java,v $
//   Revision 1.2  2000/03/17 19:18:37  chen
//   add java client
//
//   Revision 1.1  1999/12/14 15:31:38  chen
//   initial java implementation
//
//
//
package cmlog;

import java.lang.*;
import java.util.*;

class cdevTagTable 
{
    private boolean initialized_ = false;
    private static cdevTagTable gTagTable_ = null;
    
    private Hashtable ikey_;
    private Hashtable skey_;

    private Vector eventListeners_ = null;

    /**
     *  Private Constructor for cdevTagTable
     */
    private cdevTagTable () 
    {
	int HASHTABLESIZE = 137;
	float HASHLOAD = (float) 0.25;
	try {
	    ikey_ = new Hashtable(HASHTABLESIZE,HASHLOAD);
	    skey_ = new Hashtable(HASHTABLESIZE,HASHLOAD);
	} catch (IllegalArgumentException e) {
	}
	initialize ();
    }

    private void initialize () {
	if (initialized_) return;
	int MAXSTRINGS = 25;
	String strings[] = new String[MAXSTRINGS];
	int i = 0;
  
	eventListeners_ = new Vector();

	strings[i++]  = "value";
	strings[i++]  = "status";  
	strings[i++]  = "severity";
	strings[i++]  = "time";
	strings[i++]  = "units";
	strings[i++]  = "displayHigh";
	strings[i++]  = "displayLow";
	strings[i++]  = "alarmHigh";
	strings[i++]  = "alarmLow";
	strings[i++]  = "warningHigh";
	strings[i++]  = "warningLow";
	strings[i++]  = "controlHigh";
	strings[i++]  = "controlLow";
	strings[i++]  = "bitMask";
	strings[i++]  = "file";
	strings[i++]  = "class";
	strings[i++]  = "device";
	strings[i++]  = "message";
	strings[i++]  = "verb";
	strings[i++]  = "attribute";
	strings[i++]  = "collection";
	strings[i++]  = "resultCode"; 
	// the completion code of a server operation
	strings[i++]  = "clientID";	
	// Used in context to identify a client
	strings[i++]  = "cycleType"; 
	// Timing system parameter
	strings[i++]  = "timingEvent";

	try {
	    for(i=1; i<=MAXSTRINGS; i++) 
		insert (i, strings[i-1]);
	} catch (IllegalArgumentException e) {
	}
	initialized_ = true;
	
    }

    /**
     * Add an event listener
     * return true if this event listener is registered
     * return false if this event listener is already registered
     */
    public boolean addTagChangedEventListener (cdevTagChangedEventListener el)
    {
	if (eventListeners_.contains (el) == true)
	    return false;
	eventListeners_.addElement (el);
	return true;
    }

    public static void main(String argv[]) 
    {
	cdevTagTable t = cdevTagTable.tagTable();
	String temp = "Facility";

	t.insert (1024, "user");
	t.insert (1025, "host");	
	t.insert (1026, "pid");
	t.add (temp);

	int size = 0;
	int s;

	try {
	    size = t.numTags ();
	}catch (IllegalStateException e) {
	    System.err.println (e);
	    System.exit (1);
	}

	int[] itags = new int[size];
	String[] stags = new String[size];

	s = t.readTagTable (itags, stags, size);

	for (int i = 0; i < s; i++) {
	    System.out.println ("Tag: " + String.valueOf (itags[i]) + " " + stags[i]);
	}
    }

    /**
     * Convert a String tag to its corresponding integer tag.
     *  @param key String tag
     */
    public int convertKey (String key) throws IllegalArgumentException
    {
	Integer itag = (Integer) skey_.get(key);
	if (itag == null) {
	    throw new IllegalArgumentException ("String tag: " + key + " : has not been registered");
	}
	return itag.intValue();
    }

    /**
     * Convert an Integer tag to its corresponding String tag.
     *  @param key integer tag
     */
    public String convertKey (int key)
    {
	Integer itag = new Integer(key);
	String  stag = (String)ikey_.get(itag);
	return stag;
    }

    /**
     *  Insert a pair of integer and string tag.
     *  @param tag  integer tag
     *  @param ctag string tag
     */
    public void insert (int tag, String ctag) throws IllegalArgumentException 
    {
	Integer itag = new Integer(tag);

	if (ikey_.containsKey(itag)) throw new IllegalArgumentException
					 ("duplicate integer " + itag.toString());
	if (skey_.containsKey(ctag)) throw new IllegalArgumentException
					 ("duplicate tag name " + ctag);

	ikey_.put(itag,ctag);
	skey_.put(ctag,itag);

	notifyEventListeners (tag, ctag);
    }

    /**
     * Find out whether a string tag exists
     * @param ctag interested string tag
     */    
    public boolean tagExists (String ctag)
    {
	if (skey_.containsKey(ctag))
	    return true;
	return false;
    }

    /**
     * Find out whether an integer tag exists
     * @param itag interested integer tag
     */    
    public boolean tagExists (int itag)
    {
	Integer i = new Integer (itag);
	if (ikey_.containsKey(i))
	    return true;
	return false;
    }

    /**
     * Add a string tag to the system. The system will assign corresponding
     * integer tag value.
     * @param ctag a string tag that is going to be added
     */
    public void add (String ctag) throws IllegalArgumentException 
    {
	if (skey_.containsKey(ctag)) throw new IllegalArgumentException
					 ("Tag " + ctag + " Already exists");

	// find out largest integer tag
	Enumeration itaglist = skey_.elements ();
	Integer itagele = null;
	Object  tmp = null;
	int     max = 0;

	while (itaglist.hasMoreElements ()) {
	    tmp = itaglist.nextElement ();
	    itagele = (Integer)tmp;
	    if (itagele.intValue () > max)
		max = itagele.intValue();
	}
	
	// increase maximum value by 1
	max += 1;
	
	insert (max, ctag);
    }

    /**
     * Return number of tag pairs in the system
     */
    public int numTags () throws IllegalStateException 
    {
	if (ikey_.size () != skey_.size ())
	    throw new IllegalStateException 
		("Integer tag list has different size from string tag list");
	
	return ikey_.size ();
    }
	
    /**
     * Fill tag table information to an integer array and a string array
     * @param itags allocated integer array of size 'size'
     * @param stags allocated string  array of size 'size'
     * @param size  size of arrays
     * return number of tags found
     */
    public int readTagTable (int[] itags, String[] stags, int size)
    {
	Enumeration itaglist = ikey_.keys();
	Integer itagele = null;
	String  stag = null;

	int i = 0;
	while (itaglist.hasMoreElements()) {
	    if (i >= size)
		break;
	    itagele = (Integer)(itaglist.nextElement());
	    stag = (String)ikey_.get (itagele);

	    itags[i] = itagele.intValue();
	    stags[i] = stag;
	    i++;
	}
	return i;
    }

    private void notifyEventListeners (int tag, String name)
    {
	Enumeration elist = eventListeners_.elements ();
	cdevTagChangedEventListener el = null;

	while (elist.hasMoreElements()) {
	    el = (cdevTagChangedEventListener)(elist.nextElement());
	    el.tagChanged (new cdevTagChangedEvent(tag, name));
	}
    }


    /*
     * Return a single tag table of this system
     */
    static public synchronized cdevTagTable tagTable() 
    {
	if (gTagTable_ == null)
	    gTagTable_ = new cdevTagTable();
	return gTagTable_;
    }
}

