//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      Browser reader that is reading data from TCP socket
//      
//      This class could be a private class of Browser. No synchronized
//      lock inside this class. All locks are acquired from Browser
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: BrowserReader.java,v $
//   Revision 1.1  2000/01/04 14:26:10  chen
//   new implementation
//
//
//
package cmlog;

import java.io.*;
import java.net.*;
import java.util.*;

public final class BrowserReader implements Runnable
{
    // disconnection callback list
    private Vector discCbkList_ = null;

    // query callback list
    private Hashtable queryCbkTable_ = null;

    // monitor on callback list
    private Hashtable monitorCbkTable_ = null;

    // input stream
    private DataInputStream input_ = null;

    // pointer back to cmlog Browser
    private Browser Browser_ = null;


    // application query result will be executed inside this thread
    class queryThread extends Thread
    {
	private BrowserQueryEventListener qlistener_ = null;
	private BrowserEvent              qevent_ = null;

	queryThread (BrowserQueryEventListener l, BrowserEvent ev)
	{
	    qlistener_ = l;
	    qevent_ = ev;
	}

	public void run ()
	{
	    qlistener_.handleQueryEvent (qevent_);
	}
    }

    /**
     * Construct a BrowserReader from an established Browser
     */
    public BrowserReader (Browser Browser)
    {
	Browser_ = Browser;

	discCbkList_ = new Vector ();
	queryCbkTable_ = new Hashtable (20, (float)0.25);
	monitorCbkTable_ = new Hashtable (20, (float)0.25);

	InputStream tinput = null;
	Socket      socket = null;

	try {
	    socket = Browser_.getSocket();
	}catch (NullPointerException e) {
	    System.err.print (e);
	    System.exit (-1);
	}

	try {
	    tinput = socket.getInputStream();
	}catch (IOException e) {
	    System.err.print (e);
	    System.exit (-1);
	}
	input_ = new DataInputStream (tinput);
    }

    /**
     * Clean up all event handlers
     */
    public void cleanupEventHandlers ()
    {
	discCbkList_.removeAllElements ();
	queryCbkTable_.clear ();
	monitorCbkTable_.clear ();
    }

    /**
     * Dispatch a disconnection event to all disconnection listeners
     */
    public void dispatchDiscEvent ()
    {
	BrowserEvent discevent = new BrowserEvent ();
	discevent.status = Config.CMLOG_DISCONNECTED;
	
	Enumeration list = discCbkList_.elements ();
	BrowserDisconnectEventListener handler = null;

	while (list.hasMoreElements ()) {
	    handler = (BrowserDisconnectEventListener)(list.nextElement());
	    handler.handleDisconnectEvent (discevent);
	}
    }


    /**
     * Add a query listener
     */
    public void addQueryListener (int evid, BrowserEvent oev)
    {
	queryCbkTable_.put (new Integer (evid), oev);
    }

    /**
     * Check whether this monitor event listener is already registered
     */
    public boolean hasMonitorListener (BrowserMonitorEventListener l)
    {
	Enumeration list = monitorCbkTable_.elements ();
	Object      tmp = null;
	BrowserEvent tev = null;
	BrowserMonitorEventListener tl = null;

	while (list.hasMoreElements ()) {
	    tmp = list.nextElement ();
	    tev = (BrowserEvent)tmp;
	    tl = (BrowserMonitorEventListener)tev.listener;
	    if (tl == l)
		return true;
	}
	return false;
    }


    /**
     * Find monitor event associated with a listener 
     */
    public BrowserEvent findMonitorEvent (BrowserMonitorEventListener l)
    {
	Enumeration list = monitorCbkTable_.elements ();
	Object      tmp = null;
	BrowserEvent tev = null;
	BrowserMonitorEventListener tl = null;

	while (list.hasMoreElements ()) {
	    tmp = list.nextElement ();
	    tev = (BrowserEvent)tmp;
	    tl = (BrowserMonitorEventListener)tev.listener;
	    if (tl == l)
		return tev;
	}
	return null;
    }

    /**
     * Add a monitor listener
     */
    public void addMonitorListener (int evid, BrowserEvent oev)
    {
	monitorCbkTable_.put (new Integer (evid), oev);
    }

    /**
     * Add a disconnection event listener
     */
    public boolean addDisconnectListener (BrowserDisconnectEventListener listener)
    {
	if (discCbkList_.contains (listener) == true)
	    return false;
	discCbkList_.addElement (listener);
	return true;
    }


    /**
     * Find out whether an event is in the query listener table
     */
    public boolean validQueryEvent (BrowserEvent oev)
    {
	if (queryCbkTable_.containsKey (new Integer(oev.evid)) == true)
	    return true;
	return false;
    }

    /**
     * Find out whether an event is in the query listener table
     */
    public boolean validMonitorEvent (BrowserEvent oev)
    {
	if (monitorCbkTable_.containsKey (new Integer(oev.evid)) == true)
	    return true;
	return false;
    }

    /**
     * Implementation of runnable run interface
     */
    public void run ()
    {
	Packet packet = new Packet();
	cdevMessage[]  msgs = null;
	cdevData    data = null;
	cdevDataEntry de = null;
	cdevMessage idata = null;
	String      message = null;
	String      verb = null;
	String      attr = null;
	int         errcode = Config.CMLOG_ERROR;
	int         evid    = 0;

	while (true) {
	    try {
		packet.streamIn (input_);
	    }catch (IOException e) {
		System.err.println ("Server maybe down: " + e);
		Browser_.handleClose ();
		break;
	    }
	    msgs = packet.messages ();

	    // look at the first message which tells status of a command
	    idata = msgs[0];
	    message = idata.getMessage ();
	    
	    if (message != null) {
		// parse message into two words
		String[] words = new String[2];
		int      numwords = 0;

		try {
		    numwords = Browser.parseMessage (message, words, 2);
		}catch (IllegalArgumentException e) {
		    System.err.println (e);
		    continue;
		}catch (IOException ie) {
		    System.err.println (ie);
		    continue;
		}
		verb = words[0];
		attr = words[1];

		// debug printout 
		if (Config.debug == true) {
		    if (numwords == 1)
			System.out.println ("Reader Thread Verb: " + verb);
		    else
			System.out.println ("Reader Thread Verb: " + verb +" : Attribute: " + attr);
		}
		
		if (numwords == 1) {
		    String upverb = verb.toUpperCase ();
		    if (upverb.compareTo ("QUERY") == 0) {
			data = idata.getData ();
			if (data != null && 
			    (de = data.get ("resultCode")) != null)
			    errcode = de.intValue();
			evid = idata.getForeignDataIndex ();
			queryEventFromServer (errcode, evid, packet);
		    }
		}
		else if (numwords == 2) {
		    String upverb = verb.toUpperCase (); 
		    if (upverb.compareTo ("MONITORON") == 0 || 
			upverb.compareTo ("MONITOROFF") == 0) {
			data = idata.getData ();
			if (data != null &&
			    (de = data.get ("resultCode")) != null)
			    errcode = de.intValue ();
			evid = idata.getForeignDataIndex ();
			monitorEventFromServer (errcode, evid, attr, packet);
		    }
		}
	    }
	}
    }

    // dispatch query event from server
    private void queryEventFromServer (int errcode, int evid, Packet packet)
    {
	Object  tmp = null;
	BrowserEvent oevent = null;
	BrowserEvent nevent = null;
	BrowserQueryEventListener handler = null;
	Integer key = new Integer (evid);
	
	tmp = queryCbkTable_.get (key);
	if (tmp != null) {
	    oevent = (BrowserEvent)tmp;
	    handler = (BrowserQueryEventListener)oevent.listener;

	    if (Config.debug == true) {
		System.out.println ("Incoming query event id is : " + String.valueOf (evid) + " errcode is : " + String.valueOf (errcode));
		System.out.println ("Found existing event has id: " + String.valueOf (oevent.evid) + " has verb: " + oevent.verb);
	    }
				    

	    nevent = new BrowserEvent();
	    nevent.status = errcode;
	    nevent.reqid = oevent.reqid;
	    nevent.evid = oevent.evid;
	    nevent.verb = oevent.verb;
	    nevent.attr = null;
	    nevent.packet = new Packet(packet);

	    if (errcode != Config.CMLOG_INCOMPLETE)
		queryCbkTable_.remove (key);
	    
	    // spawn a new thread to handle possible long query result
	    // to free this thread
	    queryThread qthread = new queryThread (handler, nevent);
	    qthread.start ();
	}
	else {
	    if (Config.debug == true) 
		System.out.println ("Unmatched query event id is : " + String.valueOf (evid) + " errcode is : " + String.valueOf (errcode));
	}

    }

    // dispatch monitor event from server
    private void monitorEventFromServer (int errcode, int evid, 
					 String attr, Packet packet)
    {
	Object  tmp = null;
	BrowserEvent oevent = null;
	BrowserEvent nevent = null;
	BrowserMonitorEventListener handler = null;
	Integer key = new Integer (evid);
	
	tmp = monitorCbkTable_.get (key);
	if (tmp != null) {
	    oevent = (BrowserEvent)tmp;
	    handler = (BrowserMonitorEventListener)oevent.listener;

	    if (Config.debug) {
		System.out.println ("Incoming monitor event id is : " + String.valueOf (evid) + " errcode is : " + String.valueOf(errcode));
		System.out.println ("Found existing event has id: " + String.valueOf (oevent.evid) + " has verb: " + oevent.verb + " has attr: " + oevent.attr);
	    }

	    nevent = new BrowserEvent();
	    nevent.status = errcode;
	    nevent.reqid = oevent.reqid;
	    nevent.evid = oevent.evid;
	    nevent.verb = oevent.verb;
	    nevent.attr = oevent.attr;
	    nevent.packet = packet;

	    handler.handleMonitorEvent (nevent);

	    if (errcode == Config.CMLOG_NOTFOUND)
		monitorCbkTable_.remove (key);
	    else if (errcode == Config.CMLOG_CBK_FINISHED)
		monitorCbkTable_.remove (key);
	}
	else {
	    if (Config.debug == true) 
		System.out.println ("Unmatched monitor event id is : " + String.valueOf (evid) + " errcode is : " + String.valueOf (errcode));
	}
    }
   

}	

