//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG Java Browser Interface
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: Browser.java,v $
//   Revision 1.4  2002/04/04 17:17:55  chen
//   Fix a bug for multiple java clients
//
//   Revision 1.3  2001/10/18 18:44:57  chen
//   Change API for monitor
//
//   Revision 1.2  2000/01/04 18:57:08  chen
//   Bug fixes for unsigned short port number
//
//   Revision 1.1  2000/01/04 14:26:09  chen
//   new implementation
//
//
//
package cmlog;

import java.io.*;
import java.net.*;
import java.util.*;

public final class Browser implements cdevTagChangedEventListener
{
    // flag of connection
    boolean connected_ = false;

    // tcp socket to the server
    private Socket tcpsocket_ = null;

    // to notify tag map changes to the server
    private boolean tagmapNotify_ = true;

    // Server information
    private String serverHost_ = null;
    private int    serverPort_ = 0;

    // request id
    private int    qid_ = 93;
    // event handler id
    private int    evid_ = 1024;

    // internal Browser data reader
    private BrowserReader reader_ = null;
    
    // internal thread id for the reader
    private Thread readerThread_ = null;

    // output stream
    private OutputStream output_ = null;

    // udp sending back off array
    private static int udpBackOff[] = {1, 2, 4, 8, 16, 32};
    
    // maximum number of udp retries
    private static int maxUdpRetries = 3;


    public void tagChanged (cdevTagChangedEvent event)
    {
	if (connected_ && tagmapNotify_ == true) {
	    try {
		sendTagmapInfo (event.tag(), event.name());
	    }catch (IOException e) {
		System.err.println ("Cannot send tagmap info for tag: " + event.tag() + " tag name: " + event.name());
	    }
	}
    }

    /**
     * Construct an empty Browser 
     */
    public Browser ()
    {
	// set up all tags
	Config.setupTags ();

	// register this Browser to tag table to listen on the tag changed event
	cdevTagTable tbl = cdevTagTable.tagTable();
	
	tbl.addTagChangedEventListener (this);
    }

    /**
     * Override default finalize method.This allows Java virtual
     * machine to clean up resource when this object is no longer
     * needed.
     */
    protected void finalize() throws Throwable
    {
	if (tcpsocket_ != null && connected_ == true) {
	    connected_ = false;
	    tcpsocket_.close ();
	}
    }

    /**
     * Connect to a server that is on a given host at a given port
     */
    public synchronized void connect (String host, int port) 
	throws UnknownHostException, IOException
    {
	// contack udp port to find out the TCP listener port
	try {
	    udpContact (host, port);
	}catch (UnknownHostException e) {
	    throw e;
	}catch (IOException ue) {
	    throw ue;
	}
	
	if (Config.debug == true) 
	    System.out.println ("Cmlog Server has TCP port " + String.valueOf (serverPort_));
	
	try {
	    tcpsocket_ = new Socket (host, serverPort_);
	}catch (UnknownHostException ue){
	    System.err.println (ue);
	    throw ue;
	}catch (IOException e) {
	    System.err.println(e);
	    throw e;
	}
	
	// set connection flag
	connected_ = true;
	serverHost_ = new String (host);

	if (Config.debug) 
	    System.out.println ("Connection to cmlog server is established");

	try {
	    output_ = tcpsocket_.getOutputStream ();
	}catch (IOException e) {
	    System.err.println (e);
	    connected_ = false;
	    tcpsocket_.close ();
	    throw e;
	}

	// send out this Browser information
	try {
	    sendBrowserInfo ();
	}catch (IOException e) {
	    System.err.println (e);
	    connected_ = false;
	    tcpsocket_.close ();
	    throw e;
	}

	// create a data reader and spawn a new thread
	reader_ = new BrowserReader (this);

	readerThread_ = new Thread (reader_);
	readerThread_.start ();
    }

    /**
     * Return underlying socket
     */
    public Socket getSocket () throws NullPointerException
    {
	if (tcpsocket_ == null)
	    throw new NullPointerException ("Socket is invalid");

	return tcpsocket_;
    }   

    /**
     * Connected or not
     */
    public boolean connected ()
    {
	return connected_;
    }

    /**
     * Return underlying reader thread
     */
    public Thread getReaderThread () throws NullPointerException
    {
	if (readerThread_ == null)
	    throw new NullPointerException ("Reader Thread is not alive");
	return readerThread_;
    }

    /**
     * Disconnect from the server
     */
    public synchronized void disconnect () throws IOException
    {
	if (tcpsocket_ == null || connected_ == false) 
	    throw new IOException ("Socket is not connected to the server");

	// remove all event handlers
	reader_.cleanupEventHandlers ();

	try {
	    tcpsocket_.close();
	}catch (IOException e) {
	    throw e;
	}
	connected_ = false;

	if (readerThread_ != null) 
	    readerThread_.stop ();
    }

    /**
     * Enable tag map notification to the server
     */
    public void enableTagmapNotify ()
    {
	tagmapNotify_ = true;
    }

    /**
     * Disable tag map notification to the server
     */
    public void disableTagmapNotify ()
    {
	tagmapNotify_ = false;
    }

    /**
     * Check whether tagmap notification is enabled
     */
    public boolean tagmapNotifyEnabled ()
    {
	return tagmapNotify_;
    }

    /**
     * Register an event handler to handle disconnection event
     */
    public synchronized boolean addDisconnectEventListener 
	(BrowserDisconnectEventListener listener)
    {
	return reader_.addDisconnectListener (listener);
    }


    /**
     * Register an event handler to handle monitor event
     */
    public synchronized boolean addMonitorEventListener 
	(cdevData data, BrowserMonitorEventListener listener)
    {
	if (connected_ == false) {
	    System.err.println ("TCP connection is not established");
	    return false;
	}
	
	if (reader_.hasMonitorListener (listener) == true)
	    return false;

	if (qid_ > 65535)
	    qid_ = 93;
	if (evid_ == Integer.MAX_VALUE)
	    evid_ = 1024;

	String message = "monitorOn loggingData";

	cdevMessage msg = 
	    new cdevMessage ((short)0, qid_++, 0, 0, evid_++, 0, null,
			     message, data, null, null);

	try {
	    sendMessage (msg);
	} catch (IOException e) {
	    System.err.println (e);
	    return false;
	}
	    
	
	// create new BrowserEvent to register to callback table
	BrowserEvent oev = new BrowserEvent();
	oev.reqid = qid_ - 1;
	oev.evid = evid_ - 1;
	oev.verb = "monitorOn";
	oev.attr = "loggingData";
	oev.listener = listener;

	reader_.addMonitorListener (evid_ - 1, oev);

	return true;
    }

    /**
     * Remove a monitor event listener
     */
    public synchronized boolean removeMonitorEventListener
	(cdevData data, BrowserMonitorEventListener listener)
    {
	BrowserEvent oev = reader_.findMonitorEvent (listener);
	if (oev == null)
	    return false;

	if (qid_ > 65535)
	    qid_ = 93;

	String message = "monitorOff loggingData";

	cdevMessage msg = 
	    new cdevMessage ((short)0, qid_++, 0, 0, oev.evid, 0, null,
			     message, data, null, null);

	try {
	    sendMessage (msg);
	} catch (IOException e) {
	    System.err.println (e);
	    return false;
	}
	return true;
    }

    /**
     * Handle server shutdown
     */
    public void handleClose ()
    {
	try {
	    tcpsocket_.close ();
	} catch (IOException e) {
	    System.err.println (e);
	    connected_ = false;
	}
	connected_ = false;
	
	// call all disconnection event handler
	reader_.dispatchDiscEvent ();

	// clean up all event handlers
	reader_.cleanupEventHandlers ();
    }


    /**
     * parse input message into verb and attribute
     */
    public static int parseMessage (String message, String[] words, int len)
	throws IOException, IllegalArgumentException
    {
	StringReader sreader = new StringReader (message);
	StreamTokenizer ts = new StreamTokenizer (sreader);
	int             tokentype;
	int             i;

	for (i = 0; i < len; i++)
	    words[i] = null;

	i = 0;
	while (i < len) {
	    try {
		tokentype = ts.nextToken();
	    }catch (IOException e){
		throw e;
	    }
	    if (tokentype == StreamTokenizer.TT_EOF)
		break;
	    if (tokentype == StreamTokenizer.TT_WORD) {
		words[i++] = new String(ts.sval);
	    }
	}
	if (i == 0) {
	    throw new IllegalArgumentException("Query callback message error");
	}
	return i;
    }

    /**
     * Send a query to the server
     * @param data     outbound data containing all information
     * @param listener event listener for this event
     */
    public synchronized BrowserEvent query (cdevData data, 
					    BrowserQueryEventListener listener) 
	throws IOException, IllegalArgumentException
    {
	if (connected_ == false) {
	    throw new IOException ("TCP connection is not established");
	}
	
	BrowserEvent oev = null;

	// always need time range for query function
	double start_time, end_time;
	cdevDataEntry de = null;

	try {
	    de = data.get ("start");
	}catch (IllegalArgumentException e) {
	    throw e;
	}
	start_time = de.doubleValue();

	try {
	    de = data.get ("end");
	}catch (IllegalArgumentException e) {
	    throw e;
	}
	end_time = de.doubleValue();

	if (end_time <= start_time) {
	    throw new IllegalArgumentException("Query end time is <= start time");
	}
		
	if (qid_ > 65535)
	    qid_ = 93;
	if (evid_ == Integer.MAX_VALUE)
	    evid_ = 1024;

	String verb = "query";
	cdevMessage msg = 
	    new cdevMessage ((short)0, qid_++, 0, 0, evid_++, 0, null, verb,
			     data, null, null);
	// flush out message
	try {
	    sendMessage (msg);
	}catch (IOException e) {
	    throw e;
	}

	// create new BrowserEvent to register to callback table
	oev = new BrowserEvent();
	oev.reqid = qid_ - 1;
	oev.evid = evid_ - 1;
	oev.verb = verb;
	oev.listener = listener;

	reader_.addQueryListener (evid_ - 1, oev);
	
	return oev;
    }


    /**
     * Stop or cancel a command that was sent previously
     * @param data    related outbound data
     * @param event   generated event by previous execution command
     */
    public synchronized void stopQuery (cdevData data,
					BrowserEvent event)
	throws IOException, IllegalArgumentException
    {
	if (connected_ == false) {
	    throw new IOException ("TCP connection is not established");
	}
	
	String message = "stopQuery";

	BrowserEvent oev = null;
	if (reader_.validQueryEvent (event) == true) {
	    if (qid_ > 65535)
		qid_ = 93;

	    cdevMessage msg = 
		new cdevMessage ((short)0, qid_++, 0, 0, event.evid, 0, null,
				 message, null, null, null);

	    try {
		sendMessage (msg);
	    } catch (IOException e) {
		throw e;
	    }
	}
	else
	    throw new IllegalArgumentException ("Trying to stop a query with an invalid event");
    }
					    

    private void udpContact (String host, int port) 
	throws UnknownHostException, IOException
    {
	// Talk to UDP port on a given host
	DatagramSocket udpsocket = null;
	InetAddress    dest = null;
	int            isize = 2;  // receiving data buffer size
	int            osize = 4;  // output data buffsize
	try {
	    udpsocket = new DatagramSocket ();
	}catch (IOException e) {
	    System.err.println(e);
	    throw e;
	}
	try {
	    dest = InetAddress.getByName (host);
	}catch (UnknownHostException ue) {
	    throw ue;
	}

	// convert a udp integer protocol number into a byte array
	ByteArrayOutputStream pout = new ByteArrayOutputStream (osize);
	cdevDataOutputStream iout =  new cdevDataOutputStream (pout, false);
	try {
	    iout.write (Config.CMLOG_BRSERLKSVC);
	}catch (IOException e) {
	    throw e;
	}
	// convert this byte array
	byte[] outputbuffer = pout.toByteArray ();
	if (outputbuffer.length != osize) {
	    throw new IOException ("Stream UDP protocol error");
	}
	// construct a output dataPacket
	DatagramPacket outpacket = new DatagramPacket (outputbuffer,
						       osize, dest, port);
	// construct a input dataPacket
	byte[] input = new byte[isize];
	DatagramPacket inputpacket = new DatagramPacket (input, isize);

	int i = 0;
	while (i < Browser.maxUdpRetries) {
	    try {
		udpsocket.setSoTimeout (Browser.udpBackOff[i] * 1000);
	    }catch (SocketException e) {
		System.err.println (e);
		throw new IOException ("Cannot set udp socket timeout");
	    }
	    // send data
	    try {
		udpsocket.send (outpacket);
	    }catch (IOException e) {
		throw e;
	    }
	    // receive data 
	    try {
		udpsocket.receive (inputpacket);
	    }catch (InterruptedIOException e) {
		System.err.println ("Timeout on udp read");
		i++;
		continue;
	    }
	    if (Config.debug == true)
		System.err.println ("UDP read getting something");
	    break;
	}
	if (Config.debug == true) 
	    System.out.println ("UDP received " + String.valueOf (inputpacket.getLength()) + " bytes");

	if (inputpacket.getLength() < isize) {
	    throw new IOException ("UDP receiving data size is less that 2");
	}
	// convert stream into a tcp port number
	ByteArrayInputStream istream = new ByteArrayInputStream (input);
	DataInputStream  distream = new DataInputStream (istream);

	short tmpport = 0;
	try {
	    tmpport = distream.readShort ();
	}catch (IOException e) {
	    System.err.println (e);
	    serverPort_ = 0;
	    throw e;
	}
	if (tmpport < 0)  // server sends back actually an unsigned short
	    serverPort_ = 65535 + tmpport + 1; 
	else
	    serverPort_ = (int)tmpport;
    }

    // send Browser information upon connection
    private void sendBrowserInfo () throws IOException
    {
	Properties sysp = System.getProperties ();

	// get user name
	String username = sysp.getProperty ("user.name");

	// get host name
	boolean foundhost = true;
	String  host;
	InetAddress loc = null;

	try {
	    loc = InetAddress.getLocalHost();
	}catch (UnknownHostException e) {
	    foundhost = false;
	}
	if (foundhost == true)
	    host = loc.getHostName ();
	else
	    host = "unknown";

	cdevData  cxt = new cdevData ();
	cxt.insert (Config.HOST_TAG, host);
	cxt.insert (Config.USER_TAG, username);
	cxt.insert (Config.PID_TAG,  1024);
	cxt.insert (Config.DISP_TAG, "java");


	cdevMessage msg = new cdevMessage ((short)0, 0, 0, 0, 0,
					   Config.CMLOG_CONN_INFO,
					   null, null, cxt, null, null);
	try {
	    sendMessage (msg);
	}catch (IOException e) {
	    throw e;
	}
    }

    // send out a single cdev message
    private void sendMessage (cdevMessage msg) throws IOException
    {
	Packet qpacket = new Packet();

	if (qpacket.overflow (msg) == true) {
	    throw new IOException ("A single message overflows packet");
	}
	qpacket.insert (msg);

	try {
	    qpacket.streamOut (output_);
	}catch (IOException e) {
	    throw e;
	}

	// flush data out
	try {
	    output_.flush ();
	}catch (IOException e) {
	    throw e;
	}
    }
	

    // send tag map information
    private void sendTagmapInfo (int tag, String name)
	throws IOException
    {
	cdevData tagmap = new cdevData ();

	tagmap.insert (1, tag);
	tagmap.insert (2, name);
	cdevMessage msg = new cdevMessage ((short)0, 0, 0, 0, 0,
					   Config.CMLOG_TAGMAP,
					   null, null, null, null, tagmap);

	try {
	    sendMessage (msg);
	}catch (IOException e) {
	    throw e;
	}
    }
    
    
}    

    
    




