//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
// Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
//-----------------------------------------------------------------------------
// 
// Description:
//	 cmlogStrHash: cmlog hash table keyed by a variable length string
//                    Open Hash with buckets implemented by single linked lists
//    
//              Note: void *s are stored inside the table. This class
//                    will not manage these pointers. Callers have to
//                    manage these pointers
// 
//              Note: this is unsafe C++ practice. Use at your own risk
//              
//            Reason: It is so difficult to use a template class inside
//                    a shared library. (Cfront based C++ compiler cannot
//                    instantiate a template class during compilation time
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogStrHash.h,v $
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:35  bickley
// Added cmlog to application development system.
//
//
//
//
#ifndef _CMLOG_STR_HASH_H
#define _CMLOG_STR_HASH_H

#include <cmlogSlist.h>

//======================================================================
//  One simple string hash function
//======================================================================
extern unsigned int cmlogStrHashFunc (char *str);

typedef char* cmlogKeyItem;
typedef void* cmlogHashItem;

//======================================================================
//	class cmlogStrHash
//		collection of buckets indexed by hashed values
//======================================================================
class cmlogStrHashIterator;

class cmlogStrHash
{
public:
  // constructor
  // construct a table with entry max and hash function *f
  // hash function return any integer from 0 to MAX_INT_VALUE
  cmlogStrHash (unsigned int max, unsigned int (*f)(cmlogKeyItem));
  // destructor
  virtual ~cmlogStrHash (void);

  // operations

  // is the table empty: return 1: yes. return 0: no
  virtual int   isEmpty();

  // clear the elements of the set
  virtual void  deleteAllValues();

  // add an element to the collection
  virtual void  add (cmlogKeyItem key, cmlogHashItem ele);

  // test to see whether this hash table includes one particular element
  virtual int   find (cmlogKeyItem key, cmlogHashItem ele) const;

  // remove an element. return 0: ele is not inside table. return 1: success
  virtual int   remove (cmlogKeyItem key, cmlogHashItem ele);

  // return a reference to a particular bucket according to the key
  cmlogSlist&  bucketRef (cmlogKeyItem key);

protected:
  friend class cmlogStrHashIterator;

  // the actual table itself is a vector of buckets
  const unsigned int    tablesize;
  cmlogSlist*            buckets;

  // convert key into unsigned integer value in range
  unsigned int          hash(const cmlogKeyItem& key) const;

private:
  unsigned int          (*hashCode_)(cmlogKeyItem);

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutex               lock_;
#endif
};

//======================================================================
//	class cmlogStrHashIterator
//		iterator protocol for hash tables
//======================================================================
class cmlogStrHashIterator
{
public:
  // constructor and destructor
  cmlogStrHashIterator  (cmlogStrHash& v);
  ~cmlogStrHashIterator (void);

  // iterator protocol 
  int             init       (void);
  cmlogHashItem    operator ()(void);
  int             operator ! (void);
  int             operator ++(void);
  void            operator = (cmlogHashItem value);

  // new operations
  // remove current item pointed by this cursor
  void            removeCurrent (void);

protected:
  cmlogStrHash&    base;
  unsigned int    currentIndex;
  // Single iterator within a bucket
  cmlogSlistIterator*         itr;
  // getNextIterator used to set internal iterator pointer
  // return 1: got it. return 0: no more iterator
  int             getNextIterator (void);

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutex               lock_;
#endif
};
#endif
