//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of cmlogDatabase Class
//
// Author:  Jie Chen
//
// Revision History:
//   $Log: cmlogDatabase.cc,v $
//   Revision 1.4  2001/07/25 14:29:03  chen
//   64 BIT Initial Port
//
//   Revision 1.3  2000/03/01 16:56:44  chen
//   Cocurrent control + db patches
//
//   Revision 1.2  2000/02/07 16:12:07  chen
//   detect one case of corrupt files
//
//   Revision 1.1.1.1  1999/09/07 15:29:15  chen
//   CMLOG version 2.0
//
// Revision 1.3  1997/09/15  14:29:50  chen
// change PAGE_SIZE to DBASE_PAGESIZE
//
// Revision 1.2  1997/09/15  13:49:00  chen
// add page size as configuration parameter
//
// Revision 1.1  1997/08/01  15:34:36  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogConfig.h>
#include <cmlogUtil.h>
#include "cmlogDatabase.h"

#ifdef CMLOG_64BIT_LONG
#define size_t unsigned
#endif

// Note: Data stored in the database/mpool are not memory aligned.
// Use memcpy if necessary

double cmlogDatabase::tinyValue = 0.00001;

// compare function for keys
static int cmlogKeyComp (const DBT* key1, const DBT* key2)
{
  register size_t len;
  double   s1, s2, res;

  /*
   * XXX
   * If a size_t doesn't fit in an int, this routine can lose.
   * What we need is a integral type which is guaranteed to be
   * larger than a size_t, and there is no such thing.
   */
  len = key1->size;

  memcpy (&s1, key1->data, len);
  memcpy (&s2, key2->data, len);

  res = s1 - s2;
  if (res > cmlogDatabase::tinyValue) 
    return 1;
  else if (res < (-1)*cmlogDatabase::tinyValue) 
    return -1;
  else 
    return 0;
}

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogDatabase::cmlogDatabase (void)
:flag_ (0), mode_ (0), name_ (0), dbp_ (0), wsize_ (0), mutex_ ()
#else
cmlogDatabase::cmlogDatabase (void)
:flag_ (0), mode_ (0), name_ (0), dbp_ (0), wsize_ (0)
#endif
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogDatabase Class Object\n");
#endif
}

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogDatabase::cmlogDatabase (char* name, int flags, int mode)
:flag_ (flags), mode_ (mode), dbp_ (0), wsize_ (0), mutex_ ()
#else
cmlogDatabase::cmlogDatabase (char* name, int flags, int mode)
:flag_ (flags), mode_ (mode), dbp_ (0), wsize_ (0)
#endif
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogDatabase Class Object\n");
#endif

  name_ = new char[::strlen (name) + 1];
  ::strcpy (name_, name);

  // initialize btree information structure.
  // different site can change these settings if they want
  b_.flags = R_DUP;  // allow duplicated key in the tree
  b_.cachesize = 0;  // system will chose default 5 pages
  b_.psize = cmlogUtil::DBASE_PAGESIZE;   // 4 k as page size
  b_.minkeypage = CMLOG_MIN_KEY_PAGE;
  // minimum number of keys stored on a page
  b_.maxkeypage = 0; // let system pick one
  b_.prefix = 0;     // no prefix function
  b_.lorder = 0;     // use machine byte order
  b_.compare = cmlogKeyComp;  // comparition function

  dbp_ = ::dbopen (name_, flag_, mode_, DB_BTREE, &b_);

  if (dbp_ == NULL) {
    cmlogUtil::serverPrintf ("Fatal Error: cannot open database %s\n",name_);
    cmlogUtil::serverPrintf ("Errno is %d\n", errno);
  }
  else 
    // force database to write meta data on the disk
    (*dbp_->sync)(dbp_, 0);
}

cmlogDatabase::~cmlogDatabase (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogDatabase Class Object\n");
#endif

  if (name_)
    delete []name_;
  if (dbp_)
    (*dbp_->close)(dbp_);
  dbp_ = 0;
}

int
cmlogDatabase::open (char* name, int flags, int mode)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cmlogDatabaseLocker locker (this);  // lock the scope for multi-thread only
#endif

  /* first close old database */
  if (dbp_)
    (*dbp_->close)(dbp_);
  dbp_ = 0;
  if (name_)
    delete []name_;
  name_ = 0;


  /* open a new database */
  flag_  = flags;
  mode_  = mode;
  name_ = new char[::strlen (name) + 1];
  ::strcpy (name_, name);

  // initialize btree information structure.
  // different site can change these settings if they want
  b_.flags = R_DUP;  // allow duplicated key in the tree
  b_.cachesize = 0;  // system will chose default 5 pages
  b_.psize = cmlogUtil::DBASE_PAGESIZE;   // 4 k as page size
  b_.minkeypage = CMLOG_MIN_KEY_PAGE;
  // minimum number of keys stored on a page
  b_.maxkeypage = 0; // let system pick one
  b_.prefix = 0;     // no prefix function
  b_.lorder = 0;     // use machine byte order
  b_.compare = cmlogKeyComp;  // comparition function

  dbp_ = ::dbopen (name_, flag_, mode_, DB_BTREE, &b_);

  if (dbp_ == NULL) {
    cmlogUtil::serverPrintf ("Fatal Error: cannot open database %s\n",name_);
    return CDEV_ERROR;
  }
  else {
    // force database to write meta data on the disk
    (*dbp_->sync)(dbp_, 0);
  }    
  return CDEV_SUCCESS;
}

int
cmlogDatabase::close (void)
{
  cmlogDatabaseWLocker locker (this);  // lock the scope

  if (dbp_)
    (*dbp_->close)(dbp_);
  dbp_ = 0;
  if (name_)
    delete []name_;
  name_ = 0;
  return CDEV_SUCCESS;
}

int
cmlogDatabase::get (cdevData& data, cmlogKey* key)
{
  cmlogDatabaseLocker locker (this);

  if (!dbp_) 
    return CDEV_ERROR;

  DBT tkey, res;

  tkey.size = sizeof (double);
  tkey.data = (void *)key;
  
  int status;

  if ((status = (*dbp_->get)(dbp_, &tkey, &res, 0)) == 0)  // success
    return cmlogDatabase::convertData (data, &res);
  else if (status == 1) // not found
    return CDEV_NOTFOUND;
  else {
    if (errno == EIO) 
      cmlogUtil::serverPrintf ("cmlogDatabase: found corrupt file %s\n", name_);
    return CDEV_ERROR;
  }
}

int
cmlogDatabase::put (cdevData& data)
{
  cmlogDatabaseWLocker locker (this);

  int status;

  // if data size reaches the threshold, flush database
  if (wsize_ >= cmlogUtil::DBASE_PAGESIZE) {
    (*dbp_->sync)(dbp_, 0);
    wsize_ = 0;
  }
  

  if (dbp_ == 0)
    return CDEV_ERROR;

  DBT out, key;
  if (cmlogDatabase::convertData (&out, &key, data) != CDEV_SUCCESS)
    return CDEV_ERROR;

  // keep track of how many bytes one has written so far
  wsize_ += out.size;
  

  if ((status = (*dbp_->put)(dbp_, &key, &out, 0)) == 0) {
    // clean data
    char* buffer = (char *)out.data;
    delete []buffer;
    double *value = (double *)key.data;
    delete value;

    return CDEV_SUCCESS;
  }
  else
    return CDEV_ERROR;
}


int
cmlogDatabase::flush (void)
{
  cmlogDatabaseWLocker locker (this);

  if (dbp_ == 0)
    return CDEV_ERROR;

  if ((*dbp_->sync)(dbp_, 0) == 0)
    return CDEV_SUCCESS;
  else
    return CDEV_ERROR;
}

char*
cmlogDatabase::database (void) const
{
  return name_;
}

int
cmlogDatabase::fd (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cmlogDatabaseLocker locker (this);
#endif

  if (dbp_ == 0)
    return -1;

  return ((*dbp_->fd)(dbp_));
}

int
cmlogDatabase::cursorInit (cdevData& data, cmlogKey* key, int first)
{
  cmlogDatabaseLocker locker (this);

  if (!dbp_) 
    return CDEV_ERROR;

  DBT tkey, res;

  tkey.size = sizeof (double);
  tkey.data = (void *)key;
  
  int status;

  if (first == 0) 
    status = (*dbp_->seq)(dbp_, &tkey, &res, R_CURSOR);
  else
    status = (*dbp_->seq)(dbp_, &tkey, &res, R_FIRST);
  
  if (status == 0) {
    // success
    cmlogDatabase::copyKeyData (key, &tkey);
    return cmlogDatabase::convertData (data, &res);
  }
  else if (status == 1) {
    // not found
    return CDEV_NOTFOUND;
  }
  else {
    if (errno == EIO) 
      cmlogUtil::serverPrintf ("cmlogDatabase: found corrupt file %s\n", name_);
    return CDEV_ERROR;
  }
}  

int
cmlogDatabase::cursorNext (cdevData& data, cmlogKey* key)
{
  cmlogDatabaseLocker locker (this);

  if (!dbp_) 
    return CDEV_ERROR;

  DBT tkey, res;

  int status;

  if ((status = (*dbp_->seq)(dbp_, &tkey, &res, R_NEXT)) == 0) {  // success
    cmlogDatabase::copyKeyData (key, &tkey);
    return cmlogDatabase::convertData (data, &res);
  }
  else if (status == 1) // not found
    return CDEV_NOTFOUND;
  else
    return CDEV_ERROR;
}  

int
cmlogDatabase::cursorPrev (cdevData& data, cmlogKey* key)
{
  cmlogDatabaseLocker locker (this);

  if (!dbp_) 
    return CDEV_ERROR;

  DBT tkey, res;

  int status;

  if ((status = (*dbp_->seq)(dbp_, &tkey, &res, R_PREV)) == 0) {  // success
    cmlogDatabase::copyKeyData (key, &tkey);
    return cmlogDatabase::convertData (data, &res);
  }
  else if (status == 1) // not found
    return CDEV_NOTFOUND;
  else
    return CDEV_ERROR;
}  
  
int
cmlogDatabase::convertData (cdevData& data, const DBT* res)
{
  char* buffer = (char *)res->data;
  int   size   = res->size;

  // note: data out of database in not aligned

  char *nbuf = new char[size];
  memcpy (nbuf, buffer, size);
  if (data.xdrImport (nbuf, size) == CDEV_SUCCESS) {
    delete []nbuf;
    return CDEV_SUCCESS;
  }
  else {
    delete []nbuf;
    return CDEV_ERROR;
  }
}

int
cmlogDatabase::convertData (DBT* out, DBT* key, cdevData& data)
{
  cmlogKey *keyvalue = new double ();  

  if (data.get (cmlogUtil::CMLOG_KEY_TAG, keyvalue) != CDEV_SUCCESS)
    return CDEV_ERROR;

  key->data = (void *)(keyvalue);
  key->size = sizeof (cmlogKey);

  char*    buffer;
  size_t   size;

  if (data.xdrExport (&buffer, &size) == CDEV_SUCCESS) {
    out->data = (void *)buffer;
    out->size = size;

    return CDEV_SUCCESS;
  }
  else
    return CDEV_ERROR;
}

void
cmlogDatabase::copyKeyData (cmlogKey* key, DBT* data)
{
  register size_t len;
  register u_char *p1, *p2;

  /*
   * XXX
   * If a size_t doesn't fit in an int, this routine can lose.
   * What we need is a integral type which is guaranteed to be
   * larger than a size_t, and there is no such thing.
   */
  len = data->size;
  for (p1 = (u_char *)data->data, p2 = (u_char *)key; len--; ++p1, ++p2) 
    *p2 = *p1;
}

#ifndef __INLINE__
#include "cmlogDatabase.i"
#endif


#ifdef CMLOG_64BIT_LONG
#undef size_t
#endif
