//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Simple Token Class Implementation
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cpToken.cc,v $
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.2  1997/09/16  17:00:03  chen
// port to hpux-10 thread
//
// Revision 1.1  1997/08/01  15:30:35  bickley
// Added cmlog to application development system.
//
//
#include "cpToken.h"

#if !defined (__INLINE__)
#include "cpToken.i"
#endif /* __INLINE__ */

#ifndef __vxworks
cpQueueEntry::cpQueueEntry (cpMutex &m, 
			    cp_thread_t t_id)
:cv_ (m), next_ (0), thread_id_ (t_id), runable_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpQueueEntry Class Object\n");
#endif
}

cpToken::cpToken (int type)
: head_ (0), tail_ (0), lock_ (type),in_use_ (0), waiters_ (0),
  nesting_level_ (0)
{
#ifdef _TARCE_OBJECTS
  printf ("Create cpToken Class Object\n");
#endif
}

cpToken::~cpToken (void)
{
#ifdef _TARCE_OBJECTS
  printf ("Delete cpToken Class Object\n");
#endif
}

// Remove an entry from the list.  Must be 
// called with locks held.

void
cpToken::remove_entry (cpQueueEntry *entry)
{
  cpQueueEntry *curr = 0;
  cpQueueEntry *prev = 0;

  if (this->head_ == 0)
    return;

  for (curr = this->head_;
       curr != 0 && curr != entry;
       curr = curr->next_)
    prev = curr;

  if (curr == 0) // Didn't find the entry...
    return; 
  else if (prev == 0) // Delete at the head.
    this->head_ = this->head_->next_;
  else // Delete in the middle.
    prev->next_ = curr->next_;

  // We need to update the tail of the list 
  // if we've deleted the last entry. 

  if (curr->next_ == 0)
    this->tail_ = curr;
}

int 
cpToken::shared_acquire (void (*sleep_hook_func)(void *), 
			 void *arg,
			 const struct timespec *timeout)
{
  cpMutexGuard m (this->lock_);

  cp_thread_t thr_id = ::pthread_self ();

  if (in_use_) { // Someone already holds the token.
    if (pthread_equal (thr_id, this->owner_)) { // I own it!
      this->nesting_level_++;
      return 0;
    }
    // Do a quick check for "polling" behavior.
    else if (timeout != 0 && timeout->tv_sec == 0 && timeout->tv_nsec == 0) {
      errno = ETIME;
      return -1;
    }
    else {// We've got to sleep until we get the token.
      // Allocate q entry on stack.  This works since we don't
      // exit this method's activation record until we've got the
      // token.
      cpQueueEntry my_entry (this->lock_, thr_id);
      int ret = 0;

      if (this->head_ == 0) { // I'm first and only waiter in line...
	this->head_ = &my_entry;
	this->tail_ = &my_entry;
      }
      else { // I'm queued at the end of the list.
	this->tail_->next_ = &my_entry;
	this->tail_ = &my_entry;
      } 

      this->waiters_++;

      // Execute appropriate <sleep_hook> callback.
      // (@@ should these methods return a success/failure
      // status, and if so, what should we do with it?) 
      
      if (sleep_hook_func) {
	(*sleep_hook_func) (arg);
	ret++;
      }
      else { // Execute virtual method.
	this->sleep_hook ();
	ret++;
      }

      // Sleep until we've got the token (ignore signals).

      while (my_entry.cv_.wait (timeout) == -1) {
	// Note, this should obey whatever thread-specific
	// interrupt policy is currently in place...
	if (errno == EINTR)
	  continue;
	// We come here if a timeout occurs or some serious
	// Condition object error.
	this->remove_entry (&my_entry);
	return -1;
      }

      assert (my_entry.runable_);
      return ret;
    }
  }
  else {
    this->in_use_ = 1;
    this->owner_ = thr_id; // Its mine! 
    return 0;
  }
}

// By default this is a no-op.

/* virtual */
void
cpToken::sleep_hook (void)
{
  // empty
}

int 
cpToken::acquire (const struct timespec *timeout)
{
  return this->shared_acquire (0, 0, timeout);
}

// Acquire the token, sleeping until it is obtained or until
// <timeout> expires.

int 
cpToken::acquire (void (*sleep_hook_func)(void *), 
		  void *arg,
		  const struct timespec *timeout)
{
  return this->shared_acquire (sleep_hook_func, arg, timeout);
}

// Try to renew the token.
int
cpToken::renew (int requeue_position, const struct timespec *timeout)
{

  cpMutexGuard m (this->lock_);

  assert (pthread_equal (::pthread_self (), this->owner_));

  // Check to see if there are any waiters.  If not, we just keep the token.
  if (this->head_ != 0) {
    cpQueueEntry my_entry (this->lock_, this->owner_);
    int save_nesting_level_ = this->nesting_level_;

    this->owner_ = this->head_->thread_id_;
    this->nesting_level_ = 0;

    // Wake up next waiter and make it runable.
    this->head_->cv_.signal ();
    this->head_->runable_ = 1;

    this->head_ = this->head_->next_;

    if (this->head_ == 0) {// No other threads - just add me 
      this->head_ = &my_entry;
      this->tail_ = &my_entry;
    } 
    else if (requeue_position == -1) { // Insert at the end of the queue.
      this->tail_->next_ = &my_entry;
      this->tail_ = &my_entry;
    }
    else if (requeue_position == 0) { // Insert at head of queue.
      my_entry.next_ = this->head_;
      this->head_ = &my_entry;
    }
    else { // Insert in the middle of the queue somewhere.
      cpQueueEntry *insert_after = this->head_;

      // Determine where our thread should go in the queue of
      // waiters. 

      while (requeue_position-- && insert_after->next_ != 0)
	insert_after = insert_after->next_;

      my_entry.next_ = insert_after->next_;

      if (my_entry.next_ == 0)
	this->tail_ = &my_entry;

      insert_after->next_ = &my_entry;
    }

    // Sleep until we've got the token (ignore signals).

    while (my_entry.cv_.wait (timeout) == -1) {
      // Note, this should obey whatever thread-specific
      // interrupt policy is currently in place...
      if (errno == EINTR)
	continue;
      // We come here if a timeout occurs or 
      // some serious Condition object error.
      this->remove_entry (&my_entry);
      return -1;
    }

    assert (my_entry.runable_);
    this->nesting_level_ = save_nesting_level_;
    this->owner_ = my_entry.thread_id_;
  }
  return 0;
}

// Release the current holder of the token (which had
// better be the caller's thread!).
int
cpToken::release (void)
{
  cpMutexGuard m (this->lock_);

  assert (pthread_equal (::pthread_self (), this->owner_));

  if (this->nesting_level_ > 0)
    --this->nesting_level_;
  else {
    if (this->head_ == 0)
      this->in_use_ = 0; // No more waiters...
    else {
      this->owner_ = this->head_->thread_id_;
      --this->waiters_;

      // Wake up waiter and make it runable.
      this->head_->cv_.signal ();
      this->head_->runable_ = 1;

      this->head_ = this->head_->next_;

      if (this->head_ == 0)
	this->tail_ = 0;
    } 
  }
  return 0;
}


// Implicitly and automatically acquire the lock.
cpTokenGuard::cpTokenGuard (cpToken &m)
:lock_ (m) 
{ 
  this->result_ = this->lock_.acquire (); 
}

// Implicitly release the lock.
cpTokenGuard::~cpTokenGuard (void) 
{ 
  if (this->result_ != -1)
    this->lock_.release (); 
}

int
cpTokenGuard::locked (void)
{
  return (this->result_ != -1);
}

// Explicitly release the lock.
int 
cpTokenGuard::remove (void) 
{ 
  return this->lock_.remove (); 
}

// Explicitly acquire the lock.
int 
cpTokenGuard::acquire (void) 
{ 
  return this->lock_.acquire (); 
}

// Conditionally acquire the lock (i.e., won't block).
int 
cpTokenGuard::tryacquire (void) 
{ 
  return this->lock_.tryacquire (); 
}

// Explicitly release the lock.
int 
cpTokenGuard::release (void) 
{ 
  return this->lock_.release (); 
}
#endif
