//-----------------------------------------------------------------------------
// Copyright (c) 1991,1992 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
// Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
//-----------------------------------------------------------------------------
// 
// Description:
//	Simple Manager for Posix Threads or Tasks on vxworks
//	
//      For VxWorks, WIND_TCB->spare1 == manager
//                           ->spare2 == cleanup function pointer
//                           ->spare3 == cleanup function argument void*
//                           ->spare4 == magic number
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cpThreadManager.h,v $
//   Revision 1.2  2001/07/25 14:31:09  chen
//   64 BIT Initial Port
//
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:30:32  bickley
// Added cmlog to application development system.
//
//
#ifndef _CP_THREAD_MANAGER_H
#define _CP_THREAD_MANAGER_H

#include "cpSynch.h"
#include "cpThread.h"

class cpThreadManager;

class cpThreadDescriptor
{
  friend class cpThreadManager;
public:
  enum Thread_State
  {
    IDLE,
    SPAWNED,
    RUNNING,
    SUSPENDED,
    TERMINATED
  };

  // = Initialization method.
  cpThreadDescriptor (void);

  // = Accessor methods.
  cp_thread_t self (void);
  // Unique thread id.

  int grp_id (void);
  // Group ID.

  Thread_State state (void);
  // Current state of the thread.

private:

  cp_thread_t thr_id_;
  // Unique thread ID.
    
  int grp_id_;
  // Group ID.

  Thread_State thr_state_;
  // Current state of the thread.
};

class cpThreadControl;
class cpTMLocker;

class cpThreadManager
{
friend class cpThreadControl;
friend class cpTMLocker;
public:
  enum
  {
    DEFAULT_SIZE = 100
  };

  // = Initialization and termination methods.
  cpThreadManager (unsigned size = cpThreadManager::DEFAULT_SIZE);
  ~cpThreadManager (void);

  int open (unsigned size = DEFAULT_SIZE); 
  // Initialize the manager with room for SIZE threads.

  int close (void);		
  // Release all resources.

#ifndef __vxworks
  int spawn (CP_THREAD_FUNC func, 
	     void *args, 
	     long flags, 
	     cp_thread_t * = 0, 
	     unsigned int priority = 0,
	     void *stack = 0, 
	     unsigned stack_size = 0);
  // Create a new thread, which executes <func>.  

  // Returns: on success a unique group id that can be used to control
  // other threads added to the same group.  On failure, returns -1.

  int spawn_n (int n, 
	       CP_THREAD_FUNC func, 
	       void *args, 
	       long flags,
	       unsigned int priority = 0);
  // Create N new threads, all of which execute <func>.  
  // Returns: on success a unique group id that can be used to control
  // all of the threads in the same group.  On failure, returns -1.

  void *exit (void *status);	
  // Called to clean up when a thread exits.
#else
  int spawn (CP_THREAD_FUNC func, 
	     void *args, 
	     long flags = VX_FP_TASK, 
	     cp_thread_t * = 0, 
	     unsigned int priority = 100,
	     void *stack = 0, 
	     unsigned stack_size = 3000);
  // Create a new thread, which executes <func>.  

  // Returns: on success a unique group id that can be used to control
  // other threads added to the same group.  On failure, returns -1.

  int spawn_n (int n, 
	       CP_THREAD_FUNC func, 
	       void *args, 
	       long flags = VX_FP_TASK,
	       unsigned int priority = 100);
  // Create N new threads, all of which execute <func>.  
  // Returns: on success a unique group id that can be used to control
  // all of the threads in the same group.  On failure, returns -1.

  void* exit  (int code);
#endif

  int wait (const struct timespec *timeout = 0);	
  // Block until there are no more threads running or <timeout>
  // expires.  Returns 0 on success and -1 on failure.

  // = Accessors for <cpThreadDescriptors>.
  int descriptor (cp_thread_t, cpThreadDescriptor &);
  // Return the thread descriptor (indexed by thread_t).  Returns 0 on
  // success and -1 if not found.

#ifndef _ARCH_PPC
  // = Kill methods (send signals...).
  int kill_all (int signum);

  // Kill a single thread.
  int kill_grp (int grp_id, int signum);
  // Kill a group of threads.

  // cancel threads
  int cancel_all (void);
#endif
  // Send signum to all stopped threads 
  int kill (cp_thread_t, int signum);

  // cancel a singal thread
  int cancel (cp_thread_t thr);

  // = Set/get group ids for a particular thread id.
  int set_grp (cp_thread_t, int grp_id);
  int get_grp (cp_thread_t, int &grp_id);

  void dump (void) const;
  // Dump the state of an object.

private:
  int resize (unsigned);
  // Resize the pool of Thread_Descriptors.

#ifndef __vxworks
  int spawn_i (CP_THREAD_FUNC func, 
	       void *args, 
	       long flags, 
	       cp_thread_t * = 0, 
	       unsigned int priority = 0,
	       void *stack = 0, 
	       unsigned stack_size = 0);
  // Create a new thread (must be called with locks held).
#else
  int spawn_i (CP_THREAD_FUNC func, 
	       void *args, 
	       long flags = VX_FP_TASK, 
	       cp_thread_t * = 0, 
	       unsigned int priority = 100,
	       void *stack = 0, 
	       unsigned stack_size = 3000);
  // Create a new thread (must be called with locks held).

  static int taskDeleteCallback (WIND_TCB* ptr);
#endif

  int find (cp_thread_t t_id);
  // Locate the index of the table slot occupied by <t_id>.  Returns
  // -1 if <t_id> is not in the table doesn't contain <t_id>.

  int insert_thr (cp_thread_t t_id);
  // Insert a thread in the table (checks for duplicates).
  // Omitting the thread handle won't work on Win32...

  int append_thr (cp_thread_t t_id, cpThreadDescriptor::Thread_State);
  // Append a thread in the table (adds at the end, growing the table
  // if necessary).

  void remove_thr (int i);	
  // Remove thread from the table. 

  // = The following four methods implement a simple scheme for
  // operating on a collection of threads atomically.

  typedef int (cpThreadManager::*THR_FUNC)(int, int);

  int apply_grp (int grp_id, THR_FUNC, int = 0);
  // Apply <func> to all members of the table that match the <grp_id>.

  int apply_all (THR_FUNC, int = 0);
  // Apply <func> to all members of the table.

  int kill_thr (int i, int signum);
  // Send signal <signum> to the thread at index <i>.

  int cancel_thr (int i, int notused = 0);

  cpThreadDescriptor *thr_table_;
  // Vector that describes thread state within the Thread_Manager.

  unsigned max_table_size_;
  // Maximum number of threads we can manage (should be dynamically
  // allocated).

  unsigned current_count_;
  // Current number of threads we are managing.

  int grp_id_;
  // Keeps track of the next group id to assign.

  // = ACE_Mutex and condition variable for synchronizing termination.
  cpMutex          lock_;
  void lock        (void);
  void unlock      (void);
  cpConditionMutex zero_cond_;  // must be in this order
};

class cpTMLocker
{
public:
  // constructor and destructor
  cpTMLocker  (cpThreadManager* manager);
  ~cpTMLocker (void);

private:
  cpThreadManager* manager_;
  // deny access to copy and assignment
  cpTMLocker (const cpTMLocker& );
  cpTMLocker& operator = (const cpTMLocker& );
};


class cpThreadControl
{
public:
  cpThreadControl (cpThreadManager *tm, int insert = 0);
  // Initialize the thread control object.  If <insert> != 0, then
  // register the thread with the Thread_Manager.

  ~cpThreadControl (void);
  // Implicitly kill the thread on exit and remove it from its
  // associated ThreadManager.

#ifndef __vxworks
  void *exit (void *status);
  // Explicitly kill the thread on exit and remove it from its
  // associated ThreadManager.

  void *status (void *status);
  // Set the exit status (and return existing status).

  void *status (void);
  // Get the current exit status.
#else
  void* exit (int code);
#endif

  void dump (void) const;
  // Dump the state of an object.

private:
  cpThreadManager *tm_;
  // Pointer to the thread manager for this block of code.

#ifndef __vxworks
  void *status_;
  // Keeps track of the exit status for the thread.
#endif
};

#if defined (__INLINE__)
#include "cpThreadManager.i"
#endif /* __INLINE__ */

#endif

