//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      cmlog network server
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogServer.cc,v $
//   Revision 1.4  2001/07/25 14:30:47  chen
//   64 BIT Initial Port
//
//   Revision 1.3  2000/06/20 19:35:44  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.2  2000/02/07 16:14:27  chen
//   add protocol version info
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.2  1998/03/27  14:01:06  chen
// Not spawn a new thread for cmlogTcpMain
//
// Revision 1.1  1997/08/01  15:29:54  bickley
// Added cmlog to application development system.
//
//
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <assert.h>

#include <cmlogProtocol.h>
#include <cmlogPortHandler.h>
#include <cmlogClientAcceptor.h>
#include <cmlogBrowserAcceptor.h>
#include <cmlogAdminAcceptor.h>
#include <cmlogServerLocater.h>
#include <cmlogDataQueue.h>
#include <cmlogService.h>
#include <cmlogSignalHandler.h>
#include <cmlogServerAux.h>
#include <cmlogDatabase.h>
#include <cmlogDataCache.h>
#include <cmlogDbaseHandler.h>
#include <cmlogResNotification.h>
#include <cmlogUtil.h>
#include <cmlogDataManager.h>
#include <cmlogTagTblData.h>
#include <cmlogUptimeData.h>

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
#include <cpThreadManager.h>
cpThreadManager manager;
cmlogResNotification notifier;
#else
#include <cpProcessManager.h>

cpProcessManager manager;
char*            cmlog_shmemptr = 0;
#endif

sig_atomic_t    finished = 0;
cmlogDataManager* dataManager = 0;

static void  
initDataManager (cmlogDataManager* manager)
{
  // add all tags data
  cmlogData* data = 0;

  data = new cmlogTagTblData ((char *)("allTags"));
  
  manager->addData (data);

  // add live data stream data 
  data = new cmlogData ((char *)("loggingData"));
  manager->addData (data);

  // add uptime data
  data = new cmlogUptimeData ((char *)("uptime"));
  manager->addData (data);
}

static void* 
cmlogTcpMain (void *arg)
{
  // first to find out whether there is a server already running
  char serverHost[128];
  int  len = sizeof (serverHost);
  unsigned short serverPort;
  int  status;
  int  sockbuf;
  int  optlen = sizeof (sockbuf);

  cdevGlobalTagTable::tagTable ();

  cmlogServerLocater svcl (CMLOG_CLNTLKSVC, 10);
  
  if ((status = svcl.locatingServer (serverHost, len, serverPort)) == 0) {
    cmlogUtil::serverPrintf ("Error: cmlog server is running on %s at port %d\n",
			     serverHost, serverPort);
    ::exit (1);
  }
  else if (status == 1) {  // found a server but host name is too long
    cmlogUtil::serverPrintf ("Error: cmlog server is running somewhere\n");
    ::exit (1);
  }

  // create global database
  cmlogDatabase  database;
  // create global context database
  cmlogDatabase  cxtdbase;
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  // create global data queue
  cmlogDataQueue dataQueue;
#else
  // Order of the following code is important
  cmlogQueueHandler qhandler;
  cmlog_shmemptr = qhandler.shmemPtr ();
  qhandler.init ();   // create all semaphore and shared memory segments

  // data queue writer for database
  cmlogDataQueueWriter  dataQueue (cmlogDataQueue::TODBASE);
  
  // create data base process here 
  manager.spawn_n (1,
	  CP_PROCESS_FUNC (&(cmlogDataQueueToDbase::readerProcess)),
	  (void *)0);

  sleep (2);
  // queue for all commands
  cmlogDataQueueWriter  comdQueue (cmlogDataQueue::QUERY);

  // queues attach shared memory: do these last to let child process to attach
  // shared memory first
  dataQueue.attach (cmlog_shmemptr);
  unsigned int qsize = dataQueue.shmemSize ();
  char*        qhead = dataQueue.shmemHeader ();
  // attach command queue at the end of the data queue
  comdQueue.attach ((char *)(qhead + qsize));

  /*
  // parent close write fd of the first pipe
  close (cfds[1]);
  // parent close read fd of the second pipe
  close (cfds[2]);
  */
#endif

  // reactor object
  ACE_Reactor          reactor;
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cmlogClientAcceptor  clientAcceptor  (reactor, &dataQueue);
  cmlogBrowserAcceptor browserAcceptor (reactor, &dataQueue, 
					&database, &cxtdbase);
  cmlogAdminAcceptor   adminAcceptor   (reactor, &dataQueue);
#else
  cmlogClientAcceptor  clientAcceptor  (reactor, &dataQueue, &comdQueue);
  cmlogBrowserAcceptor browserAcceptor (reactor, &dataQueue, &comdQueue,
					&database, &cxtdbase);
  cmlogAdminAcceptor   adminAcceptor   (reactor, &dataQueue, &comdQueue);
#endif
  cmlogPortHandler     portHandler     (reactor, 1);

  // register client/browser acceptor to admin manager
  adminAcceptor.clientManager  (&clientAcceptor);
  adminAcceptor.browserManager (&browserAcceptor);


#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cmlogDbaseHandler    dbaseHandler    (reactor, database, cxtdbase,
					clientAcceptor);
#else
  cmlogDbaseHandler    dbaseHandler    (reactor, dataQueue, clientAcceptor);
#endif
  
  if (portHandler.open () == -1) {
    cmlogUtil::serverPrintf ("Cannot open port handler for cmlog server\n");
    ::exit (1);
  }
  if (clientAcceptor.open () == -1) {
    cmlogUtil::serverPrintf ("Cannot open log client acceptor for cmlog Server\n");
    ::exit (1);
  }
  if (browserAcceptor.open () == -1) {
    cmlogUtil::serverPrintf ("Cannot open browser acceptor for cmlog server\n");
    ::exit (1);
  }
  if (adminAcceptor.open () == -1) {
    cmlogUtil::serverPrintf ("Cannot open admin acceptor for cmlog server\n");
    ::exit (1);
  }

  cmlogUtil::serverPrintf ("cmlog server opens udp listening port %d\n",
			   portHandler.port_number ());
  cmlogUtil::serverPrintf ("cmlog server opens client connection port %d\n",
			   clientAcceptor.port_number ());
  cmlogUtil::serverPrintf ("cmlog server opens browser connection port %d\n",
			   browserAcceptor.port_number ());
  cmlogUtil::serverPrintf ("cmlog server opens admin connection port %d\n",
			   adminAcceptor.port_number ());

  // set browser socket send/recv buffer to CMLOG_SOCK_BUFSIZE
  sockbuf = CMLOG_SOCK_BUFSIZE;
  if (browserAcceptor.setSockOption (SOL_SOCKET, SO_SNDBUF, (void *)&sockbuf,
				     optlen) != 0) 
    cmlogUtil::serverPrintf ("Cannot set browser socket send buffsize to %d\n",
			     sockbuf);
  if (browserAcceptor.setSockOption (SOL_SOCKET, SO_RCVBUF, (void *)&sockbuf,
				     optlen) != 0) 
    cmlogUtil::serverPrintf ("Cannot set browser socket recv buffsize to %d\n",
			     sockbuf);
  

  portHandler.clientAcceptorPort  (clientAcceptor.port_number() );
  portHandler.browserAcceptorPort (browserAcceptor.port_number() );
  portHandler.adminAcceptorPort   (adminAcceptor.port_number() );
  
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)

#if defined (solaris)
  cp_thread_t clntAcceptorTh;
  cp_thread_t brwserAcceptorTh;
  cp_thread_t adminAcceptorTh;
  cp_thread_t portTh;

  // register pipe handler to the reactor
  if (reactor.register_handler (&notifier,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register signal notifier\n");
    ::exit (1);
    }

  // register dataqueue to the notifier
  notifier.dataqueue (&dataQueue); 

  // udp port handler thread
  manager.spawn (CP_THREAD_FUNC (&(cmlogPortHandler::portHandlerThread)),
		 (void *)&portHandler, 0,
		 &portTh);

  // client connection thread
  manager.spawn (CP_THREAD_FUNC (&(cmlogAcceptor::acceptorThread)),
		 (void *)&clientAcceptor, 0,
		 &clntAcceptorTh);

  // browser connection thread
  manager.spawn (CP_THREAD_FUNC (&(cmlogAcceptor::acceptorThread)),
		 (void *)&browserAcceptor, 0,
		 &brwserAcceptorTh);

  // browser connection thread
  manager.spawn (CP_THREAD_FUNC (&(cmlogAcceptor::acceptorThread)),
		 (void *)&adminAcceptor, 0,
		 &adminAcceptorTh);
#else
  if (reactor.register_handler (&portHandler,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register port handler\n");
    ::exit (1);
  }

  if (reactor.register_handler (&clientAcceptor,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register client acceptor\n");
    ::exit (1);
  }

  if (reactor.register_handler (&browserAcceptor,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register browser acceptor\n");
    ::exit (1);
  }

  if (reactor.register_handler (&adminAcceptor,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register admin acceptor\n");
    ::exit (1);
  }

  if (reactor.register_handler (&notifier,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register result notifier\n");
    ::exit (1);
  }
  
  // register dataqueue to the notifier
  notifier.dataqueue (&dataQueue);
#endif

  // create data cache object
  cmlogDataCache cache;

  // service threads
  cmlogService service[CMLOG_NUM_PROC_THREADS];
  for (int i = 0; i < CMLOG_NUM_PROC_THREADS; i++) {
    service[i].dataQueue    (&dataQueue);
    service[i].database     (&database);
    service[i].cxtdatabase  (&cxtdbase);
    service[i].dataCache    (&cache);
    cp_thread_t svcTh;

    manager.spawn (CP_THREAD_FUNC (&(cmlogService::serviceThread)),
		   (void *)&service[i], 0, &svcTh);
  }
#else
  if (reactor.register_handler (&portHandler,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register port handler\n");
    ::exit (1);
  }

  if (reactor.register_handler (&clientAcceptor,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register client acceptor\n");
    ::exit (1);
  }

  if (reactor.register_handler (&browserAcceptor,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register browser acceptor\n");
    ::exit (1);
  }

  if (reactor.register_handler (&adminAcceptor,
				ACE_Event_Handler::READ_MASK) == -1) {
    cmlogUtil::serverPrintf ("Cannot register admin acceptor\n");
    ::exit (1);
  }
#endif

#if !defined (CMLOG_USE_THREAD) || !defined (_REENTRANT) 
  // register a signal handler for signals
  if (cmlogSignalHandler::registerSignalHandlers () != 0) 
    cmlogUtil::serverPrintf ("Cannot register all signal handlers\n");
#endif

  // create data manager
  cmlogDataManager dm;
  dataManager = &dm;
  // create some initial data
  initDataManager (dataManager);
  // update all data from the persitent storage area
  dataManager->update ();

  cmlogUtil::serverPrintf ("Starting up cmlog server with %s........\n", 
			   CMLOG_PROTOCOL_VERSION_STRING);
  while (!finished) {
    reactor.handle_events ();
  }

  cmlogUtil::serverPrintf ("Shutting down cmlog server\n");
  return 0;
}

main (int argc, char** argv)
{
  // change number of sockets the server can open
  struct rlimit limit, nlimit;
  
  // read in configuration file
  if (argc == 1)
    cmlogUtil::parseConfigFile (0);
  else
    cmlogUtil::parseConfigFile (argv[1]);

  if (getrlimit (RLIMIT_NOFILE, &limit) != 0) 
    cmlogUtil::serverPrintf ("Cannot get resouce limit\n");
  else {
    cmlogUtil::serverPrintf ("Current limit on number of opened files is %d\n", 
			     limit.rlim_cur);
    
    if (nlimit.rlim_cur <  CMLOG_RLIMIT_NOFILE) {
      nlimit.rlim_cur = CMLOG_RLIMIT_NOFILE;
      nlimit.rlim_max = limit.rlim_max;
      if (setrlimit (RLIMIT_NOFILE, &nlimit) != 0) 
	cmlogUtil::serverPrintf ("Cannot change the limit on number of opened files\n");
      else {
	if (getrlimit (RLIMIT_NOFILE, &limit) == 0) 
	  cmlogUtil::serverPrintf ("Limit on number of opened files has been changed to %d\n", 
				   limit.rlim_cur);
	else
	  cmlogUtil::serverPrintf ("Cannot change the limit on number of opened files\n");
      }
    }
  }
  
  // setup server properly
  cmlogServerAux::auxiliaryHandlers ();

  // setup all tags
  cmlogUtil::setTags ();

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  // create a signal handler thread and main thread communication
  // pipe
  int *cfds = (int *)notifier;

  // register a signal handler for signals
  if (cmlogSignalHandler::registerSignalHandlers () != 0) 
    cmlogUtil::serverPrintf ("Cannot register all signal handlers\n");

  cp_thread_t signalTh;

  // create a singal thread to handle signals
  cpThread::spawn (CP_THREAD_FUNC (&(cmlogSignalHandler::signalFunc)),
		   (void *)&cfds[1], 0, &signalTh);
  
  cmlogTcpMain (0);
#else
  cmlogTcpMain (0);
#endif
  return 0;
}

  
