//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of CMLOG Server Query Services
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: cmlogQuerySvc.cc,v $
//   Revision 1.6  2001/07/25 14:30:47  chen
//   64 BIT Initial Port
//
//   Revision 1.5  2000/06/20 19:35:44  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.4  2000/03/02 15:01:57  chen
//   remove fprintf
//
//   Revision 1.3  2000/03/01 17:09:30  chen
//   Add concurrency control and output signa info to a file
//
//   Revision 1.2  2000/02/07 16:13:57  chen
//   do not send an empty packet
//
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
//
//
#include <cmlogUtil.h>
#include <cdevData.h>
#include <cmlog_cdevTagMap.h>
#include <cmlogMsg.h>
#include <cmlogBrowserIO.h>
#include <cmlogDatabase.h>
#include <cmlogDataCache.h>
#include <cmlogCxtDataCache.h>
#include <cmlogLogicSup.h>
#include "cmlogQuerySvc.h"

#if !defined (CMLOG_USE_THREAD) || !defined (_REENTRANT)
int cmlogQuerySvc::aborted_ = 0;

// all signals to catch
int cmlogQuerySvc::signals[] = 
{SIGINT, SIGQUIT, SIGILL, SIGFPE,
 SIGSYS, SIGPIPE, SIGBUS, SIGSEGV,
 SIGTERM, SIGIO, SIGPOLL};

int cmlogQuerySvc::numSignals = 11;
#endif


#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogQuerySvc::cmlogQuerySvc (cmlogBrowserIO* io,
			      cmlogMsg* msg)
  :io_ (io), msg_ (msg), aborted_ (0), qid_ (0), cbkid_ (0)
#else
cmlogQuerySvc::cmlogQuerySvc (cmlogBrowserIO* io,
			      cmlogMsg* msg)
  :io_ (io), msg_ (msg), qid_ (0), cbkid_ (0)
#endif
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogQuerySvc Class Object\n");
#endif
  cmlog_cdevMessage& data = (*msg_);
  qid_ =  data.getTransIndex ();
  cbkid_ = data.getForeignDataIndex ();
}

cmlogQuerySvc::~cmlogQuerySvc (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogQuerySvc Class Object\n");
#endif

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  // remove this service object from manager
  io_->queryManager().removeSvc (this);
  // this message is created inside cmlogBrowserIO
  delete msg_;
#endif
}

void *
cmlogQuerySvc::querySvcThread (void *arg)
{
  cmlogQuerySvc *obj = (cmlogQuerySvc *)arg;

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)  
  // increase this io channel reference count
  obj->io_->ref ();

  // register this query service to query manager
  obj->io_->queryManager().registerSvc (obj);

  cmlog_cdevMessage& data = (*(obj->msg_));  

  // find data
  obj->findData (&data, obj->io_->dbase(), obj->io_->cxtdbase());

  // delete this object which was created dynamicly
  delete obj;
#else
  cmlogQuerySvc::registerSignalHandlers ();
  cmlog_cdevMessage& data = (*(obj->msg_));  
  // find data
  obj->findData (&data, 0, 0);

#ifdef _CMLOG_DEBUG
  printf ("Exiting from a query process %d\n", getpid());
#endif

#endif

  return 0;
}

int
cmlogQuerySvc::aborted (void)
{
  int flag = 0;
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  if (!io_->connected ()) {
    io_->cleanup ();
    flag = 1;
  }
  else if (aborted_)
    flag = 2;
#else
  if (cmlogQuerySvc::aborted_)
    flag = 2;
#endif

  return flag;
}

// Handle query stop: 1 == browser close connection, 2 == cancel query
#define HANDLE_CMLOG_QUERY_ABORT0(returncode,io,message) { \
         int abtst = 0; \
         if((abtst = aborted()) == 2) { \
              io->sendErrorBack(returncode,*message); \
              return; \
         } \
         else if (abtst == 1) \
              return; \
    }

#define HANDLE_CMLOG_QUERY_ABORT1(code,io,msg,dbase,fnames,cxtfnames,nfs,useopeneddbase) { \
         int abtst = aborted (); \
         if (abtst > 0) { \
             if (!useopeneddbase) \
	         dbase->close (); \
             cmlogUtil::freeFileNames (fnames, cxtfnames, nfs); \
             if (abtst == 2) \
                 io->sendErrorBack(code,*msg); \
             return; \
         } \
    }

#define HANDLE_CMLOG_QUERY_ABORT2(code,io,msg,fnames,cxtfnames,nfs) { \
         int abtst = aborted (); \
         if (abtst > 0) { \
             cmlogUtil::freeFileNames (fnames, cxtfnames, nfs); \
             if (abtst == 2) \
                 io->sendErrorBack(code,*msg); \
             return; \
         } \
    }

void
cmlogQuerySvc::findData (cmlog_cdevMessage* message,
			 cmlogDatabase* dbase, cmlogDatabase* cxtdbase)
{
  double       start, end;
  double       key;
  double       cxtid;
  int          numitems = 0;
  int          count = 0;
  char         qstr[1024];          /* this is the query string */
  int          hasqstr = 0;         /* do we have query string  */
  int          logic_res = 0;
  int          j;
  cmlogPacket  packet;
  cdevData*    tdata = 0;
  cmlogDatabase* dbaseptr = 0;
  cmlogDatabase* cxtdbaseptr = 0;
  int            useopeneddbase = 0;/* use exising dbase ptr    */

  // data must be not null since we already passed from initial
  // check up from cmlogBrowserIO class
  cdevData *idata = message->getData ();
  idata->get (cmlogUtil::CMLOG_START_TAG, &start);
  idata->get (cmlogUtil::CMLOG_END_TAG, &end);

  if (idata->get (cmlogUtil::CMLOG_NUMITEMS_TAG, &numitems) != CDEV_SUCCESS)
    numitems = 0;

  if (idata->get (cmlogUtil::CMLOG_QUERYMSG_TAG, qstr, sizeof (qstr)) 
      == CDEV_SUCCESS)
    hasqstr = 1;

  // check whether this query has been aborted
  HANDLE_CMLOG_QUERY_ABORT0(CMLOG_QUERY_CANCELED, io_, message);
  
  // find file names from these time stamps
  char** filenames;
  int    numfiles = 0;
  char** cxtfnames;
  int    numcxtfs = 0;

  // get all database filenames
  if (cmlogUtil::dbaseFilenames (start, end, filenames, numfiles) != 0) {

    HANDLE_CMLOG_QUERY_ABORT0(CMLOG_QUERY_CANCELED, io_, message);    

    io_->sendErrorBack (CMLOG_NOTFOUND, *message);
    return;
  }

  // get all context filenames
  if (cmlogUtil::dbaseCxtFilenames (start, end, cxtfnames, numcxtfs) != 0) {
    HANDLE_CMLOG_QUERY_ABORT0(CMLOG_QUERY_CANCELED, io_, message);    

    io_->sendErrorBack (CMLOG_NOTFOUND, *message);
    return;
  }
  
  // number of files and real database and number of files of cxt
  // must be the same
  if (numfiles != numcxtfs) {
    HANDLE_CMLOG_QUERY_ABORT0(CMLOG_QUERY_CANCELED, io_, message);
    io_->sendErrorBack (CMLOG_NOTFOUND, *message);
    return;
  }

  cdevData data, rdata;
  key = start;
      
  for (int i = 0; i < numfiles; i++) {
    // clean data object
    data.remove ();
    // reset database ptr
    dbaseptr = 0;
    cxtdbaseptr = 0;
    useopeneddbase = 0;

    // the following is only being called for multi-threaded version
    // for multi-process version, dbase and cxtdbase are null
    if (dbase && cxtdbase && strcmp (filenames[i], dbase->database ()) == 0) {
      dbaseptr = dbase;
      cxtdbaseptr = cxtdbase;
      useopeneddbase = 1;
    }
      
    cmlogDatabase tdbase;
    if (!useopeneddbase) {
      if (tdbase.open (filenames[i], O_RDONLY) != CDEV_SUCCESS)
	continue;
      else 
	dbaseptr = &tdbase;
    }
      
    if (dbaseptr->cursorInit (data, &key) != CDEV_SUCCESS) {
      if (!useopeneddbase) 
	dbaseptr->close ();
      else
	cmlogUtil::serverPrintf ("Opened database <%s> has problem\n", 
				 dbaseptr->database ());
      continue;
    }
    // construct client context data cache
    cmlogCxtDataCache cxtcache (cxtfnames[i]);

    tdata = 0;
    if (data.get (cmlogUtil::CMLOG_CLNTCXT_TAG, &cxtid) == CDEV_SUCCESS) {
      tdata = cxtcache.getCxt (cxtid);

      // convert any data found to remote host representation
      if (tdata)
	io_->convertDataLToR (*tdata);
    }
    io_->convertDataLToR (data);

    if (tdata) { // appending data will override the appended data
      // client context takes lower priority the user data
      rdata = *tdata;
      rdata += data;
    }
    else
      rdata = data;

    // check channel connection flag
    HANDLE_CMLOG_QUERY_ABORT1(CMLOG_QUERY_CANCELED,io_,message,dbaseptr,filenames,cxtfnames,numfiles, useopeneddbase);

    /* check logic expression */
    if (hasqstr) {
      if ((logic_res = dataMatchLogic (rdata, qstr)) == -1) {
	io_->sendErrorBack (CMLOG_QUERYMSG_ERR, *message);

	/* free memory of filenames */
	cmlogUtil::freeFileNames (filenames, cxtfnames, numfiles);

	return;
      }
      else if (logic_res == 1) 
	constructPacket (&packet, message, &rdata, CMLOG_INCOMPLETE);
    }
    else 
      constructPacket (&packet, message, &rdata, CMLOG_INCOMPLETE);

    // check whether we reach number of items we want
    count++;
    if (numitems > 0 && count >= numitems) {
      if (!useopeneddbase)
	dbaseptr->close ();
      flushPacket (&packet, message, key, CMLOG_PAUSED);

      /* free memory of filenames */
      cmlogUtil::freeFileNames (filenames, cxtfnames, numfiles);

      return;
    }

    // go into a loop
    data.remove ();
    while ((dbaseptr->cursorNext(data, &key) == CDEV_SUCCESS) &&
	   key <= end) {

      // reset pointer
      tdata = 0;

      if (data.get (cmlogUtil::CMLOG_CLNTCXT_TAG, &cxtid) == CDEV_SUCCESS) {
	tdata = cxtcache.getCxt (cxtid);

	// convert any data found to remote host representation
	if (tdata)
	  io_->convertDataLToR (*tdata);
      }
      // convert any data found to remote host representation
      io_->convertDataLToR (data);

      if (tdata) {
	rdata = *tdata;
	rdata += data;
      }
      else
	rdata = data;
      
      // check channel connection flag
      HANDLE_CMLOG_QUERY_ABORT1(CMLOG_QUERY_CANCELED,io_,message,dbaseptr,filenames,cxtfnames,numfiles, useopeneddbase);

      /* check logic expression */
      if (hasqstr) {
	if ((logic_res = dataMatchLogic (rdata, qstr)) == -1) {
	  io_->sendErrorBack (CMLOG_QUERYMSG_ERR, *message);

	  /* free memory of filenames */
	  cmlogUtil::freeFileNames (filenames, cxtfnames, numfiles);

	  return;
	}
	else if (logic_res == 1) 
	  constructPacket (&packet, message, &rdata, CMLOG_INCOMPLETE);
      }
      else 
	constructPacket (&packet, message, &rdata, CMLOG_INCOMPLETE);

      // check whether we reach number of items we want
      count++;
      if (numitems > 0 && count >= numitems) {
	if (!useopeneddbase)
	  dbaseptr->close ();
	flushPacket (&packet, message, key, CMLOG_PAUSED);

	/* free memory of filenames */
	cmlogUtil::freeFileNames (filenames, cxtfnames, numfiles);

	return;
      }

      data.remove ();
    }
    if (!useopeneddbase)
      dbaseptr->close ();
  }  // end of for loop

  // check channel connection flag
  HANDLE_CMLOG_QUERY_ABORT2(CMLOG_QUERY_CANCELED,io_,message,filenames,cxtfnames,numfiles);

  if (packet.emptyPacket ()) 
    io_->sendErrorBack (CMLOG_NOTFOUND, *message);      
  else 
    flushPacket (&packet, message, key, CMLOG_SUCCESS);

  // derefence the channel
  io_->deref ();
  

  /* free memory of filenames */
  cmlogUtil::freeFileNames (filenames, cxtfnames, numfiles);
}

void
cmlogQuerySvc::constructPacket (cmlogPacket* packet,
				cmlog_cdevMessage* message,
				cdevData* data, int errcode)
{
  char*    msg   = message->getMessage ();

  // add status code
  data->insert (cmlogUtil::CMLOG_RESULT_TAG, errcode);

  // construct returning message
  cmlog_cdevMessage ret (0, qid_, 0, 0, cbkid_, 0, 0, 0, msg, data, 0, 0);
  cmlogMsg    retmsg (ret);

  if (packet->overflow (retmsg) && !packet->emptyPacket()) {
    // do not send an empty packet
    io_->sendResult (*packet);
    packet->empty ();
  }
  packet->insert (retmsg);
}

void
cmlogQuerySvc::flushPacket (cmlogPacket* packet, 
			    cmlog_cdevMessage* message,
			    double timestamp, int errcode)
{
  // flush out old packet
  if (!packet->emptyPacket ()) {
    io_->sendResult (*packet);
    packet->empty ();
  }

  // construct the last packet
  char*    msg   = message->getMessage ();

  // add status code
  cdevData data;
  data.insert (cmlogUtil::CMLOG_RESULT_TAG, errcode);
  data.insert (cmlogUtil::CMLOG_KEY_TAG, timestamp);

  // construct returning message
  cmlog_cdevMessage ret (0, qid_, 0, 0, cbkid_, 0, 0, 0, msg, &data, 0, 0);
  cmlogMsg    retmsg (ret);

  packet->insert (retmsg);
  io_->sendResult (*packet);
  packet->empty ();
}


#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
void
cmlogQuerySvc::abort (void)
{
  aborted_ = 1;
}
#else
void
cmlogQuerySvc::abort (void)
{
  kill (proc_, SIGQUIT);
}

void
cmlogQuerySvc::queryProcessId (pid_t id)
{
  proc_ = id;
}

pid_t
cmlogQuerySvc::queryProcessId (void) const
{
  return proc_;
}

int
cmlogQuerySvc::registerSignalHandlers (void)
{
  struct sigaction act, oact;

  act.sa_handler = &(cmlogQuerySvc::signalFunc);
  sigemptyset (&act.sa_mask);
  act.sa_flags = 0;
#ifdef SA_RESTART
  act.sa_flags |= SA_RESTART;
#endif

  for (int i = 0; i < cmlogQuerySvc::numSignals; i++) 
    if (sigaction (cmlogQuerySvc::signals[i], &act, &oact) < 0)
      return -1;
  
  return 0;
}

void
cmlogQuerySvc::signalFunc (int signo)
{
#ifdef solaris
  char signame[SIG2STR_MAX];
  if (sig2str (signo, signame) == -1)
    sprintf (signame, "unknown");

  cmlogUtil::serverPrintf ("Proc %d: Interrupted by %s signal\n", 
			   getpid(), signame);
#else
  cmlogUtil::serverPrintf ("Proc %d:Interrupted by %d signal\n", getpid(), signo);
#endif
  if (signo == SIGQUIT || signo == SIGPIPE)
    cmlogQuerySvc::aborted_ = 1;
  else
    ::exit (1);
}

#endif
