//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogUxToServer.cc,v $
//   Revision 1.2  1999/11/12 17:33:02  chen
//   beta 2.0
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.4  1998/10/07  17:03:24  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.3  1998/01/21  16:04:44  chen
// vxworks client daemon cannot be shutdown
//
// Revision 1.2  1997/10/01  18:11:27  chen
// add verbose flag in case server is dead but client daemon still print
//
// Revision 1.1  1997/08/01  15:29:29  bickley
// Added cmlog to application development system.
//
//
#include <cmlogUtil.h>
#include <cmlogMsg.h>
#include <cmlogServerLocater.h>
#include <cmlogServerHB.h>
#include <cmlogNetUtil.h>
#include "cmlogUxToServer.h"

cmlogUxToServer::cmlogUxToServer (ACE_Reactor& reactor, cmlogPacket& buffer)
:toServer_ (), connected_ (0), flushInterval_ (1), timerOn_ (0),
 shb_ (0), reactor_ (reactor), buffer_ (buffer)
{
#ifdef _TRACE_OBJECTS
  printf ("     Create cmlogUxToServer Class Object\n");
#endif
  serverHost_[0] = '\0';
  svr_udp_port_ = cmlogNetUtil::findServerUdpPort ();
}

cmlogUxToServer::~cmlogUxToServer (void)
{
#ifdef _TRACE_OBJECTS
  printf ("     Delete cmlogUxToServer Class Object\n");
#endif
  if (connected_)
    toServer_.close ();
  connected_ = 0;
}

int
cmlogUxToServer::connToServer (int reconnect)
{
  char hostname[128];
  int  status = 0;
  unsigned short serverPort = 0;
  
  // find the cmlogServer
  connected_ = 0;

  cmlogServerLocater svcl (CMLOG_CLNTLKSVC, svr_udp_port_, 10);
  if ((status = svcl.locatingServer (hostname, sizeof (hostname), serverPort))
      == 0) {
    printf ("Find cmlog server running at port %d on %s\n",
	    serverPort, hostname);
  }
  else {
    if (!reconnect)
      printf ("Cannot find any cmlog server running\n");
    return -1;
  }

  ACE_SOCK_Connector conn;
  ACE_INET_Addr      addr (serverPort, hostname);

  status = conn.connect (toServer_, addr);
  if (status == -1) {
#ifdef _CMLOG_DEBUG
    printf ("Cannot connect to server at %d : %s\n", serverPort, hostname);
#endif
    connected_ = 0;
  }
  else {
    ::strncpy (serverHost_, addr.get_host_name(), sizeof (serverHost_));
    connected_ = 1;
  }
  return status;
}

void
cmlogUxToServer::serverUdpPort (unsigned short port)
{
  svr_udp_port_ = port;
}

int
cmlogUxToServer::connectedToServer (void) const
{
  return connected_;
}

int
cmlogUxToServer::get_handle (void) const
{
  return toServer_.get_handle ();
}

int
cmlogUxToServer::handle_close (int, ACE_Reactor_Mask)
{
  return toServer_.close ();
}

int
cmlogUxToServer::closeDown (void)
{
  reactor_.remove_handler (this, ACE_Event_Handler::READ_MASK);
  handle_close (-1, ACE_Event_Handler::READ_MASK);
  connected_ = 0;
  return 0;
}

int
cmlogUxToServer::reConnToServer (void)
{
  int status = connToServer (1);
  if (status != -1) {
    status = reactor_.register_handler (this, ACE_Event_Handler::READ_MASK);
    if (status == -1)
      return -1;
  }
  else
    return -1;
  return 0;
}

void
cmlogUxToServer::serverHeartBeatMonitor (cmlogServerHB* hb)
{
  shb_ = hb;
}

int
cmlogUxToServer::handle_input (int)
{
  int n, status = 0;
  cmlogPacket packet;

  n = toServer_ >> packet;
#ifdef _CMLOG_DEBUG
  printf ("Unix Client Received %d bytes from server\n", n);
#endif
  
  if (n <= 0) {
    printf ("cmlog server is gone\n");

#ifdef _CMLOG_CLIENTD_AUTO_QUIT
    finished = 1;
#endif

    connected_ = 0;

    // tell server monitor to start trying to reconnect to the server 
    // try to connect to the server every 40 seconds
    shb_->registerMonTimer (40, 40);
    return -1;
  }

  long        opcode;
  int         i;

  // remember to free all msgs ansg msgs[] somewhere
  cmlogMsg** msgs = packet.messages ();

  for (i = 0; i < packet.numberOfData (); i++) {
    cmlog_cdevMessage& idata = (*msgs[i]);
    opcode = msgs[i]->type ();

    if (opcode == CMLOG_EXIT) { // quit client daemon
#ifndef __vxworks
      finished = 1;
      return -1;
#endif
    }
  }

  for (i = 0; i < packet.numberOfData (); i++)
    delete msgs[i];
  delete []msgs;

  return 0;
}

int
cmlogUxToServer::handle_timeout (const ACE_Time_Value& value,
				 const void *arg)
{
  int num = toServer_ << buffer_;
  buffer_.empty ();  
  timerOn_ = 0;
  return 0;
}

void
cmlogUxToServer::setTimerToFlush (void)
{
  ACE_Time_Value dt (flushInterval_, 0);

  reactor_.schedule_timer (this, 0, dt);
}

int
cmlogUxToServer::insertMessageToBuffer (cmlog_cdevPacket1& packet, int bufsize)
{
  int num = 0;
  unsigned opcode = 0;

  if (connected_) {
    packet.getOperationCode (opcode);
    switch (opcode) {
    case CMLOG_CONN_INFO:
      {
	// check all clients
	shb_->checkAllClients ();

	short clientid;
	int   procid;
	cdevData cxt;
	cdevData tagmap;

	packet.getData (cxt);
	packet.getTagMap (tagmap);
	packet.getClientID (clientid);
	if (cxt.get (cmlogUtil::CMLOG_PID_TAG, &procid) == CDEV_SUCCESS) {
	  // register new client
	  shb_->registerClient (procid, clientid, cxt, tagmap);
	}
      }
      if (buffer_.overflow (packet, bufsize)) {
	num = toServer_ << buffer_;
	buffer_.empty ();
      }
      buffer_.insert (packet, bufsize);
      num = toServer_ << buffer_;
      buffer_.empty ();
      break;
    case CMLOG_CLOSE_CONN:
      {
	short clientid;
	
	packet.getClientID (clientid);

	shb_->removeClient (clientid);
      }
      if (buffer_.overflow (packet, bufsize)) {
	num = toServer_ << buffer_;
	buffer_.empty ();
      }
      buffer_.insert (packet, bufsize);
      num = toServer_ << buffer_;
      buffer_.empty ();
      break;
    case CMLOG_RECONN_INFO:
      if (buffer_.overflow (packet, bufsize)) {
	num = toServer_ << buffer_;
	buffer_.empty ();
      }
      buffer_.insert (packet, bufsize);
      num = toServer_ << buffer_;
      buffer_.empty ();
      break;
    case CMLOG_TAGMAP:
      if (buffer_.overflow (packet, bufsize)) {
	num = toServer_ << buffer_;
	buffer_.empty ();
      }
      buffer_.insert (packet, bufsize);
      num = toServer_ << buffer_;
      buffer_.empty ();
      break;
    case CMLOG_FLUSH:
      num = toServer_ << buffer_;
      buffer_.empty ();
      break;
#ifdef __vxworks
    case CMLOG_ISR_DATA:
      {
	cmlog_cdevPacket1 newpacket;
	// new size is size of new packet without leading header
	size_t      newsize = 0;

	if (cmlogUxToServer::convertStrToPacket (packet, newpacket, newsize)
	    == 0) {
	  if (buffer_.overflow (newpacket, newsize + sizeof (newsize) )) {
	    num = toServer_ << buffer_;
	    buffer_.empty ();
	  }
	  buffer_.insert (newpacket, newsize + sizeof (newsize) );
	  if (!timerOn_) {
	    setTimerToFlush ();
	    timerOn_ = 1;
	  }
	}
	// this packet use static buffer, so detach itself
	newpacket.detachData ();
      }
      break;
#endif
    default:
      if (buffer_.overflow (packet, bufsize)) {
	num = toServer_ << buffer_;
	buffer_.empty ();
      }
      buffer_.insert (packet, bufsize);
      if (!timerOn_) {
	setTimerToFlush ();
	timerOn_ = 1;
      }
      break;
    }
  }
  else {
    packet.getOperationCode (opcode);
    if (opcode ==  CMLOG_CHANGE_SRP) {
      // client daemon has been notified to change port
      cdevData portdata;
      unsigned long port;
      packet.getData (portdata);
      if (portdata.get (cmlogUtil::CMLOG_VALUE_TAG, &port) == CDEV_SUCCESS &&
	  port != svr_udp_port_) {
#ifdef _CMLOG_DEBUG
	printf ("Change server udp port to %d\n", port);
#endif
	svr_udp_port_ = port;  // reconnection mechanism uses changed port
      }
    }
    else if (opcode == CMLOG_CONN_INFO) {
      // if not connect to the server, try to reconnect to cmlogServer
      if (reConnToServer () == 0) {
#ifdef _CMLOG_DEBUG
	printf ("Reconnect to cmlogServer\n");
#endif
	if (buffer_.overflow (packet, bufsize)) {
	  num = toServer_ << buffer_;
	  buffer_.empty ();
	}
	buffer_.insert (packet, bufsize);
	num = toServer_ << buffer_;
	buffer_.empty ();
      }
      else {
	// empty
      }
    }
    else {
#if defined (__vxworks) && defined (_CMLOG_VERBOSE)
      // print out the text tag
      cdevData tout;
      packet.getData (tout);

      char output[256];
      
      if (tout.get (cmlogUtil::CMLOG_TEXT_TAG, output, sizeof (output))
	  == CDEV_SUCCESS) 
	printf ("%s\n", output);
#endif
    }
  }
  return 0;
}

#ifdef __vxworks
int
cmlogUxToServer::convertStrToPacket (cmlog_cdevPacket1& sp, 
				     cmlog_cdevPacket1& dp,
				     size_t& size)
{
  char* format = 0;

  // in vxworks buffer size is limited by pipe device
  // this static buffer is accessed by daemon only
  static char buffer[CMLOG_CLNT_PIPE_MAX_BYTES];

  if (sp.getMessage (format) != CDEV_SUCCESS) 
    return -1;

  char*  bstr = 0;
  size_t blen = 0;
  int    arg[10];  // 10 additional aruguments
  int    i = 0;

  if (sp.streamOut (&bstr, &blen) != CDEV_SUCCESS)
    return -1;
  
  // the last part of the stream are 10 integers
  i = blen - sizeof (int)*10;
  bcopy (&(bstr[i]), (char *)&(arg[0]), sizeof(int)*10);

  // new cdevData holds all information
  cdevData isrdata;
  processString (format, arg, 10, isrdata);

  short cid = 0;
  sp.getClientID (cid);

  if (dp.set (buffer, sizeof (buffer),
	      cid, 0, 0, 0, 0, CMLOG_ADD_DATA, 0, 0, 0, &isrdata, 0, 0) != 0)
    return -1;

  if (dp.streamOut (&bstr, &blen) == CDEV_SUCCESS) 
    size = blen;
  else
    return -1;

  return 0;
}

int
cmlogUxToServer::processString (char* format, int* arg, int size,
				cdevData& data)
{
  char *p;
  char strtmp[128], tagName[64];
  int ival, tag, i = 0, j = 0;
  double dval;
  char cval;

  for (p = format; *p; p++) {
    if (*p == ' ' || *p == '\t' || *p == '\r' || *p == '\n')
      continue;
    if (*p == '%') {
      switch(*++p) {
      case 'd': 
      case 'i':
      case 'x':
      case 'X':
      case 'o':
      case 'u':
	if (j >= size)
	  return 0;

	ival = arg[j++];
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, ival);
	break;
      case 'f':
      case 'g':
      case 'F':
      case 'G':
	if (j >= size)
	  return 0;

	dval = (double)(arg[j++]);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, dval);
	break;
      case 's':
	if (j >= size)
	  return 0;

	::strcpy(strtmp, (char *)(arg[j++]));
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, strtmp);
	break; 
      case 'c':
	if (j >= size)
	  return 0;

	cval = (char)(arg[j++]);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, cval);
	break;
      default:
	printf ("The argument type % %c not supported  \n", *p);
	break;
      }
      i = 0;
    }   //if (*p == '%')
    else if ((tagName[i++] = *p) == '=') {
      i--;
      tagName[i] = '\0'; 
    }
  }
  return 0;
}
#endif  

    


  




