//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      cmlog Server to client IO handler
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogClientIO.cc,v $
//   Revision 1.6  2001/07/25 14:30:18  chen
//   64 BIT Initial Port
//
//   Revision 1.5  2000/06/20 19:35:16  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.4  2000/01/04 14:20:51  chen
//   remove context memory when application quits
//
//   Revision 1.3  1999/12/13 21:24:39  chen
//   Add more error checking using BAD_PTR
//
//   Revision 1.2  1999/12/13 15:17:34  chen
//   Add robust check on all incoming messages
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.4  1998/10/07  17:03:23  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.3  1998/03/27  13:57:21  chen
// add connection information to database
//
// Revision 1.2  1998/02/12  19:33:55  chen
// remove unnecessay empty tagmap
//
// Revision 1.1  1997/08/01  15:28:55  bickley
// Added cmlog to application development system.
//
//
#include <sys/ioctl.h>
#include <sys/time.h>
#include <cmlogClntCxt.h>
#include <cmlogUtil.h>
#include <cmlogDataManager.h>
#include <cmlogData.h>
#include "cmlogClientIO.h"

//=========================================================================
//       Implementation of cmlogTagMap Class
//=========================================================================
cmlogTagMap::cmlogTagMap (short id)
:id_ (id), map_ ()
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogTagMap Class Object\n");
#endif
  // empty
}

cmlogTagMap::~cmlogTagMap (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogTagMap Class Object\n");
#endif
}


int
cmlogTagMap::sameClient (short id)
{
  return (id_ == id);
}

cmlog_cdevTagMap&
cmlogTagMap::tagmap (void)
{
  return map_;
}

cmlogTagMap::operator cmlog_cdevTagMap& (void)
{
  return map_;
}


//==========================================================================
//         Implementation of cmlogClientIO class
//==========================================================================
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogClientIO::cmlogClientIO (ACE_Reactor& r, 
			      cmlogDataQueue* q)
:cmlogIO (r, q), cxtList_ (), tagmapList_ (), prevConnTime_ (0.0)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogClientIO Class Object\n");
#endif
}
#else
cmlogClientIO::cmlogClientIO (ACE_Reactor& r, 
			      cmlogDataQueue* q,
			      cmlogDataQueue* cq)
:cmlogIO (r, q, cq), cxtList_ (), tagmapList_ (), prevConnTime_ (0.0)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogClientIO Class Object\n");
#endif
}
#endif

cmlogClientIO::~cmlogClientIO (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogClientIO Class Object\n");
#endif

  // delete all context
  cmlogSlistIterator ite (cxtList_);
  cmlogClntCxt* cxt = 0;

  for (ite.init(); !ite; ++ite) {
    cxt = (cmlogClntCxt *) ite ();
    delete cxt;
  }

  // delete all tag maps
  cmlogSlistIterator iite (tagmapList_);
  cmlogTagMap* map = 0;
  
  for (iite.init (); !iite; ++iite) {
    map = (cmlogTagMap *) iite ();
    delete map;
  }
}

void
cmlogClientIO::setTagMap (cmlog_cdevMessage& msg)
{
  cdevData* data = msg.getTagMap ();
  if (data != 0) {
    int   found = 0;
    short clntid = msg.getClientID ();
    
    cmlogSlistIterator ite (tagmapList_);
    cmlogTagMap* map = 0;

    // find a tag map with the same client id
    for (ite.init (); !ite; ++ite) {
      map = (cmlogTagMap *)ite ();
      if (map->sameClient (clntid)) {
	found = 1;
	break;
      }
    }
    if (!found) {
      map = new cmlogTagMap (clntid);
      tagmapList_.add ((void *)map);
    }
    cmlog_cdevTagMap& tagmap = map->tagmap ();
    tagmap.updateTagMap (*data);

    if (tagmap.numberOfMismatches () == 0) {
      tagmapList_.remove ((void *)map);
      delete map;
    }
    else {
#ifdef _CMLOG_DEBUG
      tagmap.asciiDump ();
#endif
      // update tag database file
      updateTagTable ();
    }
  }
  else 
    cmlogUtil::serverPrintf ("Cannot find tag map inside cmlog_cdevMessage object\n");

}


void
cmlogClientIO::clearTagMap (cmlog_cdevMessage& msg)
{
  int   found = 0;
  short clntid = msg.getClientID ();
    
  cmlogSlistIterator ite (tagmapList_);
  cmlogTagMap* map = 0;

  // find a tag map with the same client id
  for (ite.init (); !ite; ++ite) {
    map = (cmlogTagMap *)ite ();
    if (map->sameClient (clntid)) {
      found = 1;
      break;
    }
  }
  if (found) {
#ifdef _CMLOG_DEBUG
    printf ("remove a tag map on disconnect\n");
#endif
    tagmapList_.remove ((void *)map);
  }
  else {
#ifdef _CMLOG_DEBUG
    fprintf (stderr, "Cannot find tag map with client id %d\n", clntid);
#endif
  }
}

void
cmlogClientIO::updateTagTable (void)
{
  FILE* fp = fopen (cmlogUtil::TAGTABLE_NAME, "w");
  
  if (fp == NULL) 
    cmlogUtil::serverPrintf ("Cannot open tag table file to write %s\n", 
			     cmlogUtil::TAGTABLE_NAME);
  else {
    cdevGlobalTagTable::tagTable ()->asciiDump (fp);
    fclose (fp);

    // update tag table internal cache in cdevData format
    cmlogData* data = 0;
    if (dataManager->findData ((char *)("allTags"), data) == 0)
      data->update ();
  }
}


void
cmlogClientIO::processDataWithTagMap (cmlog_cdevMessage& msg)
{
  cdevData* data = msg.getData ();
  if (data != 0) {
    int   found = 0;
    short clntid = msg.getClientID ();
    
    cmlogSlistIterator ite (tagmapList_);
    cmlogTagMap* map = 0;

    // find a tag map with the same client id
    for (ite.init (); !ite; ++ite) {
      map = (cmlogTagMap *)ite ();
      if (map->sameClient (clntid)) {
	found = 1;
	break;
      }
    }
    if (found) {
      cmlog_cdevTagMap& tagmap = map->tagmap ();
      tagmap.remoteToLocal (*data);
    }
    else {
#ifdef _CMLOG_DEBUG
      fprintf (stderr, "Cannot find tag map for this client %d\n", clntid);
#endif
    }
  }
  else 
    cmlogUtil::serverPrintf ("There is no cdevData inside cmlog_cdevMessage from client\n");
}
    

void
cmlogClientIO::setClientContext (cmlog_cdevMessage& msg)
{
  cdevData* data = msg.getData ();
  if (data != 0) {
    // get current time and convert it to a double value
    struct timeval tp;
    double         conntime;
    short          clntid;
    
    ::gettimeofday (&tp, 0);
    conntime = (double)(tp.tv_sec) + (double)(tp.tv_usec/1000000.0);

    if (conntime == prevConnTime_)
      conntime = prevConnTime_ + 0.0000001;
    prevConnTime_ = conntime;

    // get client id
    clntid = msg.getClientID ();

    // set key and client id to the incoming context data
    data->insert (cmlogUtil::CMLOG_KEY_TAG, conntime);
    data->insert (cmlogUtil::CMLOG_CLNTID_TAG, clntid);

    // create a new client context object
    cmlogClntCxt* cxt = new cmlogClntCxt (*data, clntid, conntime);
  
    // insert to the list
    cxtList_.add ((void *)cxt);
  }
  else 
    cmlogUtil::serverPrintf ("Cannot find data item inside context message\n");
}

void
cmlogClientIO::clearClientContext (cmlog_cdevMessage& msg)
{
  short          clntid;
  int            found = 0;
  
  // get client id
  clntid = msg.getClientID ();

  cmlogSlistIterator ite (cxtList_);
  cmlogClntCxt* cxt = 0;
    
  for (ite.init (); !ite; ++ite) {
    cxt = (cmlogClntCxt *) ite ();
    if (cxt->sameClient (clntid)) {
      found = 1;
      break;
    }
  }
  if (found) {
#ifdef _CMLOG_DEBUG
    printf ("clear a context on disc conn\n");
#endif
    cxtList_.remove ((void *)cxt);
    delete cxt;
  }
  else 
    cmlogUtil::serverPrintf ("cannot find context with client id %d\n", clntid);
}

void
cmlogClientIO::setClientContextId (cmlog_cdevMessage& msg)
{
  cdevData* data = msg.getData ();
  if (data != 0) {
    cmlogSlistIterator ite (cxtList_);
    cmlogClntCxt* cxt = 0;
    short         clntid = msg.getClientID ();
    
    for (ite.init (); !ite; ++ite) {
      cxt = (cmlogClntCxt *) ite ();
      if (cxt->sameClient (clntid)) {
	data->insert (cmlogUtil::CMLOG_CLNTCXT_TAG, cxt->connectionTime ());
	break;
      }
    }
  }
  else
    cmlogUtil::serverPrintf ("Cannot find data item inside incoming message\n");
}

int
cmlogClientIO::getContextData (cmlog_cdevMessage& msg, cdevData* &tdata)
{
  cmlogSlistIterator ite (cxtList_);
  cmlogClntCxt* cxt = 0;
  short         clntid = msg.getClientID ();
    
  for (ite.init (); !ite; ++ite) {
    cxt = (cmlogClntCxt *) ite ();
    if (cxt->sameClient (clntid)) {
      tdata = &(cxt->cxtData ());
      return 0;
    }
  }
  return -1;
}


void
cmlogClientIO::writeContextData (cmlogDatabase& cxtdbase)
{
  cmlogSlistIterator ite (cxtList_);
  cmlogClntCxt* cxt = 0;
     
  for (ite.init (); !ite; ++ite) {
    cxt = (cmlogClntCxt *) ite ();
    cxtdbase.put (cxt->cxtData ());
  }
}  

int
cmlogClientIO::handle_close (int fd, ACE_Reactor_Mask mask)
{
  // call default handle close
  int st = cmlogIO::handle_close (fd, mask);

  // no other threads referencing this io object, it is safe to delete
  delete this;
  return st;
}

int
cmlogClientIO::handle_input (int fd)
{
  return reader ();
}


int
cmlogClientIO::reader (void)
{
  cmlogPacket packet;
  int         n = 0;

  n = sockStream_ >> packet;
#if 0
  printf ("received %d bytes\n", n);
#endif
  if (n <= 0) {
    cmlogUtil::serverPrintf ("Client %s connection closed \n", clientHost_);
    packet.reset ();

    // log server msg
    char tmp[80];
    sprintf (tmp, "Client %s connection closed \n", clientHost_);
    logServerMsg (tmp);

    return -1;
  }

  return processData (packet);
}

int
cmlogClientIO::processData (cmlogPacket& packet)
{
  unsigned    opcode;
  int         needp = 1;
  double      key;

  // remember to free all msgs ansg msgs[] somewhere
  cmlogMsg** msgs = packet.messages ();

  if (msgs == (cmlogMsg **)CMLOG_BAD_PTR) { // error
    // log server msg
    char tmp[80];
    sprintf (tmp, "Client %s processing message error. Closing connection\n", 
	     clientHost_);
    logServerMsg (tmp);
    cmlogUtil::serverPrintf (tmp);
    
    return -1;
  }

  for (int i = 0; i < packet.numberOfData (); i++) {
    cmlog_cdevMessage& idata = (*msgs[i]);
    opcode = msgs[i]->type ();
    if (opcode == CMLOG_CONN_INFO || opcode == CMLOG_RECONN_INFO) {
      // set tag map first
      setTagMap             (idata);
      // process this data with tag map
      processDataWithTagMap (idata);
      // set client context in memory
      setClientContext      (idata);
    }
    else if (opcode == CMLOG_CLOSE_CONN) {
      clearTagMap      (idata);
      clearClientContext (idata);
      
      // free memory
      delete msgs[i];
      
      needp = 0;
    }
    else if (opcode == CMLOG_TAGMAP) {
      setTagMap        (idata);
      
      // free memory
      delete msgs[i];
      
      needp = 0;
    }
    else {
      processDataWithTagMap (idata);
      switch (opcode) {
      case CMLOG_ADD_DATA:
	{
	  cdevData tempdata = *(idata.getData ());

	  // insert time stamp if necessary
	  if (tempdata.get (cmlogUtil::CMLOG_KEY_TAG, &key) != CDEV_SUCCESS) {
	    key = cmlogUtil::currentTime ();
	    tempdata.insert (cmlogUtil::CMLOG_KEY_TAG, key);
	  }

	  cdevData fulldata;
	  cdevData* cxt = 0;
	  if (getContextData (idata, cxt) == 0) {
	    fulldata = *cxt;
	    fulldata += tempdata;
	  }
	  else
	    fulldata = tempdata;
	  // update incoming data and to notify all interested parties
	  updateLoggingDataVar (fulldata);
	}
	setClientContextId (idata);
	
	break;
      case CMLOG_ADD_CXT:
	setClientContext (idata);
	break;
      default:
	setClientContextId (idata);
	break;
      }
    }
      
    if (needp) { // need service thread to process
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
      // msgs[i] memory will be deleted by service threads
      dataQueue_->enqueue ((void *)msgs[i]);
#else
      dataQueue_->enqueue (*msgs[i]);
      delete msgs[i];
#endif
    }
  }
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  packet.empty ();  // do not delete messages inside the packet
#else
  packet.empty ();  // do delete messages inside tha packet
#endif
  delete []msgs;

  return 0;
}
