//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      cmlog Server Client Connection Acceptor
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogClientAcceptor.cc,v $
//   Revision 1.2  2001/07/25 14:30:18  chen
//   64 BIT Initial Port
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.4  1998/10/29  18:41:03  chen
// port to redHat 5.1
//
// Revision 1.3  1998/03/27  13:57:00  chen
// add disconnection information to database
//
// Revision 1.2  1997/12/08  18:34:25  chen
// port to solaris 2.6
//
// Revision 1.1  1997/08/01  15:28:52  bickley
// Added cmlog to application development system.
//
//
#include <cmlogUtil.h>
#include "cmlogClientAcceptor.h"
#include "cmlogClientIO.h"

#if defined (solaris)
#define TCP_NODELAY 1
#elif defined (__hpux)
#include <netinet/tcp.h>
#elif defined (__linux)
#include <netinet/tcp.h>
#endif


#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogClientAcceptor::cmlogClientAcceptor (ACE_Reactor& r, 
					  cmlogDataQueue* q)
:cmlogAcceptor (r, q)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogClientAcceptor Class Object\n");
#endif
}

void
cmlogClientAcceptor::writeAllClientContext (cmlogDatabase& cxtdbase)
{
  cmlogSlistIterator ite (connectionList_);
  cmlogClientIO*     io = 0;

  for (ite.init (); !ite; ++ite) {
    io = (cmlogClientIO *) ite ();
    io->writeContextData (cxtdbase);
  }
  cxtdbase.flush ();
}

#else
cmlogClientAcceptor::cmlogClientAcceptor (ACE_Reactor& r, 
					  cmlogDataQueue* q,
					  cmlogDataQueue* cq)
:cmlogAcceptor (r, q, cq)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogClientAcceptor Class Object\n");
#endif
}

void
cmlogClientAcceptor::writeAllClientContext (char* cxtfile)
{
  cmlogSlistIterator ite (connectionList_);
  cmlogClientIO*     io = 0;

  cmlogDatabase cxtdbase;

  if (cxtdbase.open  (cxtfile, O_CREAT | O_RDWR) != CDEV_SUCCESS) {
    cmlogUtil::serverPrintf ("Fatal: cmlog server cannot open file %s\n", 
			     cxtfile);
    return ;
  }
  for (ite.init (); !ite; ++ite) {
    io = (cmlogClientIO *) ite ();
    io->writeContextData (cxtdbase);
  }
  cxtdbase.flush ();
  cxtdbase.close ();
}
#endif

cmlogClientAcceptor::~cmlogClientAcceptor (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogClientAcceptor Class Object\n");
#endif
}

int
cmlogClientAcceptor::handle_input (int)
{
  ACE_INET_Addr rem_addr;

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cmlogClientIO *client = new cmlogClientIO (reactor_, dataQueue_);
#else
  cmlogClientIO *client = new cmlogClientIO (reactor_, dataQueue_, cmdQueue_);
#endif

  // accept the connection from a client
  if (listener_.accept (*client, &rem_addr, 0) != -1){
#ifdef _CMLOG_DEBUG
    printf ("Received client connection from host %s at port %d\n",
	    rem_addr.get_host_name (), client->get_handle ());
#endif

    // check number of connections
    if (connectionList_.count () >= cmlogUtil::MAXCLNT_CONS) {
      cmlogUtil::serverPrintf ("Too many client connections\n");
      delete client;
      return 0;
    }
    // remeber client address
    client->peerAddr (rem_addr);
    // set socket options to KEEP_ALIVE
    int alive = 1;
    client->setSockOption (SOL_SOCKET, SO_KEEPALIVE,
			   (void *)&alive, sizeof (alive));
    
    // set socket TCP_NODELAY option since we are handling buffering
    int nodelay = 1;
    client->setSockOption (IPPROTO_TCP, TCP_NODELAY, 
			   (void *)&nodelay, sizeof (nodelay));
    
    // turn non block socket option off
    // which gives problem when one transfers data from hp to solaris
    client->disable (ACE_NONBLOCK);

			   
    if (reactor_.register_handler (client,
				   ACE_Event_Handler::READ_MASK) == -1) {
#ifdef _CMLOG_DEBUG
      printf ("Unable to register cmlogClientIO handler\n");
#endif
      return -1;
    }
    // register this acceptor to IO handler
    client->manager (this);
    // add this client TCP connection to the list
    connectionList_.add ((void *)client);

    // log server message about new connection
    char tmp[128];
    sprintf (tmp,"Received client connection from host %s at port %d\n",
	     rem_addr.get_host_name (), client->get_handle ());
    client->logServerMsg (tmp);
    cmlogUtil::serverPrintf (tmp);
  }
  else {
#ifdef _CMLOG_DEBUG
    printf ("Accept failed\n");
#endif
    return -1;
  }
  return 0;
}


