//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of process manager
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cpProcessManager.cc,v $
//   Revision 1.2  2000/06/20 19:32:50  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:47  bickley
// Added cmlog to application development system.
//
//
#include <errno.h>
#include <signal.h>
#include "cpProcessManager.h"

cpProcessManager::cpProcessManager (size_t size)
:size_ (size), numProcesses_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cpProcessManager Class Object\n");
#endif
  if (size_ == 0)
    pids_ = 0;
  else {
    pids_ = new pid_t[size_];
    for (int i = 0; i < size_; i++)
      pids_[i] = 0;
  }
}

cpProcessManager::~cpProcessManager (void)
{
  if (pids_)
    delete []pids_;
}

int
cpProcessManager::spawn (CP_PROCESS_FUNC func,
			 void *arg,
			 pid_t *pid)
{
  int i;
  pid_t childpid = ::fork ();

  if (childpid < 0) {
    fprintf (stderr, "Cannot create child process\n");
    *pid = 0;
    return -1;
  }
  else if (childpid > 0) { // parent process
    if (numProcesses_ == size_) {
      pid_t* newpids = new pid_t[2*size_];

      // copy old information
      for (i = 0; i < size_; i++)  
	newpids[i] = pids_[i];
      
      // assign new value
      newpids[i] = childpid;
      numProcesses_ = i;
      size_ = 2*size_;
      
      // delete old memory
      delete []pids_;
      pids_ = newpids;
    }
    else
      pids_[numProcesses_ ++] = childpid;
    
    *pid = childpid;
    return 0;
  }
  else { // child prorcess
    func (arg);
    // should never get here
    ::exit (0);
  }
}

int
cpProcessManager::spawn_detached (CP_PROCESS_FUNC func,
				  void *arg,
				  pid_t *pid)
{
  pid_t childpid = ::fork ();

  if (childpid < 0) {
    fprintf (stderr, "Cannot create child process\n");
    *pid = 0;
    return -1;
  }
  else if (childpid == 0) { // child prorcess
    func (arg);
    // should never get here
    ::exit (0);
  }
  else if (childpid > 0) { // parent process
    *pid = childpid;
    return 0;
  }
}

int
cpProcessManager::spawn_detached_process (CP_PROCESS_FUNC func,
					  void *arg,
					  pid_t *pid)
{
  pid_t childpid = ::fork ();

  if (childpid < 0) {
    fprintf (stderr, "Cannot create child process\n");
    *pid = 0;
    return -1;
  }
  else if (childpid == 0) { // child prorcess
    func (arg);
    // should never get here
    ::exit (0);
  }
  else if (childpid > 0) { // parent process
    *pid = childpid;
    return 0;
  }
}


int
cpProcessManager::spawn_n (int n,
			   CP_PROCESS_FUNC func,
			   void *arg)
{
  int i = 0;

  if (n + numProcesses_ >= size_) {
    pid_t* newpids = new pid_t[n + numProcesses_ + size_];
    for (i = 0; i < numProcesses_; i++)
      newpids[i] = pids_[i];

    size_ = n + numProcesses_ + size_;
    delete []pids_;
  }
  pid_t childpid;

  for (i = 0; i < n; i++) {
    if (spawn (func, arg, &childpid) == 0) 
      pids_[numProcesses_ ++] = childpid;
  }
  return 0;
}

int
cpProcessManager::spawn_n_detached (int n,
				    CP_PROCESS_FUNC func,
				    void *arg)
{
  int i = 0;
  pid_t childpid;

  for (i = 0; i < n; i++) 
    spawn_detached (func, arg, &childpid);
  return 0;
}


int
cpProcessManager::sendSignal (int signum, pid_t pid)
{
  int err = 0;
  int i;
  if (pid == 0) {
    for (i = 0; i < numProcesses_; i++) {
      if (kill (pids_ [i], signum) != 0) 
	err = 1;
    }
  }
  else
    err = kill (pid, signum);

  return err;
}

int
cpProcessManager::detached (pid_t pid)
{
  int i;
  for (i = 0; i < numProcesses_; i++) {
    if (pid == pids_[i])
      return 0;
  }
  return 1;
}
