//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of cmlogVxLogMsgs that trap all messages from log
//      task and send them to cmlogClientD
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogVxLogMsg.cc,v $
//   Revision 1.3  2000/01/19 14:39:08  chen
//   fix compiler error for new gcc
//
//   Revision 1.2  1999/10/19 18:01:47  chen
//   Change errLog to errlog
//
//   Revision 1.1.1.1  1999/09/07 15:29:09  chen
//   CMLOG version 2.0
//
// Revision 1.4  1997/09/30  19:00:09  chen
// fix a potential problem when client daemon is shutdown
//
// Revision 1.3  1997/09/05  13:06:23  chen
// handle errLogAddListener and small bug fix on vxworks
//
// Revision 1.2  1997/08/26  16:57:16  chen
// cmlogClientD can be delete/restart on vx
//
// Revision 1.1  1997/08/01  15:26:41  bickley
// Added cmlog to application development system.
//
//
//
#ifdef __vxworks

#include <cpThread.h>
#include <cpThreadManager.h>
#include <cmlogClient.h>

// leave some space for cdevData headers 
static  char   recvbuff[CMLOG_CLNT_PIPE_MAX_BYTES/2];

// to prevent name mangling by C++
extern "C" int cmlogVxLogMsg (int pri, int stackSize);

// forward declaration
static int realCmlogVxLogMsg (cpThreadManager* manager);

// epics printf error handler stuff
typedef void (*errLogListener)(const char *message);
extern "C" void  errlogAddListener         (errLogListener);
extern "C" void  errlogRemoveListener      (errLogListener);
extern "C" void  cmlog_epics_print_handler (const char* msg);

// global cdevData and char buffer + mutex lock
static cdevData* cmlog_epics_gdata_ = 0;
static char*     cmlog_epics_txtbuf_ = 0;
static cpMutex*  cmlog_epics_lock_ = 0;

#ifdef _NO_EPICS_ERRLOG
void
errlogAddListener (errLogListener handler)
{
  // empty
}

void
errlogRemoveListener (errLogListener handler)
{
  // empty
}
#endif


static void
cmlog_epics_print_handler (const char* msg)
{
  cmlogClient *client = cmlogClient::logClient ();

  cmlog_epics_lock_->acquire ();
  strncpy (cmlog_epics_txtbuf_, msg, CMLOG_CLNT_PIPE_MAX_BYTES/2);
  client->postStaticData (*cmlog_epics_gdata_);
  cmlog_epics_lock_->release ();
}

int
cmlogVxLogMsg (int pri, int stackSize)
{
  cpThreadManager manager;

  // make this task safe from deletion
  cpThread::disablecancel ();

  // stack size should be roughly the same as the cmlog_packet size inside
  // cmlogMsg.cc
  if (pri == 0) {
    printf ("cmlogVxLogMsg uses default priority 199\n");
    pri = 199;
  }
  else 
    printf ("cmlogVxLogMsg runs at priority %d\n", pri);

  if (stackSize == 0) {
    printf ("cmlogVxLogMsg uses default stack size 10000\n");
    stackSize = 10000;
  }
  else
    printf ("cmlogvxLogMsg has stack size %d\n", stackSize);

  manager.spawn ((CP_THREAD_FUNC)realCmlogVxLogMsg, (void *)&manager,
		 (long)VX_FP_TASK,(cp_thread_t) 0, (unsigned int) pri, (void *)0, (unsigned int)stackSize);
  manager.wait ();

  printf ("cmlog vxlog msgs task finished\n");

  return 0;
}

static void
cmlogVxLogMsgCleanup (void* arg)
{
  int *fifo = (int *)arg;
  logFdDelete (*fifo);
  close (*fifo);

  // remove device from device list
  unsigned long ipaddr = cmlogUtil::hostIpAddr ();
  char     pipepath [128];
  sprintf (pipepath, CMLOG_CLNT_PIPE2, ipaddr);

  DEV_HDR* dev = iosDevFind (pipepath, 0);
  if (dev) 
    iosDevDelete (dev);  


  // remove epics error handler
  errlogRemoveListener (cmlog_epics_print_handler);

  // remove cdev data
  if (cmlog_epics_gdata_)
    delete cmlog_epics_gdata_;

  if (cmlog_epics_lock_)
    delete cmlog_epics_lock_;
}

static int
cmlogVxLogFifo (void)
{
  // open a new pipe to the monitor log task
  unsigned long ipaddr = cmlogUtil::hostIpAddr ();
  char     pipepath [128];
  sprintf (pipepath, CMLOG_CLNT_PIPE2, ipaddr);
  int     fifo = -1;

  if (pipeDevCreate (pipepath,
		     CMLOG_CLNT_PIPE_NUM_MSGS,
		     CMLOG_CLNT_PIPE_MAX_BYTES) != OK) {
    fprintf (stderr, "Cannot create log pipe device %s\n", pipepath);
    exit (1);
  }
  else {
    fifo = open (pipepath, O_RDWR, 0666);
    if (fifo == ERROR) {
      fprintf (stderr, "Cannot open log pipe %s\n", pipepath);
      exit (1);
    }
  }
  return fifo;
}

static void
cmlogVxLogErrorInfo (char* string, cmlogClient& client)
{
  // lock is held already
  strncpy (cmlog_epics_txtbuf_, string, CMLOG_CLNT_PIPE_MAX_BYTES/2);
  client.postStaticData (*cmlog_epics_gdata_);
}

static int
realCmlogVxLogMsg (cpThreadManager* manager)
{
  // make sure the following code get executed before anything
  taskLock ();

  cpThreadControl control (manager);

  cmlogClient *client = cmlogClient::logClient ();

  if (client->connect () != CMLOG_SUCCESS) {
    printf ("Cannot connect to cmlogClientD\n");
    return -1;
  }

  /* open a pipe to monitor log task */
  int fifo = cmlogVxLogFifo ();
  if (logFdAdd (fifo) == ERROR) {
    fprintf (stderr, "Cannot do logFdAdd on %d\n", fifo);
    return -1;
  }

  /* add cleanup handler here */
  cpThread::add_cleanup_handler (cmlogVxLogMsgCleanup, (void *)&fifo);

  /* use select to monitor this pipe */
  fd_set readfd;
  int    nfound;
  int    numtoread;
  int    bytes;
  int    i = 0;


  /* create a cdevData here */
  void* vptr = 0;
  cmlog_epics_gdata_ = new cdevData ();

  /* put some char in the buffer */
  for (i = 0; i < CMLOG_CLNT_PIPE_MAX_BYTES/2 - 1; i++)
    recvbuff[i] = 'F';
  recvbuff[i] = '\0';

  /* make sure cdevData allocate the same size of char buffer */
  cmlog_epics_gdata_->insert (cmlogUtil::CMLOG_TEXT_TAG, recvbuff);
  cmlog_epics_gdata_->insert (cmlogUtil::CMLOG_FACILITY_TAG, "EPICS");

  if (cmlog_epics_gdata_->find (cmlogUtil::CMLOG_TEXT_TAG, vptr) 
      != CDEV_SUCCESS) {
    fprintf (stderr, "Cannot access text buffer inside cdevData, Quit\n");
    taskUnlock ();
    exit (1);
  }

  /* access the allocated memory */
  cmlog_epics_txtbuf_ = (char *)vptr;

  /* create lock for text buffer */
  cmlog_epics_lock_ = new cpMutex ();

  /* register epics error handler */
  errlogAddListener (cmlog_epics_print_handler);

  taskUnlock ();  

  while (1) {
    FD_ZERO (&readfd);
    FD_SET  (fifo, &readfd);
    
    do {
      nfound = select (fifo + 1, &readfd, 0, 0, 0);
    }while (nfound == -1 && errno == EINTR);

    if (nfound == -1) { /* bad thing */
      fprintf (stderr, "Bad fifo %d\n", fifo);
      return -1;
    }
    else if (nfound >= 1) {
      /* lock the text buffer */
      cmlog_epics_lock_->acquire ();

      i = 0;
      while (1) {
	// use ioctl to find how many bytes in the first message
	ioctl (fifo,  FIONREAD, (int) &numtoread);

	if (i + numtoread - 1 >= CMLOG_CLNT_PIPE_MAX_BYTES/2) {
	  cmlog_epics_txtbuf_[i] = '\0';
	  break;
	}
	// read out this much bytes
	bytes = read (fifo, &(cmlog_epics_txtbuf_[i]), numtoread);

	if (bytes != numtoread) {
	  cmlogVxLogErrorInfo ("Error: read error for logmsgs", *client);
	  cmlog_epics_lock_->release ();
	  return -1;
	}

	if (cmlog_epics_txtbuf_[i + bytes - 1] == '\n' || 
	    cmlog_epics_txtbuf_[i + bytes - 1]=='\0') {
	  cmlog_epics_txtbuf_[i + bytes - 1] = '\0';
	  break;
	}
	i += bytes;
      }
      
      /* send to cmlogClientD */
      client->postStaticData (*cmlog_epics_gdata_);

      /* relase lock */
      cmlog_epics_lock_->release ();
    }
  }
			   
  /* should never get here */
  return 0;
}
#endif
