/* -*- C++ -*- */
// ============================================================================
//
// = LIBRARY
//    ace
// 
// = FILENAME
//    Thread_Manager.h 
//
// = AUTHOR
//    Doug Schmidt 
// 
// ============================================================================

#if !defined (ACE_THREAD_MANAGER_H)
#define ACE_THREAD_MANAGER_H

#include "ace/Log_Msg.h"
#include "ace/Thread.h"
#include "ace/Synch.h"

class ACE_Thread_Manager;

class ACE_Thread_Descriptor
  // = Title
  //    Information for controlling groups of threads.
{
friend class ACE_Thread_Manager;
public:
  enum Thread_State
  {
    IDLE,
    SPAWNED,
    RUNNING,
    SUSPENDED,
    TERMINATED
  };

  // = Initialization method.
  ACE_Thread_Descriptor (void);

  // = Accessor methods.
  thread_t self (void);
  // Unique thread id.

  void self (hthread_t &);
  // Unique kernel-level thread handle.

  int grp_id (void);
  // Group ID.

  Thread_State state (void);
  // Current state of the thread.

private:

  thread_t thr_id_;
  // Unique thread ID.
    
  hthread_t thr_handle_;
  // Unique kernel-level thread handle.

  int grp_id_;
  // Group ID.

  Thread_State thr_state_;
  // Current state of the thread.
};

#if !defined (ACE_HAS_THREADS)

class ACE_Thread_Manager
{
public:
  enum
  {
    DEFAULT_SIZE = 100
  };

  ACE_Thread_Manager (int = 0) {}
  ~ACE_Thread_Manager (void) {}

  int open (size_t = DEFAULT_SIZE) { return -1; }
  int close (void) { return -1; }
  int spawn (ACE_THR_FUNC, 
	     void *,
	     long, 
	     thread_t * = 0, 
	     hthread_t * = 0,
	     u_int = 0,
	     void * = 0, 
	     size_t = 0) { return -1;}
  int spawn_n (int, 
	       ACE_THR_FUNC, 
	       void *, 
	       long, 
	       u_int = 0) { return -1;}
  void *exit (void *) { return 0; }
  void wait (void) {}
  int descriptor (thread_t, ACE_Thread_Descriptor &) { return -1;}
  int descriptor (hthread_t, ACE_Thread_Descriptor &) { return -1;}
  int suspend_all (void) { return -1; }
  int suspend (thread_t) { return -1; }
  int suspend_grp (thread_t) { return -1; }
  int resume_all (void) { return -1; }
  int resume (thread_t) { return -1; }
  int resume_grp (thread_t) { return -1; }
  int kill_all (int) { return -1; }
  int kill (thread_t) { return -1; }
  int kill_grp (thread_t) { return -1; }
  void dump (void) const { }
};

class ACE_Thread_Control
{
public:
  ACE_Thread_Control (ACE_Thread_Manager *, int = 0) {}
  ~ACE_Thread_Control (void) {}
  void *exit (void *) { return 0; }
  void *status (void *) { return 0; }
  void *status (void) { return 0; }
  void dump (void) const { }
};
#else

// This typedef should be (and used to be) inside the
// ACE_Thread_Manager declaration.  But, it caused compilation
// problems on g++/VxWorks/i960 with -g.  Note that
// ACE_Thread_Manager::THR_FUNC is only used internally in
// ACE_Thread_Manager, so it's not useful for anyone else.
// It also caused problems on IRIX5 with g++.
#if defined (__GNUG__)
typedef int (ACE_Thread_Manager::*THR_FUNC)(int, int);
#endif

class ACE_Thread_Manager
  // = TITLE
  //    Manages a pool of threads.
  //
  // = DESCRIPTION
  //    This class allows operations on groups of threads atomically.
{
friend class ACE_Thread_Control;
public:
  enum
  {
    DEFAULT_SIZE = 100
  };

  // = Initialization and termination methods.
  ACE_Thread_Manager (size_t size = ACE_Thread_Manager::DEFAULT_SIZE);
  ~ACE_Thread_Manager (void);

  int open (size_t size = DEFAULT_SIZE); 
  // Initialize the manager with room for SIZE threads.

  int close (void);		
  // Release all resources.

  int spawn (ACE_THR_FUNC func, 
	     void *args, 
	     long flags, 
	     thread_t * = 0, 
	     hthread_t *t_handle = 0,
	     u_int priority = 0,
	     void *stack = 0, 
	     size_t stack_size = 0);
  // Create a new thread, which executes <func>.  

  // Returns: on success a unique group id that can be used to control
  // other threads added to the same group.  On failure, returns -1.

  int spawn_n (int n, 
	       ACE_THR_FUNC func, 
	       void *args, 
	       long flags,
	       u_int priority = 0);
  // Create N new threads, all of which execute <func>.  
  
  // Returns: on success a unique group id that can be used to control
  // all of the threads in the same group.  On failure, returns -1.

  void *exit (void *status);	
  // Called to clean up when a thread exits.

  int wait (ACE_Time_Value *timeout = 0);	
  // Block until there are no more threads running or <timeout>
  // expires.  Returns 0 on success and -1 on failure.

  // = Accessors for <ACE_Thread_Descriptors>.
  int descriptor (thread_t, ACE_Thread_Descriptor &);
  // Return the thread descriptor (indexed by thread_t).  Returns 0 on
  // success and -1 if not found.

  int descriptor (hthread_t, ACE_Thread_Descriptor &);
  // Return the thread descriptor (indexed by hthread_t).  Returns 0
  // on success and -1 if not found.

  // = Suspend methods.
  int suspend_all (void);
  // Suspend all threads 
  int suspend (thread_t);
  // Suspend a single thread.
  int suspend_grp (int grp_id);
  // Suspend a group of threads.

  // = Resume methods.
  int resume_all (void);
  // Resume all stopped threads 
  int resume (thread_t);
  // Resume a single thread.
  int resume_grp (int grp_id);
  // Resume a group of threads.

  // = Kill methods (send signals...).
  int kill_all (int signum);
  // Send signum to all stopped threads 
  int kill (thread_t, int signum);
  // Kill a single thread.
  int kill_grp (int grp_id, int signum);
  // Kill a group of threads.

  // = Set/get group ids for a particular thread id.
  int set_grp (thread_t, int grp_id);
  int get_grp (thread_t, int &grp_id);

  void dump (void) const;
  // Dump the state of an object.

  ACE_ALLOC_HOOK_DECLARE;
  // Declare the dynamic allocation hooks.

private:
  int resize (size_t);
  // Resize the pool of Thread_Descriptors.

  int spawn_i (ACE_THR_FUNC func, 
	       void *args, 
	       long flags, 
	       thread_t * = 0, 
	       hthread_t *t_handle = 0,
	       u_int priority = 0,
	       void *stack = 0, 
	       size_t stack_size = 0);
  // Create a new thread (must be called with locks held).

  int find (thread_t t_id);
  // Locate the index of the table slot occupied by <t_id>.  Returns
  // -1 if <t_id> is not in the table doesn't contain <t_id>.

  int insert_thr (thread_t t_id, hthread_t);
  // Insert a thread in the table (checks for duplicates).
  // Omitting the thread handle won't work on Win32...

  int append_thr (thread_t t_id, hthread_t, 
		  ACE_Thread_Descriptor::Thread_State);
  // Append a thread in the table (adds at the end, growing the table
  // if necessary).

  void remove_thr (int i);	
  // Remove thread from the table. 

  // = The following four methods implement a simple scheme for
  // operating on a collection of threads atomically.
#if !defined(__GNUG__)
  typedef int (ACE_Thread_Manager::*THR_FUNC)(int, int);
#endif

  int apply_grp (int grp_id, THR_FUNC, int = 0);
  // Apply <func> to all members of the table that match the <grp_id>.

  int apply_all (THR_FUNC, int = 0);
  // Apply <func> to all members of the table.

  int resume_thr (int i);
  // Resume the thread at index <i>.

  int suspend_thr (int i);
  // Suspend the thread at index <i>.

  int kill_thr (int i, int signum);
  // Send signal <signum> to the thread at index <i>.

  ACE_Thread_Descriptor *thr_table_;
  // Vector that describes thread state within the Thread_Manager.

  size_t max_table_size_;
  // Maximum number of threads we can manage (should be dynamically
  // allocated).

  size_t current_count_;
  // Current number of threads we are managing.

  int grp_id_;
  // Keeps track of the next group id to assign.

  // = ACE_Mutex and condition variable for synchronizing termination.
  ACE_Mutex lock_;
  ACE_Condition_Mutex zero_cond_;
};


class ACE_Thread_Control
  // = TITLE
  //     Used to keep track of a thread's activities within its entry
  //     point function.
{
public:
  ACE_Thread_Control (ACE_Thread_Manager *tm, int insert = 0);
  // Initialize the thread control object.  If <insert> != 0, then
  // register the thread with the Thread_Manager.

  ~ACE_Thread_Control (void);
  // Implicitly kill the thread on exit and remove it from its
  // associated ThreadManager.

  void *exit (void *status);
  // Explicitly kill the thread on exit and remove it from its
  // associated ThreadManager.

  void *status (void *status);
  // Set the exit status (and return existing status).

  void *status (void);
  // Get the current exit status.

  void dump (void) const;
  // Dump the state of an object.

  ACE_ALLOC_HOOK_DECLARE;
  // Declare the dynamic allocation hooks.

private:
  ACE_Thread_Manager *tm_;
  // Pointer to the thread manager for this block of code.

  void *status_;
  // Keeps track of the exit status for the thread.
};

#if defined (__INLINE__)
#include "ace/Thread_Manager.i"
#endif /* __INLINE__ */

#endif /* !defined (ACE_HAS_THREADS) */
#endif /* ACE_THREAD_MANAGER_H */

