/* -*- C++ -*- */

// ============================================================================
//
// = LIBRARY
//    ace
// 
// = FILENAME
//    Thread.h 
//
// = AUTHOR
//    Doug Schmidt 
// 
// ============================================================================

#if !defined (ACE_THREAD_H)
#define ACE_THREAD_H

#include "ace/ACE.h"

#if !defined (ACE_HAS_THREADS) 
class ACE_Thread
{
public:
  static thread_t self (void);
  // Return the unique ID of the thread.

  static void self (hthread_t &t_id);
  // Return the unique kernel ID of the thread.

  static void exit (void *status = 0);
  // Exit the current thread and return "status".
};
#else
class ACE_Thread
{
  // = TITLE
  //   Provides a wrapper for threads.
  //
  // = DESCRIPTION
  //   This class provides a common interface that is mapped onto
  //   either POSIX Pthreads, Solaris threads, or Win32 threads.

public:
  static int spawn (ACE_THR_FUNC func, 
		    void *arg, 
		    long flags, 
		    thread_t *t_id = 0, 
		    hthread_t *t_handle = 0,
		    u_int priority = 0,
		    void *stack = 0, 
		    size_t stack_size = 0);
  // Spawn a new thread, which executes "func" with argument "arg".

  static int spawn_n (size_t n, 
		      ACE_THR_FUNC func, 
		      void *arg, 
		      long flags,
		      u_int priority = 0,
		      void *stack = 0, 
		      size_t stack_size = 0);
  // Spawn N new threads, which execute <func> with argument <arg>.
  // Returns the number of threads actually spawned (if this doesn't
  // equal the number requested then something has gone wrong and
  // <errno> will explain...).

  static int spawn_n (thread_t thread_ids[], 
		      size_t n, 
		      ACE_THR_FUNC func, 
		      void *arg, 
		      long flags, 
		      u_int priority = 0,
		      void *stack = 0, 
		      size_t stack_size = 0);
  // Spawn N new threads, which execute <func> with argument <arg>.
  // The thread_ids of successfully spawned threads will be placed
  // into the <thread_ids> buffer, which must be the same size as <n>.
  // Returns the number of threads actually spawned (if this doesn't
  // equal the number requested then something has gone wrong and
  // <errno> will explain...).

  static int join (thread_t,
		   thread_t * = 0, 
		   void ** = 0);
  // Wait for one or more threads to exit.

  static int join (hthread_t, 
		   void ** = 0);
  // Wait for one thread to exit.

  static int resume (hthread_t);
  // Continue the execution of a previously suspended thread.

  static int suspend (hthread_t);
  // Suspend the execution of a particular thread.

  static int kill (thread_t, int signum);
  // Send a signal to the thread.

  static void yield (void);
  // Yield the thread to another.

  static void self (hthread_t &t_id);
  // Return the unique kernel ID of the thread.

  static thread_t self (void);
  // Return the unique ID of the thread.

  static void exit (void *status = 0);
  // Exit the current thread and return "status".

  static int getconcurrency (void);
  // Set the LWP concurrency level of the process.

  static int setconcurrency (int new_level);
  // Get the LWP concurrency level of the process.

  static int sigsetmask (int how, 
			 const sigset_t *set, 
			 sigset_t *oset = 0);
  // Change and/or examine calling thread's signal mask.

  static int keycreate (thread_key_t *keyp,
			void (*destructor)(void *value));
  // Allocates a <keyp> that is used to identify data that is specific
  // to each thread in the process.  The key is global to all threads
  // in the process.

  static int setspecific (thread_key_t key, 
			  void *value);
  // Bind value to the thread-specific data key, <key>, for the calling
  // thread.

  static int getspecific (thread_key_t key, 
			  void **valuep);
  // Stores the current value bound to <key> for the calling thread
  // into the location pointed to by <valuep>.

  static int disablecancel (struct cancel_state *old_state);
  // Disable thread cancellation.

  static int enablecancel (struct cancel_state *old_state, 
			   int flag);
  // Enable thread cancellation.

  static int setcancelstate (struct cancel_state &new_state,
			     struct cancel_state *old_state);
  // Set the cancellation state.

  static int cancel (thread_t t_id);
  // Cancel a thread.

  static void testcancel (void);
  // Test the cancel?

private:
  ACE_Thread (void);
  // Ensure that we don't get instantiated.
};
#endif /* ACE_HAS_THREADS */

#if defined (__INLINE__)
#include "ace/Thread.i"
#endif /* __INLINE__ */
#endif /* ACE_THREAD_H */
