/* Contains the definitions for the ACE_SOCK datagram abstraction. */

#include "ace/SOCK_Dgram_Bcast.h"
#include "ace/Log_Msg.h"

ACE_ALLOC_HOOK_DEFINE(ACE_SOCK_Dgram_Bcast)

void
ACE_SOCK_Dgram_Bcast::dump (void) const
{
  ACE_TRACE ("ACE_SOCK_Dgram_Bcast::dump");
}

/* Here's the simple-minded constructor. */

ACE_SOCK_Dgram_Bcast::ACE_SOCK_Dgram_Bcast (void)
  : ifptr_ (0)
{
  ACE_TRACE ("ACE_SOCK_Dgram_Bcast::ACE_SOCK_Dgram_Bcast");
}

/* Here's the general-purpose constructor used by a connectionless
   datagram ``server''... */

ACE_SOCK_Dgram_Bcast::ACE_SOCK_Dgram_Bcast (const ACE_Addr &local, 
					    int protocol_family, 
					    int protocol)
  : ACE_SOCK_Dgram (local, protocol_family, protocol), ifptr_ (0)
{
  ACE_TRACE ("ACE_SOCK_Dgram_Bcast::ACE_SOCK_Dgram_Bcast");
  if (this->mk_broadcast () == -1) 
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_SOCK_Dgram_Bcast"));
}

/* Here's the general-purpose open routine. */

int
ACE_SOCK_Dgram_Bcast::open (const ACE_Addr &local, 
			    int protocol_family, 
			    int protocol)
{
  ACE_TRACE ("ACE_SOCK_Dgram_Bcast::open");
  if (this->ACE_SOCK_Dgram::open (local, protocol_family, 
				  protocol) == -1)
    return -1;

  if (this->mk_broadcast () == -1)
    return -1;

  return 0;
}

/* Make broadcast available for Datagram socket */

int
ACE_SOCK_Dgram_Bcast::mk_broadcast (void)
{
  int one = 1;

  if (ACE_OS::setsockopt (this->get_handle (),
                          SOL_SOCKET,
                          SO_BROADCAST,
                          (char *) &one,
                          sizeof one) == -1)
    return -1;

#if !defined (ACE_WIN32)
  ACE_HANDLE s = this->get_handle ();

  char   buf[BUFSIZ];

  struct ifconf ifc;
  struct ifreq *ifr;

  struct ifreq  flags;
  struct ifreq  if_req;

  ifc.ifc_len = sizeof buf;
  ifc.ifc_buf = buf;

  if (ioctl (s, SIOCGIFCONF, (int)&ifc) < 0) {
    fprintf (stderr, "ioctl failed on get_ifnode call\n");
    return -1;
  }

  ifr = ifc.ifc_req;

  for (int n = ifc.ifc_len / sizeof (struct ifreq) ; n > 0; n--, ifr++) {
    if (ifr->ifr_addr.sa_family != AF_INET) { // not ip
      continue;
    }

    flags = if_req = *ifr;
    
    if (ioctl(s, SIOCGIFFLAGS, (int)&flags) < 0) 
      continue;
      
    if ((flags.ifr_flags & IFF_UP) == 0) 
      continue;

    if (flags.ifr_flags & IFF_LOOPBACK) 
      continue;

    if (flags.ifr_flags & IFF_BROADCAST) {
      if (ioctl(s, SIOCGIFBRDADDR, (int)&if_req) < 0) {
	fprintf (stderr, "ioctl error \n");
      }
      else {
	struct ifnode *ptr = new ifnode();
	ptr->brdAddr       = *(struct sockaddr_in *) &if_req.ifr_broadaddr;
	ptr->next	   = ifptr_;
	ifptr_ = ptr;
      }
    }
    else 
      fprintf (stderr, "Broadcast is not enabled\n");
  }
#endif
  if (!ifptr_)return -1;
  return 0;
}

/* Broadcast the datagram to every interface.  Returns the average
   number of bytes sent. */

ssize_t
ACE_SOCK_Dgram_Bcast::send (const void *buf, 
			    size_t n,
			    u_short port_number, 
			    int flags) const
{
  ACE_TRACE ("ACE_SOCK_Dgram_Bcast::send");
  size_t  iterations = 0;
  ssize_t bytes	     = 0;

  if (this->ifptr_ == 0) 
    return -1;

  for (struct ifnode *tptr = this->ifptr_;
       tptr != 0;
       tptr = tptr->next)
    {
      sockaddr_in to_addr = tptr->brdAddr;
      to_addr.sin_port	  = htons (port_number);

      bytes += ACE_OS::sendto (this->get_handle(), (const char *) buf, n, flags, 
			 (struct sockaddr *) &to_addr, sizeof to_addr);
      iterations++;
    }

  return iterations == 0 ? 0 : bytes / iterations;
}

#if defined (ACE_HAS_MSG)
/* Broadcast datagram to every interfaces */

ssize_t
ACE_SOCK_Dgram_Bcast::send (const iovec iov[], 
			    size_t n, 
			    u_short port_number, 
			    int flags) const
{
  ACE_TRACE ("ACE_SOCK_Dgram_Bcast::send");
  if (this->ifptr_ == 0)
    printf ("ACE_SOCK_Dgram_Bcast::send : ERROR invalid ifptr\n");
    return ACE_INVALID_HANDLE;
  
  for (struct ifnode *tptr = this->ifptr_; tptr != 0; tptr++) 
    {
      struct sockaddr_in to_addr = tptr->brdAddr;
      msghdr send_msg;

      to_addr.sin_port	         = htons (port_number);
      send_msg.msg_iov		 = (iovec *) iov;
      send_msg.msg_iovlen	 = n;
      send_msg.msg_name		 = (char *) &to_addr;
      send_msg.msg_namelen	 = sizeof to_addr;
      send_msg.msg_accrights	 = 0;
      send_msg.msg_accrightslen	 = 0;
      ACE_OS::sendmsg (this->get_handle (), &send_msg, flags);
    }

  return 0;
}

/* Broadcast an ACE_IO_Vector of size N to ADDR as a datagram (note
   that addr must be preassigned to the broadcast address of the
   subnet...) */

ssize_t
ACE_SOCK_Dgram_Bcast::send (const iovec iov[], 
			    size_t n, 
			    const ACE_Addr &addr, 
			    int flags) const
{
  ACE_TRACE ("ACE_SOCK_Dgram_Bcast::send");
  msghdr   msg;  

  msg.msg_iov          = (iovec *) iov;
  msg.msg_iovlen       = n;
  msg.msg_name	       = (char *) addr.get_addr ();
  msg.msg_namelen      = addr.get_size ();
  msg.msg_accrights    = 0;
  msg.msg_accrightslen = 0;
  return ACE_OS::sendmsg (this->get_handle (), &msg, flags);
}
#endif /* ACE_HAS_MSG */
