#include "ace/OS.h"

// Perhaps we should *always* include ace/OS.i in order to make sure
// we can always link against the OS symbols?
#if !defined (ACE_HAS_INLINED_OSCALLS)
#include "ace/OS.i"
#endif /* ACE_HAS_INLINED_OS_CALLS */

void 
ACE_OS::mutex_lock_cleanup (void *mutex)
{
  ACE_TRACE ("ACE_OS::rw_cleanup");
#if defined (ACE_HAS_THREADS)
#if defined (ACE_HAS_PTHREADS)
  mutex_t *p_lock = (mutex_t *) mutex;
  ACE_OS::mutex_unlock (p_lock);
#endif /* ACE_HAS_PTHREADS */
#endif /* ACE_HAS_THREADS */
}

// This is necessary to deal with POSIX pthreads insanity.  This
// guarantees that we've got a "zero'd" thread id even when thread_t
// is implemented as a structure...
thread_t ACE_OS::NULL_thread;

ACE_OS::ACE_OS (void)
{
  ACE_TRACE ("ACE_OS::ACE_OS");
}

#if defined (ACE_WIN32)

#include "ace/Synch.h"

// Thread Specific Key management.

struct ACE_TSS_Key_Info
{
  thread_key_t key_;
  void (*destructor_)(void *);
  ACE_TSS_Key_Info* link_;

  static int deletion_flag_;
  static ACE_TSS_Key_Info *first_key_;
  static ACE_Thread_Mutex lock_;
  // Special thread startup packet (used below in thr_create).
};

// Static definitions.
ACE_TSS_Key_Info *ACE_TSS_Key_Info::first_key_ = 0;
int ACE_TSS_Key_Info::deletion_flag_ = 0;
ACE_Thread_Mutex ACE_TSS_Key_Info::lock_;

// Special thread startup packet.

class ACE_Spawn_Args
{
public:
  ACE_Spawn_Args (ACE_THR_FUNC f, void *a);
  // Constructor

  static void *spawn (ACE_Spawn_Args *);

private:
  // = Arguments to thread startup.

  ACE_THR_FUNC func_;
  // Thread startup function.

  void *arg_;
  // Argument to thread startup function.
};

ACE_Spawn_Args::ACE_Spawn_Args (ACE_THR_FUNC f, void *a)
  : func_(f), 
    arg_(a) 
{
  ACE_TRACE ("ACE_Spawn_Args::ACE_Spawn_Args");
}

void *
ACE_Spawn_Args::spawn (ACE_Spawn_Args *spinfo)
{
  ACE_TRACE ("ACE_Spawn_Args::ACE_spawn");
  ACE_THR_FUNC func = spinfo->func_;
  void *arg = spinfo->arg_;
  delete spinfo;

  void *return_value = (*func) (arg);  // Call thread entry point.

  ACE_OS::thr_destroy_tss (); // If dropped off end, call destructors.
  return return_value; // Drop off end.
}

void 
ACE_OS::thr_destroy_tss (void)
{
  ACE_TRACE ("ACE_OS::thr_destroy_tss");
  ACE_Guard<ACE_Thread_Mutex> mon (ACE_TSS_Key_Info::lock_);

  // Ensure that we don't have recursive destruction.
  if (ACE_TSS_Key_Info::deletion_flag_ != 0)
    return;
  else
    ACE_TSS_Key_Info::deletion_flag_ = 1;

  for (ACE_TSS_Key_Info *next = ACE_TSS_Key_Info::first_key_; 
       next; 
       next = next->link_)
    {
      void *tsdata; 

      if ((ACE_OS::thr_getspecific (next->key_, &tsdata) == 0) 
	  && (next->destructor_) && tsdata)
	next->destructor_ (tsdata);   
    }

  ACE_TSS_Key_Info::deletion_flag_ = 0;
}

#else
void 
ACE_OS::thr_destroy_tss (void)
{
  ACE_TRACE ("ACE_OS::thr_destroy_tss");
}
#endif // WIN32

int
ACE_OS::thr_create (ACE_THR_FUNC func,
		    void *args,
		    long flags,
		    thread_t *thr_id,
		    hthread_t *thr_handle,
                    u_int priority,
		    void *stack,
		    size_t stacksize)
{
  ACE_TRACE ("ACE_OS::thr_create");

#if defined (ACE_HAS_THREADS)
#if defined (ACE_HAS_STHREADS)
  int result;
  ACE_OSCALL (ACE_ADAPT_RETVAL (::thr_create (stack, stacksize, func, args,
					      flags, thr_id), result), 
	      int, -1, result);
  if (result == 0 && thr_handle != 0)
    *thr_handle = *thr_id;
  return result;
#elif defined (ACE_HAS_PTHREADS)
  int result;
  pthread_attr_t attr;
  thread_t tmp_thr;
  thread_t *p_thr;

  if (pthread_attr_init (&attr) != 0)
     return -1;
  else if (priority != 0)
  {
    struct sched_param sparam;

    sparam.sched_priority = priority > PRIORITY_MAX ? PRIORITY_MAX : priority;

    if (pthread_attr_setschedparam (&attr, &sparam) != 0)
      {
	pthread_attr_destroy (&attr);
        return -1;
      }
  }

  if (stacksize != 0)
    {
      size_t size = stacksize > PTHREAD_STACK_MIN 
	? stacksize: PTHREAD_STACK_MIN;

      if (pthread_attr_setstacksize (&attr, size) != 0)
	{
	  pthread_attr_destroy (&attr);
	  return -1;
	}
    }
#if !defined (ACE_LACKS_THREAD_STACK_ADDR)
  if(stack != 0)
    {
      if (pthread_attr_setstackaddr (&attr, stack) != 0)
	{
	  pthread_attr_destroy (&attr);
	  return -1;
	}
    }
#endif /* !ACE_LACKS_THREAD_STACK_ADDR */
  if (flags != 0)
    {
      if (ACE_BIT_ENABLED (flags, THR_DETACHED) 
	  || ACE_BIT_ENABLED (flags, THR_UNDETACHED))
	{
	  int dstate = PTHREAD_CREATE_UNDETACHED;
	  if (ACE_BIT_ENABLED (flags, THR_DETACHED))
	    dstate = PTHREAD_CREATE_DETACHED;
	  if (pthread_attr_setdetachstate (&attr, dstate) != 0)
	    {
	      pthread_attr_destroy (&attr);
	      return -1;
	    }
	}

      if (ACE_BIT_ENABLED (flags, THR_SCHED_FIFO)
	  || ACE_BIT_ENABLED (flags, THR_SCHED_RR)
	  || ACE_BIT_ENABLED (flags, THR_SCHED_DEFAULT))
	{
	  int spolicy;

	  if (ACE_BIT_ENABLED (flags, THR_SCHED_DEFAULT))
	    spolicy = SCHED_OTHER;
	  else if (ACE_BIT_ENABLED (flags, THR_SCHED_FIFO))
	    spolicy = SCHED_FIFO;
	  else
	    spolicy = SCHED_RR;
	  if (pthread_attr_setschedpolicy (&attr, spolicy) != 0)
	    {
	      pthread_attr_destroy (&attr);
	      return -1;
	    }
	}

      if (ACE_BIT_ENABLED (flags, THR_SCHED_INHERIT)
	  || ACE_BIT_ENABLED (flags, THR_SCHED_EXPLICIT))
	{
	  int sched = PTHREAD_EXPLICIT_SCHED;
	  if (ACE_BIT_ENABLED (flags, THR_SCHED_INHERIT))
	    sched = PTHREAD_INHERIT_SCHED;
	  if (pthread_attr_setinheritsched (&attr, sched) != 0)
	    {
	      pthread_attr_destroy (&attr);
	      return -1;
	    }
	}
#if !defined (ACE_LACKS_THREAD_PROCESS_SCOPING)
      if (ACE_BIT_ENABLED (flags, THR_SCOPE_SYSTEM)
	|| ACE_BIT_ENABLED (flags, THR_SCOPE_PROCESS))
	{
	  int scope = PTHREAD_SCOPE_PROCESS;
	  if (ACE_BIT_ENABLED (flags, THR_SCOPE_SYSTEM))
	    scope = PTHREAD_SCOPE_SYSTEM;
	  if (pthread_attr_setscope (&attr, scope) != 0)
	    {
	      pthread_attr_destroy (&attr);
	      return -1;
	    }
	}
#endif /* !ACE_LACKS_THREAD_PROCESS_SCOPING */
    }

  p_thr = (thr_id == 0 ? &tmp_thr : thr_id);

  ACE_OSCALL (ACE_ADAPT_RETVAL (::pthread_create (p_thr, &attr, func, args), result), 
	      int, -1, result);

  pthread_attr_destroy (&attr);

  if (result == 0 && thr_handle != 0)
    *thr_handle = (hthread_t) p_thr;

  return result;
#elif defined (ACE_HAS_WTHREADS)
  thread_t t;
  hthread_t handle;

  if (thr_id == 0)
    thr_id = &t;
  if (thr_handle == 0)
    thr_handle = &handle;

  ACE_Spawn_Args *spawn_args = new ACE_Spawn_Args (func, args);

  if (spawn_args == 0)
    return -1;
  *thr_handle = ::CreateThread (NULL, stacksize,
				LPTHREAD_START_ROUTINE (ACE_Spawn_Args::spawn),
				spawn_args, flags, thr_id);
  return *thr_handle != 0 ? 0 : -1;
#endif /* ACE_HAS_STHREADS */
#else
  ACE_NOTSUP_RETURN (-1);
#endif /* ACE_HAS_THREADS */		
}

INLINE int 
ACE_OS::thr_keycreate (thread_key_t *key, void (*dest) (void *))
{
  ACE_TRACE ("ACE_OS::thr_keycreate");
#if defined (ACE_HAS_THREADS)
#if defined (ACE_HAS_STHREADS)
  ACE_OSCALL_RETURN (ACE_ADAPT_RETVAL (::thr_keycreate (key, dest), _result), int, -1);
#elif defined (ACE_HAS_PTHREADS)
  ACE_OSCALL_RETURN (ACE_ADAPT_RETVAL (::pthread_key_create (key, dest), _result), int, -1);
#elif defined (ACE_HAS_WTHREADS)
  *key = ::TlsAlloc ();

  if (*key != 0xFFFFFFFF)
    {
      ACE_Guard<ACE_Thread_Mutex> mon (ACE_TSS_Key_Info::lock_);

      ACE_TSS_Key_Info *next = new ACE_TSS_Key_Info;

      if (next)
	{
	  next->key_        = *key;   
	  next->destructor_ = dest;   
	  next->link_       = ACE_TSS_Key_Info::first_key_; 
	  ACE_TSS_Key_Info::first_key_ = next;
	  return 0;
	}
    }
  return -1;
#endif /* ACE_HAS_STHREADS */
#else
  ACE_NOTSUP_RETURN (-1);
#endif /* ACE_HAS_THREADS */		     
}

#if defined (ACE_NEEDS_WRITEV)

// "Fake" writev for sites without it.

extern "C" int
writev (ACE_HANDLE fd, const struct iovec *vp, int vpcount)
{
  ACE_TRACE ("::writev");

  int count;

  for (count = 0; --vpcount >= 0; count += vp->iov_len, vp++)
    if (ACE::send_n (fd, vp->iov_base, vp->iov_len) < 0)
      return -1;

  return count;
}
#endif /* ACE_NEEDS_WRITEV */

#if defined (ACE_NEEDS_READV)

/* "Fake" readv for sites without it. */
extern "C" int
readv (ACE_HANDLE fd, struct iovec *vp, int vpcount)
{
  ACE_TRACE ("::readv");

  int count;

  for (count = 0; --vpcount >= 0; count += vp->iov_len, vp++)
    if (ACE::recv_n (fd, vp->iov_base, vp->iov_len) < 0)
      return -1;

  return count;
}
#endif /* ACE_NEEDS_READV */

void 
ACE_OS::thr_exit (void *status)
{
  ACE_TRACE ("ACE_OS::thr_exit");
#if defined (ACE_HAS_THREADS)
#if defined (ACE_HAS_STHREADS)
  ::thr_exit (status);
#elif defined (ACE_HAS_PTHREADS)
  ::pthread_exit (status);
#elif defined (ACE_HAS_WTHREADS)
  {
    ACE_Guard<ACE_Thread_Mutex> mon (ACE_TSS_Key_Info::lock_);
    if (ACE_TSS_Key_Info::deletion_flag_)
      return;
  }
  ACE_OS::thr_destroy_tss ();
  ::ExitThread ((DWORD) status);
#endif /* ACE_HAS_STHREADS */
#else
  ;
#endif /* ACE_HAS_THREADS */		     
}
