/* tcpExample.h - header used by both TCP server and client examples */

/* defines */ 
#define SERVER_PORT_NUM 5001      /* server's port number for bind() */ 
#define SERVER_WORK_PRIORITY 100  /* priority of server's work task */ 
#define SERVER_STACK_SIZE 10000   /* stack size of server's work task */ 
#define SERVER_MAX_CONNECTIONS 4  /* max clients connected at a time */ 
#define REQUEST_MSG_SIZE 1024     /* max size of request message */ 
#define REPLY_MSG_SIZE 500        /* max size of reply message */ 

/* structure for requests from clients to server */ 
struct request { 
  int reply;                      /* TRUE = request reply from server */ 
  int msgLen;                     /* length of message text */ 
  char message[REQUEST_MSG_SIZE]; /* message buffer */ 
};



/* tcpClient.c - TCP client example */ 
/* DESCRIPTION This file contains the client-side of the VxWorks TCP example code. 
   The example code demonstrates the usage of several BSD 4.4-style socket routine calls. */ 

/* includes */ 
#include "vxWorks.h" 
#include "sockLib.h" 
#include "inetLib.h" 
#include "stdioLib.h" 
#include "strLib.h" 
#include "hostLib.h" 
#include "ioLib.h" 
#include "tcpExample.h" 

/**************************************************************************** 
* * tcpClient - send requests to server over a TCP socket 
* * This routine connects over a TCP socket to a server, and sends a 
* user-provided message to the server. Optionally, this routine 
* waits for the server's reply message. 
* * This routine may be invoked as follows: 
* -> tcpClient "remoteSystem" 
* Message to send: 
* Hello out there 
* Would you like a reply (Y or N): 
* y 
* value =0 =0x0 
* -> MESSAGE FROM SERVER: 
* Server received your message 
* * RETURNS: OK, or ERROR if the message could not be sent to the server. */ 

STATUS 
tcpClient ( char * serverName) 
{ 
  struct request myRequest; /* request to send to server */ 
  struct sockaddr_in serverAddr; /* server's socket address */ 
  char replyBuf[REPLY_MSG_SIZE]; /* buffer for reply */ 
  char reply; /* if TRUE, expect reply back */ 
  int sockAddrSize; /* size of socket address structure */ 
  int sFd; /* socket file descriptor */ 
  int mlen; /* length of message */

  /* create client's socket */ 
  if ((sFd = socket (AF_INET, SOCK_STREAM, 0)) == ERROR) { 
    perror ("socket"); 
    return (ERROR); 
  } 

  /* bind not required - port number is dynamic */ 
  /* build server socket address */ 
  sockAddrSize = sizeof (struct sockaddr_in); 
  bzero ((char *) &serverAddr, sockAddrSize); 
  serverAddr.sin_family = AF_INET; 
  serverAddr.sin_len = (u_char) sockAddrSize; 
  serverAddr.sin_port = htons (SERVER_PORT_NUM); 
  if (((serverAddr.sin_addr.s_addr = inet_addr (serverName)) == ERROR) && ((serverAddr.sin_addr.s_addr = hostGetByName (serverName)) == ERROR)) { 
    perror ("unknown server name"); 
    close (sFd); 
    return (ERROR); 
  } 

  /* connect to server */ 
  if (connect (sFd, (struct sockaddr *) &serverAddr, sockAddrSize) == ERROR) { 
    perror ("connect"); 
    close (sFd); 
    return (ERROR); 
  } 

  /* build request, prompting user for message */ 
  printf ("Message to send: \n"); 
  mlen = read (STD_IN, myRequest.message, REQUEST_MSG_SIZE); 
  myRequest.msgLen = mlen; 
  myRequest.message[mlen - 1] = '\0'; 
  printf ("Would you like a reply (Y or N): \n"); 
  read (STD_IN, &reply, 1); 
  switch (reply) { 
  case 'y': 
  case 'Y': 
    myRequest.reply = TRUE; 
    break; 
  default: 
    myRequest.reply = FALSE; 
    break; 
  } 

  /* send request to server */ 
  if (write (sFd, (char *) &myRequest, sizeof (myRequest)) == ERROR) { 
    perror ("write"); 
    close (sFd); 
    return (ERROR);
  } 

  if (myRequest.reply) /* if expecting reply, read and display it */ { 
    if (read (sFd, replyBuf, REPLY_MSG_SIZE) < 0) { 
      perror ("read"); 
      close (sFd); 
      return (ERROR); 
    } 
    printf ("MESSAGE FROM SERVER:\n%s\n", replyBuf); 
  } 
  close (sFd); 
  return (OK); 
} 


/* tcpServer.c - TCP server example */ 
/* DESCRIPTION This file contains the server-side of the VxWorks TCP example code. 
   The example code demonstrates the usage of several BSD 4.4-style socket routine calls. */ 

/* includes */ 
#include "vxWorks.h" 
#include "sockLib.h" 
#include "inetLib.h" 
#include "taskLib.h" 
#include "stdioLib.h" 
#include "strLib.h" 
#include "ioLib.h" 
#include "fioLib.h" 
#include "tcpExample.h" 

/* function declarations */ 
VOID tcpServerWorkTask (int sFd, char * address, u_short port); 

/**************************************************************************** 
* * tcpServer - accept and process requests over a TCP socket 
* * This routine creates a TCP socket, and accepts connections over the socket 
* from clients. Each client connection is handled by spawning a separate 
* task to handle client requests. 
* * This routine may be invoked as follows: 
* -> sp tcpServer
* task spawned: id = 0x3a6f1c, name = t1 
* value = 3829532 = 0x3a6f1c 
* -> MESSAGE FROM CLIENT (Internet Address 150.12.0.10, port 1027): 
* Hello out there 
* * RETURNS: Never, or ERROR if a resources could not be allocated. */ 

STATUS 
tcpServer (void) 
{ 
  struct sockaddr_in serverAddr; 
  struct sockaddr_in clientAddr; 
  int sockAddrSize;              /* size of socket address structure */ 
  int sFd;                       /* socket file descriptor */ 
  int newFd;                     /* socket descriptor from accept */ 
  int ix = 0;                    /* counter for work task names */ 
  char workName[16];             /* name of work task */ 

  /* set up the local address */ 
  sockAddrSize = sizeof (struct sockaddr_in); 
  bzero ((char *) &serverAddr, sockAddrSize); 
  serverAddr.sin_family = AF_INET; 
  serverAddr.sin_len = (u_char) sockAddrSize; 
  serverAddr.sin_port = htons (SERVER_PORT_NUM); 
  serverAddr.sin_addr.s_addr = htonl (INADDR_ANY); /* create a TCP-based socket */ 

  if ((sFd = socket (AF_INET, SOCK_STREAM, 0)) == ERROR) { 
    perror ("socket"); 
    return (ERROR); 
  } 

  /* bind socket to local address */ 
  if (bind (sFd, (struct sockaddr *) &serverAddr, sockAddrSize) == ERROR) { 
    perror ("bind"); 
    close (sFd); 
    return (ERROR); 
  } 

  /* create queue for client connection requests */ 
  if (listen (sFd, SERVER_MAX_CONNECTIONS) == ERROR) { 
    perror ("listen"); 
    close (sFd); 
    return (ERROR); 
  }

  /* accept new connect requests and spawn tasks to process them */ 

  FOREVER { 
    if ((newFd = accept (sFd, (struct sockaddr *) &clientAddr, &sockAddrSize)) == ERROR) { 
      perror ("accept"); 
      close (sFd); 
      return (ERROR); 
    } 

    sprintf (workName, "tTcpWork%d", ix++);

    if (taskSpawn(workName, SERVER_WORK_PRIORITY, 0, SERVER_STACK_SIZE, (FUNCPTR) tcpServerWorkTask, 
		  newFd, (int) inet_ntoa (clientAddr.sin_addr), ntohs (clien tAddr.sin_port), 
		  0, 0, 0, 0, 0, 0, 0) == ERROR) 
      { 

	/* if taskSpawn fails, close fd and return to top of loop */ 
	perror ("taskSpawn"); 
	close (newFd); 
      } 
  } 
} 

/**************************************************************************** 
* * tcpServerWorkTask - process client requests 
* * This routine reads from the server's socket, and processes client 
* requests. If the client requests a reply message, this routine 
* will send a reply to the client. 
* * RETURNS: N/A. */ 

VOID 
tcpServerWorkTask ( int sFd, char * address, u_short port) 
{ 
  struct request clientRequest;                            /* request/message from client */ 
  int nRead;                                               /* number of bytes read */ 
  static char replyMsg[] = "Server received your message"; /* read client request, display message */ 

  while ((nRead = fioRead (sFd, (char *) &clientRequest, sizeof (clientRequest))) > 0) { 
    printf ("MESSAGE FROM CLIENT (Internet Address %s, port %d):\n%s\n", address, port, clientRequest.message);
    free (address); /* free malloc from inet_ntoa() */ 

    if (clientRequest.reply) 
      if (write (sFd, replyMsg, sizeof (replyMsg)) == ERROR) 
	perror ("write"); 

  } 

  if (nRead == ERROR) /* error from read() */ 
    perror ("read"); 

  close (sFd); /* close server socket connection */ 
}
