/*************************************************************************
 *
 *   vmeUserLib.c  - Library of routines for the user to write for 
 *                   Aysyncronous readout and buffering of events
 *                   from VME. Use Universe Chip DMA Engine 
 *
 *
 *
 */


#include "vmeLib.h"
#include "../sfi/dmaPList.h"

#ifdef INCLUDE_DCPU
#include "../dualCPU/dcpu.h"
#endif

/* Define Interrupt source and address */
#define TIR_ADDR 0x0ed0
#define IRQ_SOURCE TIR_SOURCE


#define MAX_NUM_EVENTS    400
#define MAX_SIZE_EVENTS   2048      /* Size in Bytes */
#define MAX_DMA_WORDS     100       /* Must be less than MAX_SIZE_EVENTS/4 */


/* 0: External Mode   1:Trigger Supervisor Mode */
#define TS_MODE  0

/* Input and Output Partions for VME Readout */
volatile ROL_MEM_ID vmeIN, vmeOUT;

/* Dual CPU variables needed to access */
#ifdef INCLUDE_DCPU
extern DCPU_CONTROL *dcpuCntlPtr;
extern DCPU_CONTROL *dcpuRemCntlPtr;
extern int dcpuReady;
extern int dcpuSize;   /* Buffer size in Bytes */
#endif

/* system clock */
extern int vxTicks;


/* DMA library declarations */
extern void sysVmeDmaShow (void);    
extern int sysVmeDmaDone(int, int);
extern int sysVmeDmaInit(int);
extern int sysVmeDmaGet(unsigned int, unsigned int *);
extern int sysVmeDmaSet(unsigned int, unsigned int);
extern int sysVmeDmaSend(unsigned int, unsigned int, int, int);
extern int sysVmeDmaDone(int, int);




/* Initialize Memory partitions for managing events */
void
vmeUserInit()
{
  unsigned int res, laddr;

  /* Setup Universe DMA Engine for transfers, and establish DMA address offsets */
  /*initialize (no interrupts (1)) */
  sysVmeDmaInit(1); 
  /* Set for 32bit (0) or 64bit (1) PCI transfers */
  sysVmeDmaSet(4,1);
  /* A24 (1) or A32 (2) VME Slave */
  sysVmeDmaSet(11,2);
  /* BLK32 (4) or MBLK64 (5) VME transfers */
  sysVmeDmaSet(12,4);

  /* Setup Buffer memory to store events */
  dmaPFreeAll();
  vmeIN  = dmaPCreate("vmeIN",MAX_SIZE_EVENTS,MAX_NUM_EVENTS,0);
  vmeOUT = dmaPCreate("vmeOUT",0,0,0);

  dmaPStatsAll();
}


void
vmeUserDownload()
{

  /* Reinitialize the Buffer memory */
  dmaPReInitAll();

  /* Initialize VME Interrupt interface - use defaults */
  vmeIntInit(TIR_ADDR,0,0,0);

  /* Connect User Trigger Routine */
  vmeIntConnect(vmeUserTrigger,0);


  printf("vmeUserDownload: User Download Executed\n");

}

void
vmeUserPrestart()
{
  unsigned int adc_id;


  /* PRogram/Init VME Modules Here */




  /* Initialize VME Interrupt variables */
  vmeIntClearCount();

  printf("vmeUserPrestart: User Prestart Executed\n");

}

void
vmeUserGo()
{
  /* Enable Interrupts */
  vmeIntEnable(IRQ_SOURCE,TS_MODE);
}

void
vmeUserEnd()
{
  int status, count;

  /* Disable Interrupts */
  vmeIntDisable(IRQ_SOURCE,TS_MODE);

  /* Check on last Transfer */
  if(vmeIntCount > 0) {
    status = vmeDmaStatus();
    if(status < 0) {
      logMsg("ERROR: in Last Transfer Event NUMBER %d, status = 0x%x\n",vmeIntCount,status,0,0,0,0);
      *dma_dabufp++ = 0xda000bad;
    }else{
      count = (MAX_DMA_WORDS<<2) - status;
      dma_dabufp += (count>>2);
    }
    PUTEVENT(vmeOUT);
  }

  printf("vmeUserEnd: Ended after %d events\n",vmeIntCount);
  
}

void
vmeUserTrigger(int arg)
{
  int ii, status, dma, count;
  unsigned int datascan, tirval, vme_addr;

  /*vmeTirPulse(1,3);*/

  /* check on Last Transfer */
  if (vmeIntCount > 1) {
    status = vmeDmaStatus();
    if(status < 0) {
      logMsg("ERROR: in Last Transfer Event NUMBER %d, status = 0x%x\n",vmeIntCount,status,0,0,0,0);
      *dma_dabufp++ = 0xda000bad;
    }else{
      count = (MAX_DMA_WORDS<<2) - status;
      dma_dabufp += (count>>2);
    }
    PUTEVENT(vmeOUT);
  }

  /*vmeTirPulse(0,2);*/

  /*Get TIR data (event type and Sync) check for data and start transfer*/
  tirval = vmeTrigType(IRQ_SOURCE, TS_MODE);

  /* Check for valid data here */
  datascan = 1;

  /* grap a buffer from the queue */
  GETEVENT(vmeIN,vmeIntCount);
 
  /* set VME address for boards to readout */
  vme_addr = 0x08000000;

  if(datascan) {
    *dma_dabufp++ = 0xda0000da;   /* Place holder for the length */
    *dma_dabufp++ = tirval;       /* Send TIR port info for Event type and Sync info */
    vmeDmaStart((unsigned int) dma_dabufp,vme_addr,MAX_DMA_WORDS);
  }else{
    *dma_dabufp++ = 0xda0000da;   /* Place holder for the length */
    *dma_dabufp++ = tirval;       /* Send TIR port info for Event type and Sync info */
  }

  /*vmeTirPulse(0,0);*/

  if(dmaPEmpty(vmeIN)) {
    vmeNeedAck = 1;
    return;
  } else {
    vmeIntAck(IRQ_SOURCE,TS_MODE);
  }
}

void
vmeUserStatus()
{

  printf("VME DMA Status:\n\n");

  dmaPStatsAll();

  printf("\n");
  if(vmeIntStatus())
    printf(" DAQ: Active\n");
  else
    printf(" DAQ: Idle\n");
    
  printf(" Total Event Count = %d\n",vmeIntCount);

}

int
vmeUserGetCount()
{

  if(vmeOUT) {
    return(vmeOUT->list.c);
  }else{
    return(0);
  }      
}



void
vmeUserTest()
{
  int lock_key;
  DANODE *outEvent;
  
  while(1) {
    if(dmaPEmpty(vmeOUT))
      continue;
    
    /* get event - then free it */
    lock_key = intLock();
    outEvent = dmaPGetItem(vmeOUT);

    dmaPFreeItem(outEvent);
    intUnlock(lock_key);
  }


}

#ifdef INCLUDE_DCPU

void
dcpuUserPoll(int bufsize, int to)
{
  int lock_key;
  int ii, jj, mt, len;
  int csr, cmd, state, flush=0;
  int tcount, ecount, eMax=0;
  DANODE *outEvent;

  if (bufsize <= 0) 
    bufsize = (dcpuSize>>3);  /* Bufsize default in words - half the total buffer space */

  if (to <= 0) 
    to = 2*(sysClkRateGet());  /* Time out default in ticks - every 2 seconds */

  /* Wait in IDLE loop until DCPU library has been initialized */
  while(dcpuReady != 1) {
    taskDelay(60);
  }


  /* Init Buffer memory control words */
  dcpuCntlPtr->nevents = 0;
  dcpuCntlPtr->nwords = 0;
  dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);
  dcpuCntlPtr->clear = 0;
  dcpuCntlPtr->full  = 0;
  dcpuCntlPtr->first = 0;
  dcpuCntlPtr->last  = 0;
  ii=0;

  state = DC_IDLE;  /*Initial State is Idle */

  /* Enter control loop - check for commands from Destination CPU */
  printf("dcpuUserPoll: Enabled...\n");
  tcount = vxTicks;

  while(1) {
    cmd = (dcpuCntlPtr->csr)&DC_CMD_MASK;

    switch (cmd) {
      case DC_DOWNLOAD:
	vmeUserDownload();
	eMax = dcpuCntlPtr->user[0]; /* Get Max Event count */
	printf("dcpuUserPoll: Max Event count = %d, Max buffer size %d words\n",
	       eMax,bufsize);
	state = DC_DOWNLOADED;
	dcpuCntlPtr->csr = state;
	break;
      case DC_PRESTART:
	vmeUserPrestart();
	dcpuCntlPtr->nevents = 0;
	dcpuCntlPtr->nwords = 0;
	dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);
	dcpuCntlPtr->clear = 0;
	dcpuCntlPtr->full  = 0;
	dcpuCntlPtr->first = 0;
	dcpuCntlPtr->last  = 0;
	state = DC_PRESTARTED;
	dcpuCntlPtr->csr = state;
        break;
      case DC_GO:
	vmeUserGo();
	flush = 0;
	tcount = vxTicks;
	state = DC_ACTIVE;
	dcpuCntlPtr->csr = state;
        break;
      case DC_END:
	vmeUserEnd();
	flush = 1;
	dcpuCntlPtr->nevents = vmeIntCount; /* Set final event count */
	dcpuCntlPtr->csr = DC_ACTIVE;       /*Until we flush all events */
	dcpuRemCntlPtr->csr = DC_ENDING;    /* Notify Dest Board Triggers are disabled */
        break;
      default:
	jj=0;
	mt=0;
	while((state == DC_ACTIVE)&&(jj++<MAX_NUM_EVENTS)) {
	  if(dmaPEmpty(vmeOUT))
	    mt=1;
    
	  if(((dcpuCntlPtr->full) == 0)&&((dcpuCntlPtr->clear)==0)) {
	    if(mt==0) {
	      /* get event off the list*/
	      lock_key = intLock();
	      outEvent = dmaPGetItem(vmeOUT);
	      if(outEvent != 0) {
		len = outEvent->length;

		/* Copy event to dcpuMemory */
		dcpuCntlPtr->nevents = vmeIntCount;
		if((dcpuCntlPtr->nwords)==0) /*Empty Buffer*/
		  dcpuCntlPtr->first = outEvent->nevent;
		dcpuCntlPtr->last = outEvent->nevent;
	    
      
		*(dcpuCntlPtr->local)++ = len;
		for(ii=0;ii<len;ii++) {
		  *(dcpuCntlPtr->local)++ = outEvent->data[ii];
		}
		dcpuCntlPtr->nwords += (len+1); 
		ecount = dcpuCntlPtr->last - dcpuCntlPtr->first + 1;

		/* buffer is full, too many events, or there is a timeout then flag to send */
		if((dcpuCntlPtr->nwords > bufsize)||(ecount >= eMax)||(vxTicks > (tcount + to))) {
		  dcpuCntlPtr->clear = 0;
		  dcpuCntlPtr->full  = 1;
		}

	      /* Free the event and renable interrupts */
		dmaPFreeItem(outEvent);
		intUnlock(lock_key);
	    
		if(vmeNeedAck > 0) {
		  vmeIntAck(IRQ_SOURCE,TS_MODE);
		}
	      }else{
		intUnlock(lock_key);
		logMsg("Error: Could not get Event from List\n",0,0,0,0,0,0);
	      }
	    }

	  } else {

	    /*We are full - check to see if the transfer is complete */
	    if(dcpuCntlPtr->clear) {
      	      dcpuCntlPtr->nwords = 0;
	      dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);
	      dcpuCntlPtr->clear = 0;
	      dcpuCntlPtr->full  = 0;
	      tcount = vxTicks;
	    }
	  }
	  
	  /* check to see if we are Ending - flush what we have
	     Two Conditions - Ending and Last event in buffer, */
	  if((flush)&&(vmeIntCount == dcpuCntlPtr->last)) {
	    dcpuCntlPtr->full  = 1;
	    while(dcpuCntlPtr->clear == 0) {
	      sysUsDelay(50);
	    }

	    dcpuCntlPtr->nwords = 0;
	    dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);
	    dcpuCntlPtr->clear = 0;
	    dcpuCntlPtr->full  = 0;

	    flush=0;
	    state = DC_ENDED;
	    dcpuCntlPtr->csr = state;
	  }
	} /* End active While loop */
	
    } /* End switch */
  }
}

#endif
