/* Module: tsIntUtil.c
 *
 * Description: Trigger Supervisor Program
 *              Utility Library - Interrupt Specific functions. 
 *
 * Author:
 *	David Abbott
 *	CEBAF Data Acquisition Group
 *
 * Revision History:
 *	  Revision 1.0  2008/20/3   abbottd
 */

#include <intLib.h>
#include <iv.h>


/* Define default Interrupt Level and vector for the TS source - 
   Note that the Interrupt level is assigned by a hardware switch but is not
   determinable via software */
#define TS_INT_LEVEL    5
#define TS_INT_VEC      0xec


LOCAL VOIDFUNCPTR tsIntRoutine  = NULL;	            /* user interrupt service routine */
LOCAL int         tsIntArg      = 0;	            /* arg to user routine */
LOCAL UINT32      tsIntLevel    = TS_INT_LEVEL;     /* VME Interrupt Level 1-7 valid*/
LOCAL UINT32      tsIntVec      = TS_INT_VEC;       /* default interrupt Vector */
LOCAL int         tsDoAck       = 1;                /* Default - Acknowledge trigger */

/* Globals */
unsigned int      tsIntPoll     = 0;
unsigned int      tsIntCount    = 0;


/*******************************************************************************
*
* tsInt - default interrupt handler
*
* This rountine handles the ts interrupt.  A user routine is
* called, if one was specified by tsIntConnect().
*
* RETURNS: N/A
*
* SEE ALSO: tsIntConnect()
*/

static void 
tsInt (void)
{
  tsIntCount++;
  
  if (tsIntRoutine != NULL)	/* call user routine */
    (*tsIntRoutine) (tsIntArg);


  /* Acknowledge trigger */
  if(tsDoAck)
    tsP->lrocBufStatus = 0x100;
}

/* Example User routine */
void
tsIntUser(int arg)
{
  int tsdata;
  unsigned int sync, latefail, code;

  tsdata = tsIntTest();
  sync     = tsdata&TS_LROC_SYNC;
  latefail = (tsdata&TS_LROC_LATEFAIL)>>1;
  code     = (tsdata&TS_LROC_CODE_MASK)>>2;
  logMsg("tsIntUser: Got Trigger (0x%x): Event Type = %d (sync = %d, latefail = %d)\n",
	 tsdata,code,sync,latefail,0,0);

  return;
}


/*******************************************************************************
*
* tsIntConnect - connect a user routine to the TS interrupt
*
* This routine specifies the user interrupt routine to be called at each
* interrupt
*
* RETURNS: OK, or ERROR .
*/

int
tsIntConnect (unsigned int vector, VOIDFUNCPTR routine, int arg, int poll)
{

  if(tsP == NULL) {
    printf("tsIntConnect: ERROR: TS not initialized\n");
    return(-1);
  }
  if(TSENABLED) {
    printf("tsIntConnect: ERROR : Cannot connect interrupts while TS is enabled\n");
    return(-1);
  }


  /* Disconnect any current interrupts */
  if((intDisconnect(tsIntVec) !=0))
     printf("tsIntConnect: Error disconnecting Interrupt\n");

  tsIntPoll = poll;
  tsIntCount = 0;
  tsDoAck = 1;

  /* Set Vector and Level */
  if((vector < 255)&&(vector > 64)) {
    tsIntVec = vector;
  }else{
    tsIntVec = TS_INT_VEC;
  }
  tsIntLevel = TS_INT_LEVEL;
  printf("tsIntConnect: INFO: Interrupt Vector = 0x%x  Level = %d\n",tsIntVec,tsIntLevel);


  if (tsIntPoll) {
    tsP->intVec = tsIntVec;
  }else{
    tsP->intVec = tsIntVec;
    intConnect(INUM_TO_IVEC(tsIntVec),tsInt,0);
  }

  if(routine) {
    tsIntRoutine = routine;
    tsIntArg = arg;
  }else{
    tsIntRoutine = NULL;
    tsIntArg = 0;
  }

  return(0);
}


void 
tsIntDisconnect()
{
  if(tsP == NULL) {
    printf("tsIntDisconnect: ERROR: TS not initialized\n");
    return;
  }

  if(TSENABLED) {
    printf("tsIntDisconnect: ERROR: TS (GO) is Enabled - Call tsStop() first\n");
    return;
  }

  /* Reset TS Branch 5 Interrupts*/
  tsP->csr2 &= ~(0x1800);

  /* Disconnect any current interrupts */
  sysIntDisable(tsIntLevel);
  if((intDisconnect(tsIntVec) !=0))
     printf("tsIntConnect: Error disconnecting Interrupt (vector=%d)\n",tsIntVec);

  return;
}


int
tsIntEnable(int iflag)
{
  if(tsP == NULL) {
    printf("tsIntEnable: ERROR: TS not initialized\n");
    return(-1);
  }
  if(TSENABLED) {
    printf("tsIntEnable: ERROR : Cannot Enable Interrupts while TS (GO) is enabled\n");
    return(-1);
  }

  if(iflag)
    tsIntCount = 0;

  tsDoAck      = 1;

  if (tsIntPoll) {
    tsP->csr2 |= 0x1000;
  }else{    
    sysIntEnable(tsIntLevel);
    tsP->csr2 |= 0x1800;
  }

  return(OK);

}

void 
tsIntDisable()
{

  if(tsP == NULL) {
    printf("tsIntDisable: ERROR: TS not initialized\n");
    return;
  }
  if(TSENABLED) {
    printf("tsIntDisable: ERROR : Cannot Disable Interrupts while TS (GO) is enabled\n");
    return;
  }

  tsP->csr2 &= ~(0x1800);
}

void 
tsIntAck()
{
  if(tsP == NULL) {
    logMsg("tsIntAck: ERROR: TS not initialized\n",0,0,0,0,0,0);
    return;
  }
  
  tsDoAck = 1;
  tsP->lrocBufStatus = 0x100;
}


int 
tsIntTest()
{
  unsigned int   lval=0;
  int   sval=0;

  lval = tsP->lrocBufStatus;
  if (lval == 0xffffffff)
    return (-1);

  if(lval&TS_LROC_STROBE) {
     sval = (lval&TS_LROC_STROBE) | ((tsP->lrocData)&TS_LROC_DATA_MASK);
     return (sval);
   } else {
     return (0);
   }
}
