/*
 * Library for Dual CPU system
 * 
 *      The Base (VME) board is the source CPU
 *      and the PMC board is the destination CPU
 *
 */
#ifdef VXWORKS
#include <vxWorks.h>
#include <stdio.h>
#include <string.h>
#include <memLib.h>
#include <cacheLib.h>
#include <semLib.h>
#include <taskLib.h>
#endif

#include <types.h>
#include "dcpu.h"


/* global data */

unsigned int dcpuEventCnt=0;
int dcpuReady = 0;
int dcpuSource = 0;
int dcpuDest   = 0;
int dcpuSize   = DCPU_MEM_SIZE;
volatile DCPU_CONTROL *dcpuCntlPtr;
volatile DCPU_CONTROL *dcpuSourceCntlPtr=0;
volatile DCPU_CONTROL *dcpuDestCntlPtr=0;
volatile DCPU_CONTROL *dcpuRemCntlPtr=0;

SEM_ID dcpuSem=0;


/* Init Routine - Setup noncachable control space on each board for
   communication

   id 
    0 : source board       - sets up first
    1 : destination board  
   
*/

void
dcpuInit(int id, int size)
{
  int wait=0;
  int ret =0;
  unsigned int gval=0;
  unsigned int rval;

  if(dcpuReady) {
    dcpuReady = 0;
    cacheDmaFree((char *)dcpuCntlPtr);
    dcpuCntlPtr = (DCPU_CONTROL *)0;
    dcpuRemCntlPtr= (DCPU_CONTROL *)0;
    dcpuSource = 0;
    dcpuDest   = 0;
    dcpuEventCnt = 0;
  }
  if(size<=0) size = DCPU_MEM_SIZE;
  dcpuSize = size;

  if(id) {  /* We are a destination board - wait for address from source */

    /* allocate Local Memory */
    dcpuCntlPtr = (DCPU_CONTROL *)cacheDmaMalloc(size);
    dcpuDestCntlPtr = dcpuCntlPtr;
    bzero((char *)dcpuCntlPtr,size);


    /* Wait for Source board request */
    wait = 1;
    while(wait) {
      gval = usrReadGPR();
      if((gval == DC_INIT_SRC_ID)) {
	wait=0;
      }else{
	taskDelay(60);
      }
    }

    /* Send the Reply to Source to proceed */
    usrWriteGPR(DC_INIT_DEST_ID);

    wait=1;
    while(wait) {
      gval = usrReadGPR();
      if(gval != DC_INIT_DEST_ID) {
	wait=0;
	dcpuSourceCntlPtr = (DCPU_CONTROL *)(gval);
	dcpuRemCntlPtr    = (DCPU_CONTROL *)(PCI_SOURCE_MEM_OFFSET+gval);
      }else{
	taskDelay(60);
      }
    }
    /* Send back the Destination Mem Address */
    usrWriteGPR(dcpuCntlPtr);
    dcpuDest = 1;
    dcpuSource = 0;

    printf("dcpuInit: INFO: Destination CPU\n");
    printf("dcpuInit:   Memory Size   = 0x%x bytes\n",dcpuSize);
    printf("dcpuInit:   Local  Memory = 0x%08x\n",(int)dcpuCntlPtr);
    printf("dcpuInit:   Source Memory = 0x%08x\n",(int)dcpuSourceCntlPtr);
    printf("dcpuInit:   Remote Access = 0x%08x\n",(int)dcpuRemCntlPtr);



  } else { /* source board - go ahead and setup */

    /* Send the SRC ID to tell destination board it is ready */
    usrWriteGPR(DC_INIT_SRC_ID);
    
    /* allocate memory */
    dcpuCntlPtr = (DCPU_CONTROL *)cacheDmaMalloc(size);
    dcpuSourceCntlPtr = dcpuCntlPtr;
    bzero((char *)dcpuCntlPtr,2048);

    /* Wait for Destination board reply */
    wait = 1;
    while(wait) {
      gval = usrReadGPR();
      if((gval == DC_INIT_DEST_ID)) {
	wait=0;
      }else{
	taskDelay(60);
      }
    }
      
    /* Send Local Memory structure pointer */
    usrWriteGPR(dcpuCntlPtr);

    wait = 1;
    while(wait) {
      gval = usrReadGPR();
      if((gval != (unsigned int)dcpuCntlPtr)) {
	wait=0;
	dcpuDestCntlPtr = (DCPU_CONTROL *)(gval);
	dcpuRemCntlPtr  = (DCPU_CONTROL *)(PCI_DEST_MEM_OFFSET+gval);
      }else{
	taskDelay(10);
      }
    }
    dcpuDest = 0;
    dcpuSource = 1;


    printf("dcpuInit: INFO: Source CPU\n");
    printf("dcpuInit:   Memory Size   = 0x%x bytes\n",dcpuSize);
    printf("dcpuInit:   Local Memory      = 0x%08x\n",(int)dcpuCntlPtr);
    printf("dcpuInit:   Destinaton Memory = 0x%08x\n",(int)dcpuDestCntlPtr);
    printf("dcpuInit:   Remote Access     = 0x%08x\n",(int)dcpuRemCntlPtr);

    /* Clear the GPR */
    usrWriteGPR(0);

  }


  /* check that remote memory is addressable */
  rval = dcpuRemCntlPtr->full;
  if(rval == 0xffffffff) {
    printf("dcpuInit: ERROR: Remote memory not addressable.\n");
    dcpuReady=-1;
    return;
  }else{
    /* Init was successful */
    dcpuReady = 1;
  }

}

int
dcpuStatus(int iflag)
{

  if(dcpuReady) {

    if(iflag) {
      if(dcpuSource)
	printf("DCPU: Source CPU\n");
      if(dcpuDest)
	printf("DCPU: Destination CPU\n");
      
      printf("dcpuStatus: Memory Size    = 0x%x bytes\n",dcpuSize);
      printf("            Local  Memory  = 0x%08x\n",(int)dcpuCntlPtr);
      printf("            Remote Memory  = 0x%08x\n",(int)dcpuRemCntlPtr);
      printf("            Event Count    = %d\n"),dcpuEventCnt;
    }

  }

  return(dcpuReady);

}


void
dcpuGetTest()
{

  int state=DC_ACTIVE;
  int ii, res, last=0;
  int words, exit=0;
  int total=0;
  unsigned int data[2048];

  if(dcpuReady == 0) {
    printf("dcpuGetTest: Error dcpu not ready\n");
    return;
  }

  /* Init the DCPU Control structure */
  dcpuCntlPtr->full   = 0;
  dcpuCntlPtr->clear  = 0;
  dcpuCntlPtr->nwords = 0;
  dcpuCntlPtr->first  = 0;
  dcpuCntlPtr->last   = 0;
  dcpuEventCnt = 0;

  while(state == DC_ACTIVE) {
    if(dcpuPoll()) {
      words = *dcpuCntlPtr->local++;
	
      for(ii=0;ii<words;ii++) {
	  data[ii] = *dcpuCntlPtr->local++;
      }

      if(dcpuEventCnt > dcpuCntlPtr->last) {
	printf("Error: Event count %d > than events trasfered %d\n",dcpuEventCnt,dcpuCntlPtr->last);
	return;
      }

    } else {

      res = (dcpuCntlPtr->csr)&DC_STATE_MASK;
      if(res == DC_ENDING) {
	total = dcpuRemCntlPtr->nevents;
	if(total< dcpuCntlPtr->last) {
	  printf("Ending - waiting for %d events\n",total);
	  continue;
	}else{
	  printf("Exiting after %d events\n",dcpuEventCnt);
	  dcpuCntlPtr->csr = DC_IDLE;
	  return;
	}
      }

    }

  }

}


/* Routine to poll for data from source board and transfer it to local memory. If 
   there is already transfered data available then simply bump pointer to next event */
int
dcpuPoll()
{

  int state=DC_ACTIVE;
  int csr, res;
  int words, exit=0;
  int size;
  unsigned int lastword;

  static int cur_event;

  /* Do we still have events to process */
  if(dcpuCntlPtr->nevents > 0) {
    dcpuCntlPtr->nevents--;
    cur_event++;
    dcpuEventCnt++;
    return(1);
  }

  /* No events - Check for a full buffer to transfer */
  if(dcpuRemCntlPtr->full) {
    words = dcpuRemCntlPtr->nwords;

#ifndef DCPU_USE_DMA
    /* Transfer data - Programmed IO*/
    bcopyLongs((char *)&(dcpuRemCntlPtr->data[0]),(char *)&(dcpuCntlPtr->data[0]),words);
#else	  
    /* DMA */
    lastword = dcpuRemCntlPtr->data[(words-1)];
    dcpuCntlPtr->data[(words-1)] = 0;

    usrPci2MemDmaStart(0,
                      (char *)&dcpuSourceCntlPtr->data[0],
                      (char *)&dcpuDestCntlPtr->data[0],
		      (words<<2));
    while(lastword != dcpuCntlPtr->data[(words-1)]) {
      continue;
    }
#endif

    dcpuCntlPtr->nwords = words;
    dcpuCntlPtr->first  = dcpuRemCntlPtr->first;
    dcpuCntlPtr->last  = dcpuRemCntlPtr->last;
    dcpuCntlPtr->nevents = (dcpuCntlPtr->last) - (dcpuCntlPtr->first);
    cur_event = dcpuCntlPtr->first;
    dcpuCntlPtr->local = &(dcpuCntlPtr->data[0]);

    dcpuRemCntlPtr->clear = 1; /* Trigger a clear on Remote CPU */
    dcpuRemCntlPtr->full = 0;
    dcpuEventCnt++;
    return(1);
  }

  return(0);

}



void
dcpuSendDownload(int usrArg)
{
  int state;


  if(dcpuReady == 0) {
    logMsg("dcpuSendDownload: Error dcpu not ready\n",0,0,0,0,0,0);
    return;
  }

  state = (dcpuCntlPtr->csr)&DC_STATE_MASK;

  if(state != DC_ACTIVE) {
    dcpuRemCntlPtr->user[0] = usrArg;
    dcpuRemCntlPtr->csr = DC_DOWNLOAD;
    dcpuCntlPtr->csr = DC_DOWNLOADED;
  } else {
    logMsg("Cannot Download - current state is ACTIVE\n",0,0,0,0,0);
  }
  
}

void
dcpuSendPrestart()
{
  int state;


  if(dcpuReady == 0) {
    logMsg("dcpuSendPrestart: Error dcpu not ready\n",0,0,0,0,0,0);
    return;
  }

  state = (dcpuCntlPtr->csr)&DC_STATE_MASK;


  if(state != DC_ACTIVE) {
    dcpuCntlPtr->full    = 0;
    dcpuCntlPtr->clear   = 0;
    dcpuCntlPtr->nevents = 0;
    dcpuCntlPtr->nwords  = 0;
    dcpuCntlPtr->first   = 0;
    dcpuCntlPtr->last    = 0;
    dcpuEventCnt = 0;
    dcpuRemCntlPtr->csr = DC_PRESTART;   /* Send Prestart request */
    dcpuCntlPtr->csr = DC_PRESTARTED;    
  } else {
    logMsg("Cannot Prestart - current state is ACTIVE\n",0,0,0,0,0);
  }
  
}

void
dcpuSendGo()
{
  int state;


  if(dcpuReady == 0) {
    logMsg("dcpuSendGo: Error dcpu not ready\n",0,0,0,0,0,0);
    return;
  }

  state = (dcpuCntlPtr->csr)&DC_STATE_MASK;

  if(state != DC_ACTIVE) {
    dcpuRemCntlPtr->csr = DC_GO;
    dcpuCntlPtr->csr = DC_ACTIVE;
  } else {
    logMsg("Cannot Go - current state is ACTIVE\n",0,0,0,0,0);
  }
  
}

void
dcpuSendEnd()
{
  int state;

  if(dcpuReady == 0) {
    logMsg("dcpuSendEnd: Error dcpu not ready\n",0,0,0,0,0,0);
    return;
  }

  state = (dcpuCntlPtr->csr)&DC_STATE_MASK;

  /* Send End request */
  dcpuRemCntlPtr->csr = DC_END;

  sysUsDelay(50);

  /* Wait for ending state */
  while(dcpuCntlPtr->csr != DC_ENDING) {
    sysUsDelay(50);
  }

}
