/******************************************************************************
*
*  c-ramsLib.c  -  Driver library for CAEN C-RAMS system
*
*  Author: David Abbott 
*          Jefferson Lab Data Acquisition Group
*          January 2000
*
*/

#include "vxWorks.h"
#include "stdio.h"
#include "string.h"
#include "intLib.h"
#include "iv.h"
#include "logLib.h"
#include "taskLib.h"
#include "vxLib.h"

/* Include definitions */
#include "c-rams.h"

IMPORT  STATUS sysBusToLocalAdrs(int, char *, char **);
IMPORT  STATUS intDisconnect(int);
IMPORT  STATUS sysIntEnable(int);

IMPORT  STATUS sysVmeDmaSend(UINT32, UINT32, int, BOOL);
IMPORT  int sysVmeDmaDone(int);

/* Define global structure pointers for board access */
volatile struct c_rams_s_struct *crams_s;
volatile struct c_rams_struct *crams[10];
int nCRAMS = 0; 


/* Use function to program C-RAMS Sequencer T Registers 
   Make sure that the Correct value gets programed in */
int
cramsTP(int reg, unsigned short val)
{
  int ii=0;
  unsigned short retVal=0;
  
  crams_s->T[(reg-1)] = val;
  taskDelay(1);
  retVal = crams_s->T[(reg-1)];

  while ((retVal != val) && (ii<100)) {
    taskDelay(5);
    retVal = 0;
    crams_s->T[(reg-1)] = val;
    taskDelay(1);
    retVal = crams_s->T[(reg-1)];
    ii++;
  }

  if(retVal != val)
    return(-1);
  else
    return(0);
   
}

/*******************************************************************************
*
* cramsInit - C-RAMS Sequencer and ADC boards
*
*
* RETURNS: OK, or ERROR if the address is invalid.
*/

STATUS 
cramsInit (UINT32 saddr, UINT32 addr_inc, int ncrams)
{
  int res, ii;
  short rdata;
  unsigned long laddr, raddr;
  
  
  /* Check for valid address */
  if(saddr==0) {
    printf("cramsInit: ERROR: Must specify VME(A24) or CPU(A32) address for CRAM Sequencer\n");
    printf("cramsInit: cramsInit <C-RAMS seq addr>,<C-RAMS addr increment>,<# of C-RAMS>\n");
    return(ERROR);
  }else if (saddr > 0x00ffffff) {
    crams_s = (struct c_rams_s_struct *)saddr;
  }else{
    /* determine the  A32 address */
    res = sysBusToLocalAdrs(0x39,(char *)saddr,(char **)&laddr);
    if (res != 0) {
      printf("cramsInit: ERROR in sysBusToLocalAdrs(0x39,0x%x,&laddr) \n",saddr);
      return(ERROR);
    } else {
      crams_s = (struct c_rams_s_struct *)laddr;
    }	
  }
  /* Check if board really exists at the specified address */
  res = vxMemProbe((char *) ((int)crams_s + 0xfa),0,2,(char *)&rdata);
  if(res < 0) {
    printf("cramsInit: ERROR: No addressable board at addr=0x%x\n",(UINT32) crams_s);
    crams_s = NULL;
    return(ERROR);
  }
  printf("Clearing C-RAMS Sequencer at address 0x%x\n",(UINT32) crams_s);
  crams_s->clear = 1;
  
  
  /* Initialize and check for C-RAMS modules */
  
  if(ncrams > 0) {
    nCRAMS = 0;
    if((addr_inc >= 0x10000)&&(addr_inc < 0x01000000)) {
      raddr = laddr;
      for (ii=0;ii<ncrams;ii++) {
	raddr += addr_inc;
	crams[ii] = (struct c_rams_struct *)raddr;
	res = vxMemProbe((char *) ((int)crams[ii] + 0xfa),0,2,(char *)&rdata);
	if(res < 0) {
	  printf("cramsInit: ERROR: No addressable board at addr=0x%x\n",(UINT32) crams[ii]);
	  crams[ii] = NULL;
	  return(ERROR);
	}
	printf("Clearing C-RAMS at address 0x%x\n",(UINT32) crams[ii]);
	crams[ii]->clear = 1;
	nCRAMS++;
      }
    }else{
      printf("cramsInit: ERROR address increment between C-RAMS boards is out of range\n");
      return(ERROR);
    }
  }
  
  return(OK);
}

STATUS
cramsTprog(unsigned short ch, unsigned short t1, 
           unsigned short t2, unsigned short t3,
           unsigned short t4, unsigned short t5)
{

  int stat=0;

  taskDelay(1);
  crams_s->channels =  ch;

  stat = cramsTP(1,t1);
  if(stat<0) {
    printf("cramsTprog: ERROR programming T1 register\n");
    return(ERROR);
  }
  stat = cramsTP(2,t2);
  if(stat<0) {
    printf("cramsTprog: ERROR programming T2 register\n");
    return(ERROR);
  }
  stat = cramsTP(3,t3);
  if(stat<0) {
    printf("cramsTprog: ERROR programming T3 register\n");
    return(ERROR);
  }
  stat = cramsTP(4,t4);
  if(stat<0) {
    printf("cramsTprog: ERROR programming T4 register\n");
    return(ERROR);
  }
  stat = cramsTP(5,t5);
  if(stat<0) {
    printf("cramsTprog: ERROR programming T5 register\n");
    return(ERROR);
  }
  
  return(OK);
}

/* Print out values of T registers */
void
cramsTprint()
{

  int ii;
  unsigned short t[5];

  if(crams_s == NULL) {
    printf("cramsTprint: ERROR: C-RAMS Sequencer Not Initialized\n");
    return;
  }

  for(ii=0;ii<5;ii++) {
    t[ii] = crams_s->T[ii];
    taskDelay(1);
  }

  printf("%d %d %d %d %d\n",
	 t[0]&0xff,
	 t[1]&0x1ff,
	 t[2]&0xff,
	 t[3]&0x1ff,
	 t[4]&0x1ff);
}

void 
cramsTrig()
{

  if(crams_s == NULL) {
    logMsg("cramsTrig: ERROR: C-RAMS Sequencer Not Initialized\n",0,0,0,0,0,0);
    return;
  }

  crams_s->trigger = 1;
}

void
cramsAutoTrig(int on)
{

  unsigned short stat;

  if(crams_s == NULL) {
    logMsg("cramsAutoTrig: ERROR: C-RAMS Sequencer Not Initialized\n",0,0,0,0,0,0);
    return;
  }

  stat = (crams_s->status&CRAMS_S_STATUS_MASK);

  if(on)
    crams_s->status = (stat | CRAMS_S_STATUS_AUTO);
  else
    crams_s->status = (stat & ~CRAMS_S_STATUS_AUTO);
}

unsigned short
cramsTest(unsigned short val)
{
  unsigned short ret;

  if(crams_s == NULL) {
    logMsg("cramsTest: ERROR: C-RAMS Sequencer Not Initialized\n",0,0,0,0,0,0);
    return (0xffff);
  }

  if(val > 15) {
    logMsg("cramsTest: ERROR: Invalid value for Test Register (%d)\n",val,0,0,0,0,0);
    return(0xffff);
  }

  crams_s->test = val;
  T_DELAY(TDELAY_COUNT_L);
  ret = ((crams_s->test)&CRAMS_S_TEST_MASK);
  return(ret);
}

void
cramsTestPulse(int chanID, int pcount, int delay)
{
  int jj;

  if(crams_s == NULL) {
    logMsg("cramsTestPulse: ERROR: C-RAMS Sequencer Not Initialized\n",0,0,0,0,0,0);
    return;
  }

  if(delay <=0) delay = TDELAY_COUNT_L;

  CRAMS_TEST_MODE_ON;
  T_DELAY(delay);
  CRAMS_TEST_SHIFT_ON;
  T_DELAY(delay);
  CRAMS_TEST_CLOCK_ON;
  T_DELAY(delay);
  CRAMS_TEST_SHIFT_OFF;
  T_DELAY(delay);
  CRAMS_TEST_CLOCK_OFF;
  T_DELAY(delay);

  for(jj=0;jj<chanID;jj++) {
    CRAMS_TEST_CLOCK_ON;
    T_DELAY(delay);
    CRAMS_TEST_CLOCK_OFF;
    T_DELAY(delay);
  }

  for(jj=0;jj<pcount;jj++) {
    CRAMS_TEST_PULSE_ON;
    T_DELAY(delay);
    CRAMS_TEST_PULSE_OFF;
    T_DELAY(delay);
  }

  CRAMS_TEST_MODE_OFF;

}

/******************************************************************************
*
*  cramsClear - Clear Sequencer and/or ADCs
*
*       cflag    = 0  (Clear all modules)
*                = 1  (Clear ADCs only)
*                = 2  (Clear Sequencer only)
*
*/
void
cramsClear(int cflag)
{
  int ii;

  if(crams_s == NULL) {
    logMsg("cramsClear: ERROR: C-RAMS Sequencer Not Initialized\n",0,0,0,0,0,0);
    return;
  }
  if((cflag <= 0)||(cflag >= 2))
    crams_s->clear = 1;

  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    logMsg("cramsClear: WARN: No C-RAMS Initialized",0,0,0,0,0,0); 
    return;
  }

  if((cflag <= 0)||(cflag == 1)||(cflag >=3)) {
    for(ii=0;ii<nCRAMS;ii++)
      crams[ii]->clear = 1;
   }
 
}

STATUS
cramsWaitActive(int count)
{
  int ii=0;
  unsigned int active=0;

  if(count == 0) count = 500;

  active = (crams_s->status)&CRAMS_S_STATUS_ACTIVE;
  while (active && (ii<count)) {
     active = (crams_s->status)&CRAMS_S_STATUS_ACTIVE;
     ii++;
  }

  if(ii >= count)
    return(ERROR);
  else
    return(OK);
}

void
cramsStatus()
{
  int ii;
  unsigned short stat, chan, test;
  unsigned short t[5];


  if(crams_s == NULL) {
    printf("cramsStatus: ERROR: C-RAMS Sequencer Not Initialized\n");
    return;
  }


  test = (crams_s->test)&CRAMS_S_TEST_MODE;
  taskDelay(0);
  stat = (crams_s->status)&CRAMS_S_STATUS_MASK;

  if(test)
    printf("C-RAMS Sequencer status: 0x%x : Test Mode Enabled\n",stat);
  else
    printf("C-RAMS Sequencer status: 0x%x\n",stat);

  if(stat&CRAMS_S_STATUS_DRDY)
    printf(" Data Ready :");
  if(stat&CRAMS_S_STATUS_BUSY)
    printf(" BUSY On :");
  if(stat&CRAMS_S_STATUS_ID)
    printf(" Internal Delay :");
  if(stat&CRAMS_S_STATUS_VETO)
    printf(" Veto State :");
  if(stat&CRAMS_S_STATUS_AUTO)
    printf(" AutoTrigger :");
  if(stat&CRAMS_S_STATUS_ACTIVE)
    printf(" SEQ ACTIVE \n");
  else
    printf(" SEQ IDLE  \n"); 


  for(ii=0;ii<5;ii++) {
    t[ii] = crams_s->T[ii];
    taskDelay(1);
  }
  printf("C-RAMS Sequencer Number of Channels: %d \n",
	 (crams_s->channels)&CRAMS_S_NCHAN_MASK);
  printf(" T1:0x%04x T2:0x%04x T3:0x%04x T4:0x%04x T5:0x%04x \n",
	 t[0]&0xff,
	 t[1]&0x1ff,
	 t[2]&0xff,
	 t[3]&0x1ff,
	 t[4]&0x1ff);
  printf("\n");


  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    printf("cramsStatus: WARN: No C-RAMS Initialized"); 
    return;
  }

  for(ii=0;ii<nCRAMS;ii++) {
    stat = (crams[ii]->status)&CRAMS_STATUS_MASK;
    printf("C-RAMS %d status: 0x%04x\n",(ii+1),stat);

    if(stat&CRAMS_STATUS_TEST_MODE)
      printf(" Test Mode :");

    if(stat&CRAMS_STATUS_MEMORY_OWNER)
      printf(" P/T Memory Enabled \n");
    else
      printf(" P/T Memory Disabled \n");


    printf(" ADC 0 :");
    if(!(stat&CRAMS_STATUS_CH0_DATA))
      printf(" Data Ready :");

    if(!(stat&CRAMS_STATUS_FIFO0_EMPTY))
      printf(" FIFO Empty : ");

    if(!(stat&CRAMS_STATUS_FIFO0_FULL))
      printf(" FIFO Full \n");
    else
      printf(" \n");


    printf(" ADC 1 :");
    if(!(stat&CRAMS_STATUS_CH1_DATA))
      printf(" Data Ready :");

    if(!(stat&CRAMS_STATUS_FIFO1_EMPTY))
      printf(" FIFO Empty : ");

    if(!(stat&CRAMS_STATUS_FIFO1_FULL))
      printf(" FIFO Full \n");
    else
      printf(" \n");

    chan = (crams[ii]->channels)&CRAMS_CHANNELS_MASK;
    printf(" ADC 0 Word Count: %5d       ADC 1 Word count: %5d \n",
	   (crams[ii]->wc[0])&CRAMS_WC_MASK,
           (crams[ii]->wc[1])&CRAMS_WC_MASK);
    printf("       # Channels: %5d             # Channels: %5d \n",
	   ((chan)&0x3f)<<5,((chan)&0xfc0)>>1);

  }
  printf("\n");
}


/***********************************************************************************
*
*  cramsPrintData - Read Data from FIFOs and Print to standard out 
*
*     pflag:   0 - Print all data from all ADC FIFOs
*              1 - Print data from specified ADC 
*                - Print data from specified ADC
*                    (without descriptive headers)
*
*  Returns: N/A
*/
void
cramsPrintData(int pflag, int detID, unsigned short maxWords)
{

  int ii, jj, maxID, adcID, cramsID;
  unsigned short nData=0;
  unsigned long tmpData, chanID;

  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    printf("cramsPrintData: ERROR : No C-RAMS are Initialized\n");
    return;
  }

  maxID = (nCRAMS<<1) - 1;
  if(detID > maxID) {
    printf("cramsPrintData: ERROR : detector ID (%d) out of range (0 - %d)\n",detID,maxID);
    return;
  }
  
  cramsID = detID/2;
  adcID = detID%2;

  switch (pflag) {

  case 0:
    for(ii=0;ii<nCRAMS;ii++) {
      printf("C-RAMS at address 0x%x :\n",(UINT) crams[ii]);
      nData = (crams[ii]->wc[0])&CRAMS_WC_MASK;
      printf("  CHANNEL 0 Data: %d words",nData);
      for(jj=0;jj<nData;jj++) {
	if((jj % 8) == 0) printf("\n  ");
	printf("0x%08x ",(UINT) crams[ii]->FIFO(0));
      }
      printf("\n");
      
      nData = (crams[ii]->wc[1])&CRAMS_WC_MASK;
      printf("  CHANNEL 1 Data: %d words",nData);
      for(jj=0;jj<nData;jj++) {
	if((jj % 8) == 0) printf("\n  ");
	printf("0x%08x ",(UINT) crams[ii]->FIFO(1));
      }
      printf("\n\n");
    }
    break;

  default:
    nData = (crams[cramsID]->wc[adcID])&CRAMS_WC_MASK;
    if(nData > maxWords) nData = maxWords;
    for(jj=0;jj<nData;jj++) {
      if((jj % 8) == 0) printf("\n");
      tmpData = crams[cramsID]->FIFO(adcID);
      if(tmpData&CRAMS_DATA_VALID) {
	chanID = (tmpData&CRAMS_DATA_CHANNEL_MASK)>>12;
	if(chanID == jj)
	  printf("%d ",(UINT) (tmpData)&CRAMS_DATA_MASK);
	else
	  printf("0 ");
      } else {
	printf("0 ");
      }
    }
    printf("\n\n");

  }

}
/***********************************************************************************
*
*  cramsReadData - Read Data from FIFOs 
*
*     buf :  Address where the data should be stored
*     mode:  0 (CPU controlled Read)
*            1 (Block 32 Mode) Requires Universe DMA routines Initialized
*
*  Returns: Number of Data Words Read
*/
int
cramsReadData(int detID, unsigned long *buf,  unsigned short maxWords, int mode)
{

  int jj, maxID, cramsID=0, adcID=0;
  int nloops, remWords, retVal;
  unsigned short nData=0;

  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    logMsg("cramsReadData: ERROR : No C-RAMS are Initialized\n",0,0,0,0,0,0);
    return (-1);
  }
  
  maxID = (nCRAMS<<1) - 1;
  if(detID > maxID) {
    logMsg("cramsReadData: ERROR: detector ID (%d) out of range (0 - %d)\n",detID,maxID,0,0,0,0);
    return(-1);
  }else{
    cramsID = detID/2;
    adcID = detID%2;
  }

  /* printf("cramsID = %d adcID = %d\n",cramsID,adcID); */
  if((crams_s->status)&(CRAMS_S_STATUS_DRDY)) {

    if(mode > 0) {
      nData = (crams[cramsID]->wc[adcID])&CRAMS_WC_MASK;
      if((nData>maxWords)&&(maxWords>0)) nData = maxWords;
      nloops = nData/CRAMS_BR_MAX_WORDS;
      remWords = nData%CRAMS_BR_MAX_WORDS;
      /* printf("nloops = %d remWords = %d\n",nloops,remWords); */
      for(jj=0;jj<nloops;jj++) {
	retVal = sysVmeDmaSend((UINT32)&buf[jj*CRAMS_BR_MAX_WORDS], 
			       (UINT32)&(crams[cramsID]->FIFO(adcID)), (CRAMS_BR_MAX_WORDS<<2), 0);
	if(retVal < 0) {
	  logMsg("cramsReadData: ERROR in DMA transfer Initialization 0x%x\n",retVal,0,0,0,0,0);
	  return(retVal);
	}
	/* Wait until Done or Error */
	retVal = sysVmeDmaDone(1000);
	if(retVal < 0) {
	  logMsg("cramsReadData: ERROR in DMA transfer Initialization 0x%x\n",retVal,0,0,0,0,0);
	  return(retVal);
	}
      }
      if(remWords) {
	retVal = sysVmeDmaSend((UINT32)&buf[jj*CRAMS_BR_MAX_WORDS], 
			       (UINT32)&(crams[cramsID]->FIFO(adcID)), (remWords<<2), 0);
	if(retVal < 0) {
	  logMsg("cramsReadData: ERROR in DMA transfer Initialization 0x%x\n",retVal,0,0,0,0,0);
	  return(retVal);
	}
	/* Wait until Done or Error */
	retVal = sysVmeDmaDone(1000);
	if(retVal < 0) {
	  logMsg("cramsReadData: ERROR in DMA transfer Initialization 0x%x\n",retVal,0,0,0,0,0);
	  return(retVal);
	}
      }

    }else{
      nData = (crams[cramsID]->wc[adcID])&CRAMS_WC_MASK;
      if((nData>maxWords)&&(maxWords>0)) nData = maxWords;
      for(jj=0;jj<nData;jj++) {
	buf[jj] = crams[cramsID]->FIFO(adcID);
      }
    }

  }else{
    logMsg("cramsReadData: ERROR: No data ready for readout!\n",0,0,0,0,0,0);
    return(0);
  }

  return((int)nData);
}


STATUS
cramsPTRead(int detID, int ch, unsigned short *ped_val, unsigned short *th_val)
{

  int maxID, adcID, cramsID;
  unsigned long temp;

  /*check that there is a valid address*/
  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    printf("cramsPTRead: ERROR : No C-RAMS ADCs are Initialized\n");
    return(ERROR);
  }
  maxID = (nCRAMS<<1) - 1;
  if(detID > maxID) {
    printf("cramsPTRead: ERROR : detector ID (%d) out of range (0 - %d)\n",detID,maxID);
    return(ERROR);
  }

  cramsID = detID/2;
  adcID = detID%2;

  crams[cramsID]->status = 0;  /* Set Memory owner to VME */
  temp = crams[cramsID]->ch[adcID][ch];
  crams[cramsID]->status = 2; /* Set Memory Owner back to Conv Logic */

  *ped_val = ((temp&CRAMS_PEDESTAL_MASK)>>12);
  *th_val =   (temp&CRAMS_THRESHOLD_MASK);

  return(OK);
}

STATUS
cramsPTWrite(int detID, int ch, unsigned short ped_val, unsigned short th_val)
{

  int maxID, adcID, cramsID;
  unsigned long temp, ret;

  /*check that there is a valid address*/
  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    printf("cramsPTWrite: ERROR : No C-RAMS ADCs are Initialized\n");
    return(ERROR);
  }
  maxID = (nCRAMS<<1) - 1;
  if(detID > maxID) {
    printf("cramsPTWrite: ERROR : detector ID (%d) out of range (0 - %d)\n",detID,maxID);
    return(ERROR);
  }

  cramsID = detID/2;
  adcID = detID%2;

  temp = (unsigned long) (((ped_val<<12)&CRAMS_PEDESTAL_MASK) | 
                          (th_val&CRAMS_THRESHOLD_MASK)) ; 


  crams[cramsID]->status = 0;  /* Set Memory owner to VME */
  crams[cramsID]->ch[adcID][ch] = temp;

  /* Read back the value */
  ret = crams[cramsID]->ch[adcID][ch];

  if(temp != ret) {
    printf("cramsPTWrite: ERROR writing to register\n");
    return(ERROR);
  } else {
    crams[cramsID]->status = 2; /* Set memory Owner back to Conv Logic */
    return(OK);
  }

}

STATUS
cramsPTZero(int detID)
{

  int ii, maxID, adcID, cramsID;
  unsigned int ret;

  /*check that there is a valid address*/
  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    printf("cramsPTZero: ERROR : No C-RAMS ADCs are Initialized\n");
    return(ERROR);
  }
  maxID = (nCRAMS<<1) - 1;
  if(detID > maxID) {
    printf("cramsPTZero: ERROR : detector ID (%d) out of range (0 - %d)\n",detID,maxID);
    return(ERROR);
  }

  cramsID = detID/2;
  adcID = detID%2;

  crams[cramsID]->status = 0;  /* Set Memory owner to VME */

  for(ii=0;ii<CRAMS_NCHAN_MAX;ii++) {
    ret = 1;
    crams[cramsID]->ch[adcID][ii] = 0;
    ret = crams[cramsID]->ch[adcID][ii];
    if(ret != 0) 
      printf("cramsPTZero: ERROR writing Ped/Th for detID = %d (chan = %d)\n",detID,ii);
  }

  crams[cramsID]->status = 2; /* Set memory Owner back to Conv Logic */

  return (0);
}


/*******************************************************************************
*
* cramsProg - Setup CRAMS Sequencer and ADCs for Normal Acquisition/Readout
*
*
* RETURNS: OK.
*/

STATUS 
cramsProg(int detID, int nChan)
{
  int nChanMod32;
  int maxID, adcID, cramsID;
  unsigned short tch, nch=0;

  if(crams_s == NULL) {
    printf("cramsProg: ERROR: call cramsInit() first\n");
    return(ERROR);
  }

  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    printf("cramsProg: ERROR : No C-RAMS ADCs are Initialized\n");
    return(ERROR);
  }
  maxID = (nCRAMS<<1) - 1;
  if(detID > maxID) {
    printf("cramsProg: ERROR : detector ID (%d) out of range (0 - %d)\n",detID,maxID);
    return(ERROR);
  }

  cramsID = detID/2;
  adcID = detID%2;

  if(nChan <= 0) nChan = CRAMS_NCHAN_MIN;
  if(nChan > 2016) nChan = CRAMS_NCHAN_MAX;
  nch = (unsigned short) (0x7f0&nChan); /* Make sure nChan is integral number of 32
                                          with a Max value 2016 */ 
  nChanMod32 = (int) nch>>5;
  printf("cramsProg: INFO : Initializing C-RAM %d (ADC %d) for %d Channel Readout\n",cramsID,adcID,nch);

  crams_s->clear = 1;
  crams_s->test = 0;

  crams[cramsID]->clear = 1;
  if(adcID == 0) {
    tch = (crams[cramsID]->channels)&CRAMS_ADC1_CHAN_MASK;
    crams[cramsID]->channels = nChanMod32 | tch;  
  }else{
    tch = (crams[cramsID]->channels)&CRAMS_ADC0_CHAN_MASK;
    crams[cramsID]->channels = (nChanMod32<<6) | tch;  
  }
  
  return(OK);
}

/*******************************************************************************
*
* cramsTestProg - Setup a testpulse configuration for a given detector
*
*
* RETURNS: OK or ERROR.
*/

STATUS
cramsTestProg( int detID, unsigned short nConv)
{
  int nChanMod32, maxID, cramsID, adcID;
  unsigned short tch;

  if(crams_s == NULL) {
    printf("ERROR: call cramsInit() first\n");
    return(ERROR);
  }

  if((crams[0] == NULL) || (nCRAMS <= 0) ) {
    printf("cramsTestSeq: ERROR : No C-RAMS ADCs are Initialized\n");
    return(ERROR);
  }
  maxID = (nCRAMS<<1) - 1;
  if(detID > maxID) {
    printf("cramsTestSeq: ERROR : detector ID (%d) out of range (0 - %d)\n",detID,maxID);
    return(ERROR);
  }

  cramsID = detID/nCRAMS;
  adcID = detID%2;


  if(nConv <= 0) nConv = 32;
  if(nConv > 64) nConv = 64;
  nChanMod32 = (int) nConv>>5;
/*  printf("cramsTestSeq: INFO : Initializing C-RAMS Test (%d Samples) for Detector %d\n",nConv,detID); */

  /* Program Sequencer and T registers */
  crams_s->clear = 1;
  crams_s->iDAC = 0;


  /* Program the ADC */
  crams[cramsID]->clear = 1;
  if(adcID == 0) {
    tch = (crams[cramsID]->channels)&CRAMS_ADC1_CHAN_MASK;
    crams[cramsID]->channels = nChanMod32 | tch;  
  }else{
    tch = (crams[cramsID]->channels)&CRAMS_ADC0_CHAN_MASK;
    crams[cramsID]->channels = (nChanMod32<<6) | tch;  
  }

  return(OK);
}






