/*
 *	tsServ.c
 *
 * This file contains the routines necessary to implement a CODA run control
 * component which controls the CEBAF Trigger Supervisor module.
 */

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include "ts.h"
#define O_RDONLY 0

TS_DATABASE *tsDb;
int tsState;
TRIGGER_SUPERVISOR tsmemory,*ts=&tsmemory;

void daLogMsg(arg1,arg2,arg3,arg4)
     char *arg1;
     char *arg2,arg3,arg4;
{
  printf(arg1,arg2,arg3,arg4);
}

void tsCompressToLower(char *);
void tsParse2(char *,char **,char **);

int daCreate(name,number)
     char *name;
     int number;
{
  tsDb = (TS_DATABASE *) malloc(sizeof(TS_DATABASE));
  if (tsDb != NULL) {
    tsDb->name = (char *) malloc(strlen(name)+1);
    strcpy(tsDb->name,name);
    tsDb->number = number;
    ts->csr = TS_RESET;
    tsState = DA_CREATED;
  } else {
    tsState = DA_FATAL;
  }
  return(tsState);
}

int daDownload(arg)
     char *arg;
{
  FILE *fd;
  char line[128], *namep, *stringp;
  int value, lineno, errs, maxlen;

  errs=0;
  maxlen=127;			/* leave room to append a blank */
  if ((tsState==DA_CREATED) || (tsState==DA_DOWNLOADED)) {
    fd = fopen(arg,"r");
    if (fd==NULL) {
      daLogMsg("error opening file %s\n",arg);
    } else {
      lineno = 0;
      while ((fgets(line,maxlen,fd)) != NULL) {
	lineno++;
	(void) tsParse2(line,&namep,&stringp);
	if (*namep=='\0') continue; /* comment or blank line */
	if (*stringp=='\0') {
	  daLogMsg("***missing value on line %d:%s\n",lineno,line);
	  errs++;
	  continue;
	}
	if ((strcmp(namep,"tstriggerclass")==0) ||
	    (strcmp(namep,"tsroccode")==0) ||
	    (strcmp(namep,"tsl1accept")==0) ) {
	  if (!daWriteString(namep,stringp)) errs++;
	} else {
	  if (sscanf(stringp,"%i",&value)!=1) {
	    daLogMsg("%s\n***bad value on line %d\n",
		     line,lineno);
	    errs++;
	    continue;
	  }
	  if (!daWriteInt(namep,value)) errs++;
	}
      }			/* end of loop over lines */
      fclose(fd);
      if (errs==0) tsState = DA_DOWNLOADED;
    }
  } else
    daLogMsg("invalid Download command while in state %d\n",tsState);
  return(tsState);
}

int daPreStart(run,runtype)
     int run, runtype;
{
  if (tsState==DA_DOWNLOADED) 
    tsState=DA_PAUSED;
  return(tsState);
}

int daGo(arg)
     int arg;
{
  if (tsState==DA_PAUSED) {
    ts->csr = TS_DIS_POS;
    ts->csr = TS_GO;
    tsState=DA_ACTIVE;
  }
  return(tsState);
}

int daPause(reason)
     int reason;
{
  int count;
  if (ts->sync) {
    ts->csr = TS_ENA_POS;
    for (count=0;count<TS_TICKS_TO_STOP;count++) {
      if (ts->csr | TS_SYNC_OCCURED) break;
#ifdef VXWORKS
      taskDelay(1);
#endif
    }
  }
  ts->csr = TS_STOP;		/* stop TS even if state is wrong */
  if (tsState==DA_ACTIVE) {
    tsState=DA_PAUSED;		/* only allow valid transitions */
  }
  return(tsState);
}

int daEnd()
{
  daPause(0);			/* end and pause identical for TS */
  if (tsState!=DA_CREATED) tsState=DA_DOWNLOADED;
  return(tsState);
}

int daReport()
{
  return(tsState);
}

int daWriteInt(name,value)
     char *name;
     int value;
{
  char *cp;
  int bad,index;
  bad = 0;
  for (cp=name;*cp;cp++) tolower(*cp);
  switch ((int)name[2]) {
  case 'p':
    if (sscanf(name,"tsprescale%d",&index)!=1)
      bad=1;
    else if ((index<1) || (index>12))
      bad=1;
    else
      ts->prescale[index-1] = value;
    break;
  case 'e':
    if (strcmp(name,"tsenable")==0)
      ts->ena = value;
    else if (sscanf(name,"tsenable%d",&index)==1) {
      if ((index<1) || (index>12))
	bitset(ts->ena,index);
      else
	bad=1;
    } else
      bad=1;
    break;
  case 's':
    if (strcmp(name,"tsstrobe")==0) {
      if (value)
	bitset(ts->ena,TS_STROBE);
      else
	bitclear(ts->ena,TS_STROBE);
    } else if (strcmp(name,"tssyncinterval")==0) {
      ts->sync = value;
      ts->csr = value ? TS_ENA_SYNC : TS_DIS_SYNC;
    }
    else
      bad=1;
    break;
  case 'r':
    if (strcmp(name,"tsrocenable")==0)
      ts->rocena = value;
    else if (strcmp(name,"tsroclock")==0)
      ts->csr = value ? TS_ENA_RLOCK : TS_DIS_RLOCK;
    else if (strcmp(name,"tsroclock4")==0)
      ts->csr = value ? TS_ENA_RLOCK4 : TS_DIS_RLOCK4;
    else
      bad=1;
    break;
  case 'c':
    if (strcmp(name,"tsclearpermittimer")==0) {
      ts->cptimer = value;
      ts->csr = value ? TS_ENA_CPTIMER : TS_DIS_CPTIMER;
    } else if (strcmp(name,"tsclearholdtimer")==0) {
      ts->chtimer = value;
      ts->csr = value ? TS_ENA_CHTIMER : TS_DIS_CHTIMER;
    } else
      bad=1;
    break;
  case 'b':
    if (strcmp(name,"tsbusytimer")==0) {
      ts->fbtimer = value;
      ts->csr = value ? TS_ENA_FBTIMER : TS_DIS_FBTIMER;
    } else
      bad=1;
    break;
  case 'l':
    if (strcmp(name,"tslevel2timer")==0) {
      ts->l2timer = value;
    } else if (strcmp(name,"tsl3timer")==0) {
      ts->l3timer = value;
    } else
      bad=1;
    break;
  default:
    bad=1;
  }
  if (bad) {
    daLogMsg("unknown parameter %s\n",name);
    return(0);
  } else
    return(1);
}

int daWriteString(name,string)
     char *name,*string;
{
  char *cp;
  int status;
  for (cp=name;*cp;cp++) tolower(*cp);
  if (strcmp(name,"tstriggerclass")==0) 
    status = tsSetMLU(TS_MLU_TC|TS_MLU_L1OK,string);
  else if (strcmp(name,"tsroccode")==0)
    status = tsSetMLU(TS_MLU_RC,string);
  else if (strcmp(name,"tsl1accept")==0)
    status = tsSetMLU(TS_MLU_L1A,string);
  else
    status = 0;
  return(status);
}

int tsSetMLU(mask,string)
     int mask;
     char *string;
{
  int defmask,trig,rpn[256];
  if (tsMLUparse(string,&defmask,rpn)==0) {
    daLogMsg("unknown bit definition: %s",string);
    return(0);
  }
  if ((mask&defmask)!=defmask) {
    daLogMsg("illegal bit definition: %s",string);
    return(0);
  }
  for (trig=0;trig<4096;trig++) {
    ts->mlu[trig] &= -1-mask;		/* clear bits being defined */
    ts->mlu[trig] |= tsEvaluate(trig,rpn);
  }
  return(1);
}

void tsParse2(line,namep,stringp)
     char *line,**namep,**stringp;
{
  char *cp;
  for (cp=line;*cp;cp++) {
    if (*cp=='\t') *cp=' ';
    else if (*cp=='\n') *cp=' ';
    else if (*cp=='!') {
      *cp='\0';		/* terminate line at ! */
      break;
    }
    else *cp = tolower(*cp);
  }
  *cp++=' '; *cp=0;	/* append a blank for later tests */
  *namep=line;
  while(**namep==' ') (*namep)++; /* find first non-white space */
  if (**namep!='\0') { /* skip comment only and blank lines */
    *stringp=*namep+1;
    while(**stringp!=' ') (*stringp)++;
    *((*stringp)++) = '\0';	/* terminate the name */
    while(**stringp==' ') (*stringp)++;	/* find start of second field */
  } else
    *stringp=*namep;
}

void tsCompressToLower(string)
     char *string;
{
  char *cp;
  for (cp=string;*string;string++) {
    if ((*string!=' ')&&(*string!='\t')) {
      tolower(*string);
      *cp++=*string;
    }
  }
  *cp='\0';
}

int tsMLUparse(string,mask,rpn)
     char *string;
     int *mask,*rpn;
{
  char *start,*semi;
  int mask1,first;
  tsCompressToLower(string);
  *mask=0;
  first=1;
  for (start=string;*string;string++) {
    if (*string=='=') {
      *string=0;
      if (!tsNameToMask(start,&mask1)) return(0);
      *mask |= mask1;
      *string = '=';
      start = string+1;
      semi = 0;
      while (*string) {
	if (*string==';') {
	  semi=string;
	  *string=0;
	  break;
	}
	string++;
      }
      if (!tsStrToRpn(start,&rpn)) return(0);
      *rpn++ = TS_RPN_EVAL;
      *rpn++ = mask1;
      if (!first) *rpn++ = TS_RPN_OR;
      first = 0;
      if (semi) *semi = ';';
      start = string+1;
    }
  }
  *rpn++ = 0;
  return(1);
}    

int tsNameToMask(string,mask)
     char *string;
     int *mask;
{
  int num;
  if (*string=='l') {
    if (*(++string)!='1') return(0);
    if (*(++string)=='o') {
      if (*(++string)!='k') return(0);
      *mask = TS_MLU_L1OK;
    } else if (*string++=='a') {
      if (sscanf(string++,"%d",&num)!=1) return(0);
      if ((num<1)||(num>8)) return(0);
      *mask = 1<<(num+15);
    } else 
      return(0);
  } else if (*string=='t') {
    if (*(++string)!='c') return(0);
    if (*(++string)=='1')
      *mask = TS_MLU_TC1;
    else if (*string=='2')
      *mask = TS_MLU_TC2;
    else if (*string=='3')
      *mask = TS_MLU_TC3;
    else
      return(0);
  } else if (*string++=='r') {
    if (*string++!='c') return(0);
    if (*string=='0')
      *mask = TS_MLU_RC0;
    else if (*string=='1')
      *mask = TS_MLU_RC1;
    else if (*string=='2')
      *mask = TS_MLU_RC2;
    else if (*string=='3')
      *mask = TS_MLU_RC3;
    else
      return(0);
  } else
    return(0);
  if (*(++string)!='\0') return(0);
  return(1);
}

int tsStrToRpn(string,rpn)
     char *string;
     int **rpn;			/* (*rpn) is a pointer to the rpn code */
{
  char opstack[32],*sp;
  int tbit;
  if ((*string=='1') && (*(string+1)==0)) {
    *(*rpn)++ = TS_RPN_SETT;	/* most trival expression: "1" */
    return(1);
  }
  sp = opstack;
  *sp = '\0';			/* start with NULL on stack */
  while (*string) {
    switch(*string++) {
    case 't': case 'T':
      *(*rpn)++ = TS_RPN_GET;
      if (sscanf(string++,"%d",&tbit)!=1) return(0);
      if (tbit<1 || tbit>12) return(0);
      *(*rpn)++ = 1<<(tbit-1);	/* save as address mask */
      if (tbit>9) string++;	/* bump again if 2 digits */
      break;
    case '/':
      *(*rpn)++ = TS_RPN_GET;
      if (*string++!='t') return(0);
      if (sscanf(string++,"%d",&tbit)!=1) return(0);
      if (tbit<1 || tbit>12) return(0);
      *(*rpn)++ = 1<<(tbit-1);	/* save as address mask */
      if (tbit>9) string++;	/* bump again if 2 digits */
      *(*rpn)++ = TS_RPN_NEG;
      break;
    case '+':
      if (*sp=='*') {
	*(*rpn)++ = TS_RPN_AND;
	sp--;
      }
      if (*sp=='+') {
	*(*rpn)++ = TS_RPN_OR;
	sp--;
      }
      *(++sp) = '+';
      break;
    case '*':
      if (*sp=='*') {
	*(*rpn)++ = TS_RPN_AND;
	sp--;
      }
      *(++sp) = '*';
      break;
    case '(':
      *(++sp) = '(';
      break;
    case ')':
      if (*sp=='*') {
	*(*rpn)++ = TS_RPN_AND;
	sp--;
      }
      if (*sp=='+') {
	*(*rpn)++ = TS_RPN_OR;
	sp--;
      }
      if (*sp-- != '(') return(0);
      break;
    default:
      return(0);
    }
  }
  if (*sp=='*') {
    *(*rpn)++ = TS_RPN_AND;
    sp--;
  }
  if (*sp=='+') {
    *(*rpn)++ = TS_RPN_OR;
    sp--;
  }
  if (*sp != '\0') return(0);	/* stack at this point should be empty */
  return(1);
}

int tsEvaluate(trig,rpn)
     int trig;
     int *rpn;
{
  int stack[32], *sp, mask, data;
  sp = stack;
  *sp = 0;
  while (*rpn) {
    switch (*rpn++) {
    case 1:			/* '1': bit is always true */
      *(++sp) = 1;
      break;
    case 2:			/* test specified bit */
      mask = *rpn++;
      *(++sp) = (trig&mask) ? 1 : 0;
      break;
    case 3:			/* negate signal */
      *sp = (*sp) ? 0 : 1;
      break;
    case 4:			/* OR last 2 expressions */
      data = *sp--;
      *sp = *sp | data;
      break;
    case 5:			/* AND last 2 expressions */
      data = *sp--;
      *sp = *sp & data;
      break;
    case 6:			/* set desired bit */
      data = *rpn++;
      *sp = (*sp) ? data : 0;
      break;
    default:
      daLogMsg("bad MLU rpn code");
      return(0);
    }
  }
  return(*sp);
}

