/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * TJNAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *       heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363         *
 *----------------------------------------------------------------------------*
 * Description: follows this header.
 *
 * Author:
 *	David Abbott
 *	TJNAF Data Acquisition Group
 *
 * Revision History:
 *	  Initial revision
 *
 *
 *----------------------------------------------------------------------------*/

 /* vme_trigLib.c -- Primitive trigger control for VME CPUs using the TJNAF 
                     Trigger Supervisor interface card or Trigger Supervisor

 File : vme_trigLib.h

 Supported Interrupt/Trigger sources:

       code       Hardware type
        1     VME Trigger Supervisor Interface  (TIR)
        2     Trigger Supervisor (Direct via Branch 5)

 Routines:
	   void vmetriglink();       link isr to trigger
	   void vmetenable();        enable trigger
	   void vmetdisable();       disable trigger
           void vmetack();           Acknowledge/Reset trigger
	   char vmettype();          read trigger type
	   int  vmettest();          test for trigger  (POLL)

------------------------------------------------------------------------------*/

/* Define Hardware sources */
#define TIR_SOURCE 1
#define TS_SOURCE  2

/* Define global variable for TIR board revision */
unsigned int tirVersion = 0; /* Default is an invalid Version ID */

/* Define TIR Version IDs (readback of a reset CSR register */
#define TIR_VERSION_1   0x05c0
#define TIR_VERSION_2   0x0580

/* Define default Interrupt vector and Levels for possible sources */
#define TIR_DEFAULT_ADDR 0x0ed0
#define TS_DEFAULT_ADDR  0xed0000
#define VME_VEC 0xec
#define TIR_LEVEL 5
#define TS_LEVEL 3

#ifdef VXWORKSPPC
#define INUM_TO_IVEC(intNum)    ((VOIDFUNCPTR *) (intNum))
#else
#define INUM_TO_IVEC(intNum)    ((VOIDFUNCPTR *) (intNum<<2))
#endif

int
tirInit(unsigned int tAddr)
{
  int stat = 0;
  unsigned long laddr;
  unsigned short rval;

  if (tAddr == 0) {
    tAddr = TIR_DEFAULT_ADDR;
  }

  stat = sysBusToLocalAdrs(0x29,tAddr,&laddr);
  if (stat != 0) {
     printf("tirInit: ERROR: Error in sysBusToLocalAdrs res=%d \n",stat);
  } else {
     printf("TIR address = 0x%x\n",laddr);
     tir[1] = (struct vme_tir *)laddr;
  }

  /* Check if TIR board is readable */
  stat = vxMemProbe((char *)laddr,0,2,(char *)&rval);
  if (stat != 0) {
    printf("tirInit: ERROR: TIR card not addressable\n");
    return(-1);
  } else {
    tir[1]->tir_csr = 0x80; /* Reset the board */
    tirVersion = (unsigned int)rval;
    printf("tirInit: tirVersion = 0x%x\n",tirVersion);
  }

  return(0);
  
}



static inline void 
vmetriglink(int code, VOIDFUNCPTR isr)
{

  switch (code) {
  case TIR_SOURCE:
    if(tir[1]) {
      tir[1]->tir_csr = 0x80;
      tir[1]->tir_vec = VME_VEC;
    }else{
      printf(" vmetriglink: ERROR: tir[1] uninitialized\n");
      return;
    }
    break;
  case TS_SOURCE:
    if(ts) {
      ts->intVec = VME_VEC;
    }else{
      printf(" vmetriglink: ERROR: ts uninitialized\n");
      return;
    }    
    break;
  default:
    printf(" vmetriglink: ERROR: source type %d undefined\n",code);
    return;
  }


#ifdef VXWORKSPPC
  if((intDisconnect(INUM_TO_IVEC(VME_VEC)) !=0))
     printf("Error disconnecting Interrupt\n");
#endif
  intConnect(INUM_TO_IVEC(VME_VEC),isr,0);

}

static inline void 
vmetenable(int code, unsigned int intMask)
{
 int lock_key;

#ifndef POLLING_MODE
  lock_key = intLock();

 if(code == TIR_SOURCE)
   sysIntEnable(TIR_LEVEL);
 if(code == TS_SOURCE)
   sysIntEnable(TS_LEVEL);
#ifdef VXWORKSPPC
  intEnable(11);         /* Enable VME Interrupts on IBC chip (Radstone PPC604 only) */
#endif

#ifdef TRIG_SUPV
/*Set TIR to accept Trigger Supervisor triggers*/
 if(code == TIR_SOURCE)
   tir[1]->tir_csr = 0x6;
 if(code == TS_SOURCE)
   ts->csr2 |= 0x1800;
#else
/*Set TIR for External triggers*/
 if(code == TIR_SOURCE)
   tir[1]->tir_csr = 0x7;
 if(code == TS_SOURCE)
   ts->csr2 |= 0x0000;
#endif
  intUnlock(lock_key);

#else

#ifdef TRIG_SUPV
/*Set TIR for Trigger Supervisor triggers*/
 if(code == TIR_SOURCE)
   tir[1]->tir_csr = 0x2;
 if(code == TS_SOURCE)
   ts->csr2 |= 0x1000;
#else
/*Set TIR for External triggers*/
 if(code == TIR_SOURCE)
   tir[1]->tir_csr = 0x3;
 if(code == TS_SOURCE)
   ts->csr2 |= 0x0000;
#endif

#endif
}

static inline void 
vmetdisable(int code,unsigned int intMask)
{
 if(code == TIR_SOURCE)
   tir[1]->tir_csr = 0x80;
 if(code == TS_SOURCE)
   ts->csr2 &= ~(0x1800);

}

static inline void 
vmetack(int code, unsigned int intMask)
{
 if(code == TIR_SOURCE)
   tir[1]->tir_dat = 0x8000;
 if(code == TS_SOURCE)
   ts->lrocBufStatus = 0x100;

}

static inline unsigned long 
vmettype(int code)
{
 unsigned long tt;
#ifdef TRIG_SUPV

 if(code == TIR_SOURCE) {
   tt = (((tir[1]->tir_dat)&0x3c)>>2);
   syncFlag = (tir[1]->tir_dat)&0x01;
   lateFail = ((tir[1]->tir_dat)&0x02)>>1;
 }
 if(code == TS_SOURCE) {
   tt = (((ts->lrocData)&0xfc)>>2);
   syncFlag = (ts->lrocData)&0x01;
   lateFail = ((ts->lrocData)&0x02)>>1;
 }

#else
/* Set default trigger type to 1 */
  tt=1;
#endif

  return(tt);
}

static inline int 
vmettest(int code)
{
  unsigned short sval=0;
  unsigned int   lval=0;

 if(code == TIR_SOURCE) {
   sval = tir[1]->tir_csr;
   if( (sval != 0xffff) && ((sval&0x8000) != 0) ) {
     return (1);
   } else {
     return (0);
   }
 }

 if(code == TS_SOURCE) {
   lval = ts->lrocBufStatus;
   if( (lval != 0xffffffff) && ((lval&0xf) != 0) ) {
     return (1);
   } else {
     return (0);
   }
 }

  return(0);
}

