//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
//
// Description:
//       CODA service class 
//
// Author:  Jie Chen
//
// Revision History:
//   $Log: codaService.cc,v $
//   Revision 1.5  1998/05/28 17:47:13  heyes
//   new GUI look
//
//   Revision 1.4  1998/01/05 18:22:04  rwm
//   Added ifdef for Linux compilation.
//
//   Revision 1.3  1997/02/03 13:46:27  heyes
//   add ask and msg
//
//   Revision 1.2  1997/01/24 16:38:43  chen
//   fix bug of monitoring unknown component
//
//   Revision 1.1.1.1  1996/10/11 13:39:32  chen
//   run control source
//
//
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef solaris
extern "C" int gethostname (char *, int);
#endif

#include <cdevDirectory.h>
#include "codaService.h"
#include "codaRequestObject.h"
#include "dplite.h" 

int codaService::CODA_TAG_VALUE = 0;
int codaService::CODA_TAG_STATUS = 0;
int codaService::CODA_TAG_SEVERITY = 0;
int codaService::CODA_TAG_TIME = 0;
int codaService::CODA_TAG_UNITS = 0;
int codaService::CODA_TAG_DISPHI = 0;
int codaService::CODA_TAG_DISPLO = 0;
int codaService::CODA_TAG_ALRMHI = 0;
int codaService::CODA_TAG_ALRMLO = 0;
int codaService::CODA_TAG_WRNHI = 0;
int codaService::CODA_TAG_WRNLO = 0;
int codaService::CODA_TAG_CTRLHI = 0;
int codaService::CODA_TAG_CTRLLO = 0;

// new tags
int codaService::CODA_TAG_PV = 0x1001;
int codaService::CODA_TAG_DFV = 0x1002;
int codaService::CODA_TAG_RO = 0x1003;
int codaService::CODA_TAG_NAME = 0x1004;
//=========================================================================
//        Implementation of class codaService
//=========================================================================
codaService::codaService (char* name, cdevSystem& system)
:cdevService (name, system), codaClient_ (), fds_ (0), numFds_ (0),
 numComps_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("              Create codaService Class Object\n");
#endif
  // convert char string tags to integer tags
  if (codaService::CODA_TAG_CTRLLO == 0) {
    cdevData::insertTag (codaService::CODA_TAG_PV, "PV");
    cdevData::insertTag (codaService::CODA_TAG_DFV, "DEFAULT");
    cdevData::insertTag (codaService::CODA_TAG_RO, "readonly");
    codaService::mapCtagToItag ();
  }
}

codaService::~codaService (void)
{
#ifdef _TRACE_OBJECTS
  printf ("              Delete codaService Class Object\n");
#endif
  if (codaClient_.connected ()) 
    codaClient_.disconnect ();
  if (numFds_ > 0)
    delete []fds_;

  for (int i = 0; i < numComps_; i++)
    delete []components_[i];
  numComps_ = 0;
}

int
codaService::connect (char* session)
{
  if (codaClient_.connected ())
    return CDEV_SUCCESS;

  // open connection to a remote coda run control server
  int error = 0;
  char *tmp = ::getenv ("EXPID");

  if (!tmp) {
    reportError (CDEV_SEVERITY_SEVERE, serviceName_, 0, 
		 "cannot find EXPID environment variable\n");
    error = 1;
  }

  // msql server host
  char mhost[128];
  int  len = sizeof (mhost);
  
  char* mtmp = ::getenv ("MSQL_TCP_HOST");
  if (!mtmp) {
    reportError (CDEV_SEVERITY_SEVERE, serviceName_, 0,
		 "MSQL_TCP_HOST must be set\n");
    printf("MSQL_TCP_HOST must be set!\n");
    return CDEV_NOTCONNECTED;
  } else 
    strncpy (mhost, mtmp, sizeof (mhost));

  DP_cmd_init(mhost);

  if (!error) {
    if (codaClient_.connect (tmp, session, mhost) != CODA_SUCCESS) {
      reportError (CDEV_SEVERITY_SEVERE, serviceName_, 0,
		   "Cannot find coda run control server anywhere!!!\n");
    }
    else {
      if (numFds_ > 0) 
	delete []fds_;

      // setup file descriptor array
      fds_ = new int[1];
      numFds_ = 1;
      fds_[0] = codaClient_.getFd ();
      // register disconnect callback
      codaClient_.disconnectCallback ((rcCallback)&(codaService::discCallback),
				      (void *)this);

      // get initial information from server
      codaClient_.pendIO (CODA_DEFAULT_TIMEOUT);
    }
  }
  if (codaClient_.connected ())
    return CDEV_SUCCESS;
  else
    return CDEV_NOTCONNECTED;
}


int
codaService::getFd (int* &fd, int& numFd)
{
  fd = fds_;
  numFd = numFds_;
  return CDEV_SUCCESS;
}

int
codaService::flush (void)
{
#ifdef _CDEV_DEBUG
  printf ("Flush by codaService\n");
#endif
  // all requests will be sent out without waiting
  return CDEV_SUCCESS;
}

int
codaService::poll (void)
{
#ifdef _CDEV_DEBUG
  printf ("Poll by codaService\n");
#endif
  if (codaClient_.connected ()) {
    if (codaClient_.pendIO (0.0) == CODA_SUCCESS)
      return CDEV_SUCCESS;
    else
      return CDEV_IOFAILED;
  }
  else
    return CDEV_NOTCONNECTED;
}

int
codaService::pend (int fd)
{
#ifdef _CDEV_DEBUG
  printf ("Pend by codaService\n");
#endif  
  if (codaClient_.connected ()) {
    if (codaClient_.pendIO (0.0) == CODA_SUCCESS)
      return CDEV_SUCCESS;
    else
      return CDEV_IOFAILED;
  }
  else
    return CDEV_NOTCONNECTED;
}

int
codaService::pend (double seconds, int)
{
#ifdef _CDEV_DEBUG
  printf ("Pend by codaService for %lf seconds\n", seconds);
#endif  
  int status;
  if (codaClient_.connected ()) {
    if ((status = codaClient_.pendIO (seconds)) == CODA_SUCCESS)
      return CDEV_SUCCESS;
    else if (status == CODA_WARNING)
      return CDEV_TIMEOUT;
    else
      return CDEV_IOFAILED;
  }
  else
    return CDEV_NOTCONNECTED;
}

int
codaService::getRequestObject (char* deviceName, char* msg,
			       cdevRequestObject* &req)
{
  if (strcmp(msg,"msg") != 0) {
    if (!codaClient_.connected ()) {
      if (connect (deviceName) != CODA_SUCCESS) {
	printf("failed to connect to %s\n",deviceName);
	return CODA_ERROR;
    } else {
	codaClient_.monitorOnCallback (deviceName, "components",
				       (rcCallback)&(codaService::dynamicCompCallback),
				       (void *)this);
      }
    }
  } else { 
    // msql server host
    static int once = 1;
    if (once) {
      char mhost[128];
      int  len = sizeof (mhost);
      char* mtmp = ::getenv ("MSQL_TCP_HOST");
      if (!mtmp) {
	reportError (CDEV_SEVERITY_SEVERE, serviceName_, 0,
		     "MSQL_TCP_HOST must be set\n");
	return CDEV_ERROR;
      } else 
	strncpy (mhost, mtmp, sizeof (mhost));
      
      once = 0;
      DP_cmd_init(mhost);
    }
  }
  printf("here\n");
  req = new codaRequestObject (deviceName, msg, this, system_);
  return CDEV_SUCCESS;
}

void
codaService::mapCtagToItag (void)
{
  cdevData::tagC2I ("value", &codaService::CODA_TAG_VALUE);
  cdevData::tagC2I ("name", &codaService::CODA_TAG_NAME);
  cdevData::tagC2I ("status", &codaService::CODA_TAG_STATUS);
  cdevData::tagC2I ("severity", &codaService::CODA_TAG_SEVERITY);
  cdevData::tagC2I ("time", &codaService::CODA_TAG_TIME);
  cdevData::tagC2I ("units", &codaService::CODA_TAG_UNITS);
  cdevData::tagC2I ("displayHigh", &codaService::CODA_TAG_DISPHI);
  cdevData::tagC2I ("displayLow", &codaService::CODA_TAG_DISPLO);
  cdevData::tagC2I ("alarmHigh", &codaService::CODA_TAG_ALRMHI);
  cdevData::tagC2I ("alarmLow", &codaService::CODA_TAG_ALRMLO);
  cdevData::tagC2I ("warningHigh", &codaService::CODA_TAG_WRNHI);
  cdevData::tagC2I ("warningLow", &codaService::CODA_TAG_WRNLO);
  cdevData::tagC2I ("controlHigh", &codaService::CODA_TAG_CTRLHI);
  cdevData::tagC2I ("controlLow", &codaService::CODA_TAG_CTRLLO);
  // Initialize tag to attribute table
#ifdef _CDEV_DEBUG
  printf("value tag value is %d\n", codaService::CODA_TAG_VALUE);
  printf("status tag value is %d\n", codaService::CODA_TAG_STATUS);
  printf("severity tag value is %d\n", codaService::CODA_TAG_SEVERITY);
  printf("time tag value is %d\n", codaService::CODA_TAG_TIME);
  printf("units tag value is %d\n", codaService::CODA_TAG_UNITS);
  printf("displayHigh tag value is %d\n", codaService::CODA_TAG_DISPHI);
  printf("displayLow tag value is %d\n", codaService::CODA_TAG_DISPLO);
  printf("alarmHigh tag value is %d\n", codaService::CODA_TAG_ALRMHI);
  printf("alarmLow tag value is %d\n", codaService::CODA_TAG_ALRMLO);
  printf("warningHigh tag value is %d\n", codaService::CODA_TAG_WRNHI);
  printf("warningLow tag value is %d\n", codaService::CODA_TAG_WRNLO);
  printf("controlHigh tag value is %d\n", codaService::CODA_TAG_CTRLHI);
  printf("controlLow tag value is %d\n", codaService::CODA_TAG_CTRLLO);
#endif
}

int
codaService::getNameServer (cdevDevice* &ns)
{
  ns = 0;
  return CDEV_SUCCESS;
}

void
codaService::discCallback (int status, void* arg, daqNetData* )
{
  codaService* obj = (codaService *)arg;

  if (status == CODA_SUCCESS) {
    for (int i = 0; i < obj->numFds_; i++)
      obj->registerFd (obj->fds_[i], 0);
    obj->numFds_ = 0;
    delete [](obj->fds_);
  }
}

void
codaService::dynamicCompCallback (int status, void* arg, daqNetData* data)
{
  codaService* obj = (codaService *)arg;
  char ddlstr[80];
  int i, st;
  cdevSystem& system = obj->system_;
  cdevData out, result;

  if (status == CODA_SUCCESS) {
    // clear out old information
    for (i = 0; i < obj->numComps_; i++) {
      out.remove ();
      result.remove ();

      if (::strcasecmp (obj->components_[i], "unknown") != 0) {
	out.insert ("device", obj->components_[i]);
	out.insert ("class", "DAQ");
	st = system.nameServer ().send ("remove", out, result);
	if (st != CDEV_SUCCESS) 
	  fprintf (stderr, "Cannot remove CODA device %s\n", 
		   obj->components_[i]);
      }
      delete []obj->components_[i];
    }
    obj->numComps_ = 0;

    // get new components
    int count = CODA_MAX_DYN_COMPS;
    
    if (data->getData (obj->components_, count) != CODA_ERROR) {
      obj->numComps_ = count;
      for (i = 0; i < obj->numComps_; i++) {
	out.remove ();
	result.remove ();
	if (::strcasecmp (obj->components_[i], "unknown") != 0) {
	  sprintf (ddlstr, "DAQ : %s;", obj->components_[i]);
	  out.insert ("value", ddlstr);
	  if (system.nameServer().send ("update", out, result) != CDEV_SUCCESS)
	    fprintf (stderr, "Cannot update new ddlstr for CODA service\n");
	
	}
      }
    }
  }
}
    

cdevService*
newCodaService (char* name, cdevSystem* system)
{
  return new codaService (name, *system);
}
