/*----------------------------------------------------------------------------*
 *  Copyright (c) 1991, 1992  Southeastern Universities Research Association, *
 *                            Continuous Electron Beam Accelerator Facility   *
 *                                                                            *
 *    This software was developed under a United States Government license    *
 *    described in the NOTICE file included as part of this distribution.     *
 *                                                                            *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606 *
 *       heyes@cebaf.gov   Tel: (804) 249-7030    Fax: (804) 249-7363         *
 *----------------------------------------------------------------------------*
 * Description: follows this header.
 *
 * Author:
 *	David Abbott
 *	CEBAF Data Acquisition Group
 *
 * Revision History:
 *	  Initial revision
 *
 *
 *----------------------------------------------------------------------------*/

 /* sfi_trigLib.c -- Primitive trigger control for the SFI 340 using available
                     interrupt sources including CEBAF designed AUX
                     Trigger supervisor interface card.
                            intmask -- refers to the bit pattern one wishes
                                       to use to enable certain trigger sources
                                       (see the VME IRQ Source and Mask Register)
                                            NIM1,NIM2,NIM3
                                            ECL1
                                            AUX_B42
                                            etc...

 File : sfi_trigLib.h

 Routines:
           void sfiAuxWrite();       write 32bit word to AUX port
           void sfiAuxRead();        read 32bit word from AUX port
	   void sfitriglink();       link in trigger isr
	   void sfitenable();        enable trigger
	   void sfitdisable();       disable trigger
           void sfitack();           Acknowledge/Reset trigger
	   char sfittype();          read trigger type
	   int  sfittest();          test for trigger (POLL)

------------------------------------------------------------------------------*/


#define mem32(a) (*(unsigned long *) (a))

static inline void
sfiAuxWrite(val32)
     unsigned long val32;
{
  *sfi.sequencerDisable = 0;
  *sfi.writeAuxReg = val32;
  *sfi.writeVmeOutSignalReg = 0x4000;
  *sfi.generateAuxB40Pulse = 0;
  *sfi.writeVmeOutSignalReg = 0x40000000;
}

static inline void
sfiAuxRead(val32)
     unsigned long *val32;
{
  *sfi.sequencerDisable = 0;
  *sfi.writeVmeOutSignalReg = 0x2000;
  *val32 = *sfi.readLocalFbAdBus;
  *sfi.writeVmeOutSignalReg = 0x20000000;
}

static inline void 
sfitriglink(isr)
     VOIDFUNCPTR isr;
{
  static FUNCPTR handlerPtr = (FUNCPTR) 0;
  
  if (handlerPtr)
    free(handlerPtr);

/* Enable level 5 interrupts on SFI*/  
  *sfi.VmeIrqLevelAndVectorReg = (0xd00 | 0xED);
  handlerPtr = intHandlerCreate(isr,0);
  intVecSet((VOIDFUNCPTR *) (0xED<<2),(VOIDFUNCPTR) handlerPtr);

}

static inline void 
sfitenable(intMask)
 unsigned long intMask;
{
 int lock_key;

#ifndef POLLING_MODE

  lock_key = intLock();
  *sfi.VmeIrqMaskReg  = intMask;      /* Enable source */

#ifdef TRIG_SUPV
/*Set AUX Port for Trigger Supervisor triggers*/
  sfiAuxWrite(0);
#else
/*Set AUX Port for External triggers*/
  if(intMask == 0x10) {
    sfiAuxWrite(0x180);
  }
#endif
  intUnlock(lock_key);

#else

  *sfi.VmeIrqLevelAndVectorReg = (0x500 | 0xED);     /* Enable Internal Interrupts */
  *sfi.VmeIrqMaskReg  = intMask;                     /* Enable source*/

#ifdef TRIG_SUPV
/*Set AUX Port for Trigger Supervisor triggers*/
  sfiAuxWrite(0);
#else
/*Set AUX Port for External triggers*/
  if(intMask == 0x10) {
    sfiAuxWrite(0x180);
  }
#endif

#endif
}

static inline void 
sfitdisable(intMask)
 unsigned long intMask;
{

  *sfi.VmeIrqMaskReg  = (intMask<<8);      /* Clear Source  */

/*Reset AUX Port */
  if(intMask == 0x10) {
    sfiAuxWrite(0x8000);
  }
}

static inline void 
sfitack(intMask)
 unsigned long intMask;
{
#ifdef POLLING_MODE
  *sfi.VmeIrqMaskReg  = (intMask<<8);      /* Clear Source  */
#endif

  if(intMask == 0x10) {
    *sfi.writeVmeOutSignalReg = 0x1000;       /* Set A10 Ack       */
    *sfi.VmeIrqMaskReg  = intMask;            /* Enable Source  */
    *sfi.writeVmeOutSignalReg = 0x10000000;   /* Clear A10 Ack     */
  } else {
    *sfi.VmeIrqMaskReg  = intMask;            /* Enable Source  */
  }
}

static inline unsigned long 
sfittype()
{
 unsigned long tt;
#ifdef TRIG_SUPV
/* Read Trigger type from TS AUX port */
  sfiAuxRead(&tt);                               /* read data from AUX port */
  tt = ((tt&0x3c)>>2);
#else
/* Set default trigger type to 1 */
  tt=1;
#endif
  return(tt);
}

static inline int 
sfittest()
{
  int ii=0;
  /* NOTE: See that internal VME IRQ is set on the SFI*/
  ii = (((*sfi.VmeIrqLevelAndVectorReg) & 0x4000) != 0);
  return(ii);
}

