/*-----------------------------------------------------------------------------
 * Copyright (c) 1991,1992 Southeastern Universities Research Association,
 *                         Continuous Electron Beam Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
 * Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
 *-----------------------------------------------------------------------------
 * 
 * Description:
 *	Fastbus Standard Routines Library, level 1 subroutines
 *	
 * Author:  Chip Watson, CEBAF Data Acquisition Group
 *
 * Revision History:
 *   $Log: fb_fscc_sys.c,v $
 *   Revision 1.1.1.1  1996/08/21 19:20:38  heyes
 *   Imported sources
 *
*	  Revision 1.3  94/06/16  14:40:46  14:40:46  abbottd (David Abbott)
*	  "Changed back to previous version"
*	  
*	  Revision 1.1  94/03/15  12:00:37  12:00:37  heyes (Graham Heyes)
*	  Initial revision
*	  
 *	  Revision 1.6  1993/05/13  15:19:06  watson
 *	  minor edits to remove compiler warnings
 *
 *	  Revision 1.5  1993/05/11  12:36:43  heyes
 *	  renamed sysIntEnable etc
 *
 *	  Revision 1.4  1993/05/10  13:57:43  heyes
 *	  removed sysIntEnable
 *
 *	  Revision 1.3  1993/04/28  16:41:48  heyes
 *	  correct bus error handling
 *
 *	  Revision 1.2  1993/04/26  20:15:36  watson
 *	  ?
 *
 *	  Revision 1.1  1993/04/14  20:40:33  watson
 *	  Initial revision
 *
 *	  Revision 1.4  1992/09/18  20:14:03  watson
 *	  (FNAL) int disable routine, completion routine, IPL support
 *
 *	  Revision 1.3  1992/08/26  14:45:42  watson
 *	  (FNAL) fixed name of error code
 *
 *	  Revision 1.2  1992/07/17  15:00:57  watson
 *	  switched back to reporting fsr errors in fb_errno
 *
 *	  Revision 1.1  1992/07/15  19:11:48  watson
 *	  Initial revision
 *
 *
 *	This file contains low level routines to be called by the Fastbus
 *	standard routines to perform Fastbus actions on the FSCC (PC3). 
 *	Only those routines for which there is NOT a macro version (i.e.
 *	those for which performance is not an issue) are included here.
 *	See also fb_fscc_1.c and fb_fscc_macro.h.  Most of the routines in
 *	this file are in fact routines internal to the layer 1 routines.
 *
 *
 *  Public Routines:
 *  ---------------
 *      fb_init_1()		initialize hardware, install handlers
 *      fb_task_init_1()	perform all task-specific initialization
 *
 *
 *  Private Routines: (used by layer 1 only)
 *  ----------------
 *	fscc_fbhw_init()	initialize hardware
 *	fscc_bus_error(...)	bus error handler
 *	fscc_error_isr()	error interrupt service routine
 *	fscc_error_decode(...)	decode error information into standard
 *
 *	fbSysIntEnable(which)	MFP manipulations (will move to sysLib.c)
 *	fbSysIntDisable(which)	MFP manipulations (will move to sysLib.c)
 */

/******************************** Globals ***********************************/

#include "vxWorks.h"
#include "iv.h"
#include "mfp.h"
#include "regs.h"
#include "sigLib.h"
#include "semLib.h"
#include "setjmp.h"
#include "intLib.h"		/* for intConnect */

#include "fb_fscc_def.h"

#include "fb_types.h"
#include "fb_error.h"
#include "fb_errtxt.h"

STATUS fbSysIntEnable(int level);

typedef struct berrFrame *berrFramePtr;

typedef struct berrFrame {
  short sr;
  long  pc;
  short vector;
  short internal1;
  short ssw;
  short pipeC;
  short pipeB;
  long  faultAddr;
} berrFrm;

berrFramePtr  fbBerrFrame;
long fltAddr;
long frame[100];
long cpix;

ULONG fb_errno;			/* global error code for layers 1 & 2 */
SEM_ID fb_port_sem;		/* semaphore for FASTBUS port allocation */

/* port allocation:  semaphore and ID of allocating task */
int 			fb_alloc_task;
/* FB_environment_id 	fb_pending_eid; */

int fscc_csr[32];		/* 0 = ID, 8 = arbitration level, 
				   9,1c-1f timers */
int fscc_error;			/* error indicator from int handler */
int fscc_error_seq;		/* last good sequencer status */
int fscc_error_slv;		/* latched slave status */
int fscc_int_count;		/* count of interrupts */
int fscc_berr_count;		/* count of bus errors */
int fscc_seq_buf_action;	/* type of last action called w/ a seq buf */
int fscc_frlen_info;		/* auxilliary info for frlen */

unsigned char fscc_imra;	/* interrupt mask a */
unsigned char fscc_imrb;	/* interrupt mask b */

int fb_init_flag;		/* flag to say we are initialized */

/***************************** Function Prototypes ***************************/

int fscc_bus_error();
void fscc_error_isr();
void fscc_request_isr();
int  fscc_fbhw_init();
/* extern void fb_completion_isr(); */
void fscc_error_decode(int noarb,int nopa,int nosa,
		       int noda,int holdas,int hold);

/*************************** FB_TASK_INIT_1 *****************************
 *									*
 *  	Performs all task-specific intialization        :		*
 * (no longer needed, kept for backward compatibility)                  *
 *									*
 ************************************************************************/

void fb_task_init_1()
  {


  }

/***************************** FB_INIT_1 ********************************
 *									*
 *  1. Initialize FSCC Fastbus hardware					*
 *  2. Install an interrupt handler (for other Fastbus errors)		*
 *  3. Clear error flags & counters					*
 *									*
 ************************************************************************/

void fb_init_1()
{
  int err;

  /*  install signal handler for BUS ERROR signal  */

  *(FUNCPTR *) ((int) intVecBaseGet() + (int) IV_BUS_ERROR) = fscc_bus_error;

  err = fscc_fbhw_init();	/* initialize fastbus hardware */
  if (err!=OK) {
    fb_errno = FB_ERR_ERROR;
    return;
  }


				/* install fastbus error interrupt handler */
  err = intConnect( (VOIDFUNCPTR *)(INUM_TO_IVEC(0x43)), 
		    (VOIDFUNCPTR) fscc_error_isr, 0);
  if (err!=OK) {
    fb_errno = FB_ERR_ERROR;
    return;
  }
  fbSysIntEnable(3);		/* turn on interrupt in MFP */
  PP2_PCDR = FSCC_M_LTO | FSCC_M_STEN; /* enable short time out (AK/DK) */

/*  err = intConnect( INUM_TO_IVEC(0x40), fb_completion_isr, NULL );
  if (err!=OK) {
    fb_errno = FB_ERR_ERROR;
    return;
  }
  fbSysIntEnable( 0 ); */		/* turn on interrupt in MFP */

				/* install fastbus request interrupt handler */
  err = intConnect( (VOIDFUNCPTR *) INUM_TO_IVEC(0x41), 
		    (VOIDFUNCPTR) fscc_request_isr, 0);
  if (err!=OK) {
    fb_errno = FB_ERR_ERROR;
    return;
  }
  fbSysIntEnable( 1 );		/* turn on interrupt in MFP */

  fb_port_sem = semBCreate ( SEM_Q_FIFO, SEM_FULL );
  fscc_error = 0;
  fscc_int_count = 0;
  fscc_berr_count = 0;

#define FB_FB_ID 0x91010000
/*  FB_SET_CSR( 0, FB_FB_ID );
  FB_SET_CSR( 8, FPARBL_DEFAULT );
  FB_SET_CSR( 9, ( (!FPDLOTDEF << 4) | (!FPDWTTDEF << 5) | 
		   (!FPDAKTDEF << 6) | (!FPDDKTDEF << 7) ) );
  FB_SET_CSR( 0x1c, FPLOT_DEFAULT );
  FB_SET_CSR( 0x1d, FPWTT_DEFAULT );
  FB_SET_CSR( 0x1e, FPAKT_DEFAULT );
  FB_SET_CSR( 0x1f, FPDKT_DEFAULT );
*/
  fb_init_flag = 1;
}

/******************************* FSCC_FBHW_INIT *******************************
 *
 *  1. Initialize parallel ports (set bit direction and deassert all signals)
 *  2. Reset sequencer, fifo's, error status bits
 *  3. Initialize OPORT, HCR, APORT
 *  4. Check sequencer w/ NOP
 *
 ******************************************************************************/

int fscc_fbhw_init()
{
/*  attempt a graceful deassertion of FASTBUS lines  */
/*
  PP2_TCR = 0;
  PP2_TSR = 1;
  PP2_PGCR = 0;
  PP2_PSRR = 0;
  PP2_PACR = 0x80;
  PP2_PBCR = 0;
  PP2_PADDR = PP2_PADDR_DEF;
  PP2_PADR &= ~FSCC_M_FDRB;
  PP2_PADR &= ~FSCC_M_FDGK;
  PP2_PADR &= ~(FSCC_M_FDSS0 | FSCC_M_FDSS1 | FSCC_M_FDSS2);
  PIT_PORT_DEF(PP2_PB);
  PIT_PORT_DEF(PP2_PC);
  PP2_PGCR = 0x24;
*/

  PIT_INIT(PP1_);		/* initialize parallel ports  */
  PIT_INIT(PP2_);		/* (mode and directions */
  FSCC_SRESET;			/* reset sequencer & errors */
  PP2_CPRH = 0; PP2_CPRM = 0xff; PP2_CPRL = 0xff; /* long timer = 255*50ns */
  PP2_TCR = 0xf3;		/* use TOUT, TIN (start), CLK */
  PP1_PCDR = ~(FSCC_M_PRS|FSCC_M_DRS|FSCC_M_SRS|FSCC_M_COPYEN);
  PP1_PCDR = ~FSCC_M_COPYEN;	/* reset fifo's (toggle) */

  HCR_CSR = 4;			/* HCR reset & hold */

  OPORT_CSR1 = 4;		/* take OPORT offline */
  OPORT_CSR1 = 8;		/* reset OPORT */
  OPORT_CSR1 = 0;		/* OPORT HOLD */

  HCR_CSR = 2;			/* enable global word counter */

  APORT_DR = 0xff;		/* aux parallel port set to known state */

  FSCC_SEQS_RESET;		/* reset sequencer status */
  FPNOP;			/* check sequencer w/ null instruction */
  FSCC_WAIT;
  if (FSCC_STATUS!=12) {	/* does sequencer respond OK? */
    return(!OK);
  }
  if (FSCC_STATUS!=0) {		/* is sequencer now idle? */
    return(!OK);
  }
  return(OK);
}

/******************************* FSCC_BUS_ERROR *******************************
 *
 *  Bus error handler:  if (1st error) { if (arbitration error) error = -1 }
 *                      reset sequencer
 *
 ******************************************************************************/

int fscc_bus_error()
  {
  asm("unlk a6");
  asm("movl sp,_fbBerrFrame");
  for (cpix=0;cpix<40;cpix++) frame[cpix] = ((long *) fbBerrFrame)[cpix];
  
  fbBerrFrame->ssw &= 0xff; /* get special status word, 
				      remove retry bit */
  /* If address != fb port then call normal berr handler */

  fltAddr = fbBerrFrame->faultAddr;

  if ((fbBerrFrame->faultAddr & 0xfff00000) != 0x00600000) 
    {
      asm("jmp _excStub");
    }
  else 
    {
      fscc_berr_count++;	/* keep count of entries */
      if( fscc_error == 0 ) {	/* only report 1st error */
	if( FSCC_CSEQS == 1 ) { /* sequencer hung on FPARB? */
	  fscc_error = -1;
	}
	fscc_error = -1;
      }
      fscc_fbhw_init();
      
      /* FSCC_SRESET;*/		/* reset sequencer on any error */
    }
  asm("rte");
  return(0);
  
  }


/******************************* FSCC_ERROR_ISR *******************************
 *
 * Error handler:  1. if (first) determine source of error and save
 *                 2. reset sequencer
 *
 ******************************************************************************/

void fscc_error_isr()
{
  unsigned char tmp;
  fscc_int_count++;
  if (fscc_error==0) {		/* only report 1st error */
    if (FSCC_STO) { 		/* short timeout? */
      if (!FSCC_FRAK)
	fscc_error=-2;		/* no AK, so AK timeout */
      else 
	fscc_error=-3;		/* else DK or AK release timeout */
    }
    else if (FSCC_DFF) {
      fscc_error=-4;		/* data fifo overrun */
      return;
    }
    else
      fscc_error=-5;		/* must have been a bad slave status */
    tmp = FSCC_STATUS;		/* get last good sequencer status */
    while (tmp==12) tmp = FSCC_STATUS; /* ignore NOP status */
    fscc_error_seq = tmp;	/* first non-NOP status */
    while ((tmp!=15)&&(tmp!=0)) {
      fscc_error_seq=tmp;	/* last good non-NOP status */
      tmp=FSCC_STATUS;
      while (tmp==12) tmp = FSCC_STATUS; /* get next non-NOP status */
    }
    if (fscc_error_seq==15) fscc_error_seq=0; /* no useful info */
    fscc_error_slv = FSCC_SS;	/* get last latched slave status */
  }
/*  FSCC_SRESET;	*/		/* reset sequencer on any error */
  fscc_fbhw_init();		/* fully reset hardware on any error */
  PP2_PCDR = FSCC_M_LTO | FSCC_M_STEN; /* re-enable short time out (AK/DK) */
}

/****************************  fscc_error_decode ******************************
 *
 *  decode error information cached by the isr
 *
 ******************************************************************************/

#define FB_MAP_SS_ERROR(val,prefix,source) \
      switch((source)) { \
      case FSCC_M_SS1: \
	(val) = prefix##SS1; \
	break; \
      case FSCC_M_SS2: \
	(val) = prefix##SS2; \
	break; \
      case FSCC_M_SS3: \
	(val) = prefix##SS3; \
	break; \
      case FSCC_M_SS4: \
	(val) = prefix##SS4; \
	break; \
      case FSCC_M_SS5: \
	(val) = prefix##SS5; \
	break; \
      case FSCC_M_SS6: \
	(val) = prefix##SS6; \
	break; \
      case FSCC_M_SS7: \
	(val) = prefix##SS7; \
	break; \
      }

void fscc_error_decode(int noarb,int nopa,int nosa,
		       int noda,int holdas,int hold)
{
  if (fscc_error==-1)		/* sequencer was stuck at arbitration */
    fb_errno = FB_ERR_LONG_TIMEOUT; /* arbitration timeout */
  else if (fscc_error==-2) {	/* no FRAK */
    if (nopa)
      fb_errno = FB_ERR_NO_ADDRESS_CONNECTION;
    else
      fb_errno = FB_ERR_AK_TIMEOUT;
  }
  else if (fscc_error==-3) {	/* STO */
    switch (fscc_error_seq) {	/* branch on last sucessful operation */
    case 0: case 1: case 3:	/* nothing or ARB or PA */
      if (!nosa) {
	fb_errno = FB_ERR_SAD_TIMEOUT;
	break;
      }
    case 6:			/* SA succeeded */
      if (!noda) {
	fb_errno = FB_ERR_DK_TIMEOUT;
	break;
      }
    default:			/* data transfer must have succeeded */
      fb_errno = FB_ERR_ADDRESS_RELEASE_TIMEOUT;
    }
  }
  else if (fscc_error==-4)	/* DFF */
    fb_errno = FB_ERR_WRITE_FIFO;
  else if (fscc_error==-5) {	/* bad SS */
    switch (fscc_error_seq) {	/* branch on last successful operation */
    case 0: case 1: case 3:	/* nothing or ARB or PA */
      if (!nopa) {
	FB_MAP_SS_ERROR(fb_errno,FB_ERR_PRIMARY_ADDRESS_,fscc_error_slv);
	break;
      }
    case 6:			/* SA succeeded */
      if (!nosa) {
	FB_MAP_SS_ERROR(fb_errno,FB_ERR_SECONDARY_ADDRESS_,fscc_error_slv);
	break;
      }
    default:			/* data transfer must have succeeded */
      if (!noda) {
	FB_MAP_SS_ERROR(fb_errno,FB_ERR_DATA_,fscc_error_slv);
	break;
      }
    }
  }
}


/***************************** FSCC_REQUEST_ISR *******************************
 *
 * Request handler:  1. if self-access, return error for now
 *                   2. if slave access, permit limited transactions
 *		     3. if RB, reset seq, clear errors, wait with NOP!!!!
 *
 ******************************************************************************/

#define FRMS	( ((PP1_PADR) >> 2) & 7)
#define FRRD	( (PP1_PADR) & 1 )

void fscc_request_isr()
  {
unsigned long nta;

/*  if( (PP1_PADR & FSCC_M_FRDY) ) we are only master on this bus !! */

    {
    fscc_error = -2;			/* if self-access, AK timeout */
    B_REL_ = 0;				/* release address */
    /*  reset sequencer  */
    fscc_fbhw_init();
    return;
    }

  if( !(PP1_PADR & FSCC_M_FSLV) )
    {
    while ( FRMS != 2);     /* wait for secondary addr cycle */
    if ( FRRD )       /* read?  return 0 */
      SD_OPT = nta;
    else                /* write?  accept value */
      nta = SD_IPT;

    while ( FRMS == 2);

/*  data cycle */
    if ( FRRD )   /* read */
      if ( /* nta >= 0 && */ nta < 32 ) /* nta is always positive */
        SD_OPT = fscc_csr[nta];
    else
      if( nta >= 1 && nta < 32 )
        fscc_csr[nta] = SD_IPT;

    B_REL_ = 0;
    }

  else
    /*  OK, this is for handling RB from other masters  */
    {
      /*  reset sequencer  */
      fscc_fbhw_init();
    }

  }

/***************************** fbSysIntEnable ***********************************
 *
 * Handle the MFP to enable interrupts (will be part of sysLib.c)
 *
 ******************************************************************************/

STATUS fbSysIntEnable (intLevel)
     int intLevel;		/* interrupt level to enable */
{
#define MFP_BASE 0x480000
  register char *cr = (char *) MFP_BASE;
  if (intLevel < 4) {			/* 0-3 on bits 0-3 of IxRB */
    cr[MFP_IMRB] |= (1<<intLevel);
    cr[MFP_IERB] |= (1<<intLevel);
  } else if (intLevel < 6) {		/* 4-5 on bits 6-7 of IxRB */
    cr[MFP_IMRB] |= (1<<(intLevel+2));
    cr[MFP_IERB] |= (1<<(intLevel+2));
  } else {				/* 6-7 on bits 6-7 of IxRA */
    cr[MFP_IMRA] |= (1<<intLevel);
    cr[MFP_IERA] |= (1<<intLevel);
  }
  return(OK);
}

/***************************** fbSysIntDisable **********************************
 *
 * Handle the MFP to disable interrupts (will be part of sysLib.c)
 *
 ******************************************************************************/

STATUS fbSysIntDisable (intLevel)
     int intLevel;		/* interrupt level to disable */
{
#define MFP_BASE 0x480000
  register char *cr = (char *) MFP_BASE;
  if (intLevel < 4) {			/* 0-3 on bits 0-3 of IxRB */
    cr[MFP_IMRB] &= ~(1<<intLevel);
    cr[MFP_IERB] &= ~(1<<intLevel);
  } else if (intLevel < 6) {		/* 4-5 on bits 6-7 of IxRB */
    cr[MFP_IMRB] &= ~(1<<(intLevel+2));
    cr[MFP_IERB] &= ~(1<<(intLevel+2));
  } else {				/* 6-7 on bits 6-7 of IxRA */
    cr[MFP_IMRA] &= ~(1<<intLevel);
    cr[MFP_IERA] &= ~(1<<intLevel);
  }
  return(OK);
}


