/*
 * dd_user.c :  general user calls for the dd system (start with ddu_).
 *           :  These are the routines that the kofia <--> dd 
 *           :  void ddxxxx_(int *status) "interface routines" in 
 *           :  kofia_dd.c should call.
 * Author    :  C.Witzig
 * Date      :  Apr 1, 1992 (no joke!)
 * Mods      :  May 29: Add ddu_rel_fev(struct fifo_entry fev)
 * 
 *           :  May 28, 1994: add cleanup procedure: atexit(dd_cleanup)
 *                            that makes sure that exiting processes
 *                            do detach from the system.
 *              May 18, 1995: fix dd_set_brc when called without being
 *                            attached to a fifo.
 *              July 11, 1995: speeding things up... 
 *                             1. ddu_attached:
 *                             static variables for dd_filename and
 *                             call getenv only once.
 *                             2. all ddu_xxx routines have a
 *                             call to ddu_attached at the beginning
 *                             remove this and only call ddu_attached
 *                             if the dbfi_xxx call failed, thus
 *                             only check whether the DD system is
 *                             still there after a system call failed.
 *                             3. add fast version of ddu_attached
 *                             called dd_attached which is only
 *                             used internally to check whether 
 *                             dd_dcom is still there.
 *              July 14, 1995: change dd_shutdown to properly 
 *                             end after all the changes in ddu_attached
 *
 *              July 31, 1995: add ddu_set_dd_wait in order to allow 
 *                             processes to change the read mode any time
 *                             they want
 *              Jan 2, 1996:   add ddu_get_free_buffers to get the 
 *                             number of free buffers in the system
 *                             add ddu_creq_fev for conditional event request
 *
 *              Feb 15, 1996:  add ddu_fifo_exist(char *fifoname) which returns
 *                             0 if fifo does not exist.
 *              Feb 15, 1996:  added exclusive requirement for ddu_init:
 *                             If exclusive <> 0 then the process creates its fifo only
 *                             if it doesn't exist yet.
 *              Feb 28, 1996:  added ddu_set_brccmd(int cmd, BRCCMD_FNC brc_cmd_fcn)
 *              Mar  5, 1996:  added ddu_set_netswap(int swap_flag)
 */

#include <stdio.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>

#include <unistd.h>
#include <sys/types.h>

#include <fifo.h>
#include <db.h>
#include <dd_dcom.h>
#include <dd_network.h>
#include <dd_user_entries.h>
#include <dd_signal_def.h>
#include <dd_filter.h>
#include <dlfcn.h>
#include <dd_log.h>

#define TRUE 1
#define FALSE 0

/*
 *                      dd  USER  CALLS :
 *--------------------------------------------------------------------
 *
 * This file contains all the routines that are callable from any 
 * user process that wants to join the dd system.
 *
 * 1. INITIALISATION ROUTINES:
 * ===========================
 * ddu_init(char *fname, struct fifo_mode fmode)
 *         initialisation routine, which has to be called once at
 *         the beginning.
 * ddu_close()
 *         Has to be called when the process wants to end (on purpose or
 *         not!). It detaches the process from the dd and enables that
 *         the fifo it was attached to it becomes free for other processes.
 *
 * ddu_start()
 *         Has to be called before the process can actually get data
 *         from its fifo.
 * ddu_stop()
 *         Disables data taking. 
 * 
 *
 * 2. ROUTINES FOR HANDLING EVENTS:
 * ================================
 * Then there are two routines that can be called to handle events. 
 * ddu_put_fev(struct fifo_entry fev)
 *         Puts the fifo event into the corresponding fifo.
 * ddu_get_fev(struct fifo_entry *fev)
 *         Gets the next fifo event from corresponding fifo.
 * ddu_rel_fev(struct fifo_entry fev)
 *         Releases the fifo event fev, i.e. puts it back directly
 *         into INPUT_FIFO
 *
 *
 * NOTE:
 * IF ONE OF THESE CALLS FAIL MISERABLY, THEN THE DD STATUS IS SET TO 
 *                         DD_STATUS_FKDUP
 */

static int  dd_active   = FALSE;
static int  dd_shutdown = FALSE;
static char dd_filename[1024] = "";

/*
 * declarations for implementation of 
 * user-defined event selection routine
 */
FILTERFUNCPTR ddu_filter;
void *ddu_filter_handle;
 
static
int ddu_filter_register()
/*
 * ddu_init calls this routine to find the user-defined shared library
 * called libdd_filter.so in the location $CODA_LIB. This library contains
 * a single function - ddu_filter_fev. Once this thing is loaded,
 * we're on our way to a whole new horizon of event selection.
 */
{
    char libname[100], *clib;
    void  *ret;

    if ((clib = getenv("CODA_LIB")) == NULL) {
	printf("ddu_filter_register: Specify CODA_LIB!\n");
	sprintf(dd_log_msg,"ddu_filter_register: Specify CODA_LIB!\n");
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
    }
    if (strlen(clib) > 83) {
	sprintf(dd_log_msg,"ddu_filter_register: CODA_LIB is too long!\n");
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
    }
    strcpy(libname,clib);
    strcat(libname,"/libdd_filter.so");
    
#if defined(__sun) || defined(__linux)
    ddu_filter_handle = dlopen(libname, RTLD_NOW | RTLD_GLOBAL);
    if (ddu_filter_handle == 0) {
	printf("ddu_filter_register: dlopen failed: %s\n", dlerror());
	sprintf(dd_log_msg,"ddu_filter_register: dlopen failed: %s\n", dlerror());
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
    }

    ret = dlsym(ddu_filter_handle, "ddu_filter_fev");
    ddu_filter = (FILTERFUNCPTR) ret;
    if (ret == NULL) {
	printf("ddu_filter_register: ddu_filter_fev routine NOT found: %s\n", dlerror());
	sprintf(dd_log_msg,"ddu_filter_register: ddu_filter_fev routine NOT found: %s\n", dlerror());
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
    } 
#else
    sprintf(dd_log_msg,"ddu_filter_register: dynamic loading not supported\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
#endif

    return 0;
}

/*
 * shutdown and cleanup routines
 */

void dd_shutdown_alert(int sig)
{
  signal(DD_SHUTDOWN_SIGNAL, dd_shutdown_alert);
  dd_dcom->dd_proc[my_own_ddpid].status = DD_PROC_STATE_EXIT;
/*  sprintf(dd_log_msg,"dd_shutdown_alert seen - status %d\n",dd_dcom->dd_proc[my_own_ddpid].status); */
  dd_shutdown = TRUE;

  return;
}

void dd_cleanup()
/*
 * called atexit: - make sure that we are not a child that is also attached...
 *                - calls ddu_stop if ddu_start has been called
 *                  but not ddu_stop
 *                - calls ddu_close if ddu_init has been called
 *                  but not ddu_close
 */
{
  int i;

  if ( dd_dcom ) {
    if ( dd_dcom->dd_nmstr == 1 ) {
      for ( i=0; i<DD_MAX_NMSTR; i++) {
	if ( dd_dcom->dd_master[i].pid == getpid() ) {
	  sprintf(dd_log_msg,"dd_cleanup: last master - call dds_close\n");
	  dd_log(DD_INFO,dd_log_msg);
	  dds_close(0);
	  return;
	}
      }
    }
    if ( dd_dcom->dd_nmstr == 0 ) {
      dds_close(1);
      return;
    }
    if ( dd_dcom->dd_proc[my_own_ddpid].pid != getpid() )
      return;
  }

  if ( dd_active ){
    ddu_stop();
/*    sprintf(dd_log_msg,"proc %d calling ddu_stop atexit\n",getpid());
    dd_log(DD_INFO, dd_log_msg); */
  }

  if ( dd_dcom || ( ddn_server_running() && 
		   (i_am_dd2tcp_producer || i_am_dd2tcp_consumer) ) ) {
    ddu_set_brc(DD_BRC_OFF);
    ddu_close();
/*    sprintf(dd_log_msg,"proc  %d calling ddu_close atexit\n",getpid());
    dd_log(DD_INFO, dd_log_msg); */
  }


  if ( dd2tcp ) {
    if ( ddn_server_running() ) 
      ddnc_exit();
  }

  return;
}


int dd_attached()
{
/*
 * fast version of ddu_attached: 
 * checks only whether server is running and attached
 * to dcom.
 * Is only used internally and no time consuming 
 * checks on whether the DD file in /tmp etc
 * is still there
 */

  if ( dd2tcp )
    return ddn_server_running();

  if ( dd_shutdown  && ( dd_dcom != NULL ) ) {
    ddu_close();
    dd_shutdown = FALSE;
  }

  return ( (dd_dcom) ? TRUE : FALSE );
}


int ddu_attached()
{
/*
 * see whether we are still attached to DD system
 * if so, check dcom exists
 */

  if ( dd2tcp )
    return ddn_server_running();

  if ( dd_dcom ) {
    if ( dd_filename[0] == '\0' ) 
      dds_get_filename(dd_filename);
    if ( ! dd_dcom_exists(dd_filename) ) {
      dd_dcom = NULL;
      sprintf(dd_log_msg,"ddu_attached: DD system disappeared !!\n");
      dd_log(DD_ERROR, dd_log_msg);
    }
  }

  if ( dd_shutdown  && ( dd_dcom != NULL ) ) {
    ddu_close();
    dd_shutdown = FALSE;
  }
  
  return ( (dd_dcom) ? TRUE : FALSE );
}

int ddu_exists()
/*
 * Determines if the dd system exists.
 * 1 = exists, 0 = not, -1 = error
 */
{
  FILE *fp;
  
  if (dd2tcp)
    return ddnc_exists();

  dds_get_filename(dd_filename);
 
  if ((fp = fopen(dd_filename,"r")) == NULL) {
    return 0;
  }

  fclose(fp);
  return 1;
}


int ddu_attach()
/*
 * attaches to the DD system  but does not create any fifo.
 * is called from ddu_init or directly from user processes
 * that do not have any fifos (e.g. monitoring processes).
 * Is ONLY called by processes on the same host as DD 
 * system.
 */
{
  char *p2c;
  FILE *fp;
  
  if ( ddu_attached() ) 
    return 0;
 
  atexit(dd_cleanup);

  dds_get_filename(dd_filename);

  if (ddn_set_hostnames(dd_filename) < 0){
    sprintf(dd_log_msg,"ddu_attach: error in ddn_set_hostnames\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( dd2tcp ) {
    int status;
    if ( ddn_server_running() == FALSE ) {
      if ( ddn_client_startup(dd_hostname) ) {
	return -1;
      }
    }
    if ( (status = ddnc_attach()) ) 
      ddnc_exit();
    return status;
  }

  if ( (fp = fopen(dd_filename,"r")) == NULL ) {
    sprintf(dd_log_msg,"ddu_attach: DD system does not exist %s %s \n",dd_filename,strerror(errno));
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  fclose(fp);
  
  if ( (dd_dcom = dd_dcom_attach(dd_filename)) == NULL ){
    sprintf(dd_log_msg,"ddu_attach: error in dd_dcom_attach\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd_dcom_lock() ){
    sprintf(dd_log_msg,"ddu_attach: error in dd_dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if(!ddu_exists()) {
    sprintf(dd_log_msg,"ddu_attach: dd gone, 1\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( fifos_attach(dd_filename) ){
    sprintf(dd_log_msg," ddu_attach: fifos_attach failed \n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg,"ddu_attach: error in dd_dcom_unlock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_cleanup();
    dd_dcom->dd_status  = DD_STATUS_FKDUP;
    return -1;
  }
  
  if(!ddu_exists()) {
    sprintf(dd_log_msg,"ddu_attach: dd gone, 2\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( db_init(dd_filename, dd_dcom->dd_config.db_maxnbev, dd_dcom->dd_config.db_maxevsize) ) {
    sprintf(dd_log_msg," ddu_attach: db_init failed \n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg,"ddu_attach: error in dd_dcom_lock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_cleanup();
    return -1;
  }

  /* Need to add the following reset of my_own_* 
   * If a dd system dies and is restarted, and if a process which is
   * attached reattaches, then when dd_dcom_addproc(NULL,NULL) is called,
   * unfortunately "my_own_fifo" has a non-negative value "left over" from
   * before the dd recreation. The routine dd_dcom_addproc will then try
   * to find the length of a null string. CTCTCT
   */
  my_own_ddpid = -1;
  my_own_fifo  = -1;

  if(!ddu_exists()) {
    sprintf(dd_log_msg,"ddu_attach: dd gone, 3\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd_dcom_addproc(NULL,NULL) ) {
    sprintf(dd_log_msg,"ddu_attach: error in dd_dcom_addproc\n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg,"ddu_attach: error in dd_dcom_lock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_cleanup();
    return -1;
  }

  if ( dd_dcom_unlock() ){
    sprintf(dd_log_msg,"ddu_attach: error in dd_dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    dd_cleanup();
    return -1;
  }
  return 0;
}


int ddu_init(char *fname, struct fifo_mode fmode)
/*
 * Called when a process intends to attach to the dd system.
 * 1. Get the host names.
 * 2. If client call client initialisation and return immediately.
 * 3. else:
 *    1. Attaches the shared memory and semaphores.
 *    2. Attaches the shared data buffer db.
 *    3. If fname is not equal to INPUT then we assume that the
 *       process wants to be a consumer (but still can be a producer). So he
 *       creates his own (DYN) fifo and obtains the fifo number (my_own_fifo),
 *       which is used for subsequent calls to its fifo
 * Note: fname: Here we split between hostname and fifoname searching
 *       for a separating ':' (if there is any).
 */
{
  int max_len;
  char *p2c, fifoname[FIFO_NAME_LEN];

  dds_get_filename(dd_filename);

  if (ddn_set_hostnames(dd_filename) < 0){
    sprintf(dd_log_msg,"ddu_init: error in ddn_set_hostnames\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  strncpy(fifoname,fname,FIFO_NAME_LEN);
  fifoname[FIFO_NAME_LEN-1] = '\0';

  if ( dd2tcp ){
    if ( (ddn_server_running()) == FALSE  ){
      if ( ddn_client_startup(dd_hostname) )
	return -1;
      atexit(dd_cleanup);
    }
    return ( (ddnc_init(fname, fmode) ) ? -1 : 0);
  }
  
  if ( ddu_attach() ) {
    sprintf(dd_log_msg,"ddu_init: cannot attach to DD system\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( fifoname == NULL ) {
    return 0;
  }
  if ( fmode.prescale <= 0 ) {
    fmode.prescale = 1;
    sprintf(dd_log_msg,"ddu_init: bad prescale value: %d set to 1\n",fmode.prescale);
    dd_log(DD_ERROR, dd_log_msg);
  }

/* set the mode to multi user */
  if ( fmode.suser != FMODE_SINGLE_USER )
    fmode.suser = FMODE_MULTI_USER;
  
  dbfi_link_signal();
  ddu_link_usersignal();
  
  if ( dd_dcom_lock() ){
    sprintf(dd_log_msg,"ddu_init: error in dd_dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if(!ddu_exists()) {
    sprintf(dd_log_msg,"ddu_init: dd gone, 1\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

#if defined(__sun) || defined(__linux)
  if (ddu_filter_register() < 0) {
    sprintf(dd_log_msg,"ddu_init: error in ddu_filter_register\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
#endif  

  if(!ddu_exists()) {
    sprintf(dd_log_msg,"ddu_init: dd gone, 2\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( (my_own_fifo = fifo_make(fifoname, "DYN", fmode )) < 0){
    sprintf(dd_log_msg," ddu_init: can't make my own fifo\n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg,"ddu_init: error in dd_dcom_lock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_cleanup();
    return -1;
  }

  if(!ddu_exists()) {
    sprintf(dd_log_msg,"ddu_init: dd gone, 3\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd_dcom_addproc(fifoname, fmode.wait) ){
    sprintf(dd_log_msg," ddu_init: error in dd_dcom_addproc \n");
    dd_log(DD_ERROR, dd_log_msg);
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg,"ddu_init: error in dd_dcom_lock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_cleanup();
    return -1;
  }
  
  if ( dd_dcom_unlock() ){
    sprintf(dd_log_msg,"ddu_init: error in dd_dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  return 0;
}


int ddu_close()
     /*
      * Called when a process wants to deattach from the dd system.
      * If client process then call init ddnc_close and return immediately.
      * When the process is not a producer, then
      * 1. Sets its fifo status to IDLE.
      * 2. Empties its fifo by moving the remaining events into the INPUT fifo.
      *    Note that dd_dcom should not be locked while emptiing the fifos.
      * 3. Removes the fifos and its associated data structures.
      */
{
  int iperm;
  
  if ( ! dd_shutdown ) {
    if ( ! ddu_attached() ) {
      sprintf(dd_log_msg,"ddu_close: dd system not attached \n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }
  /*
    if ( ! ddu_attached() ) {
    sprintf(dd_log_msg,"ddu_close: dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
    }
    */
  if ( dd2tcp ) {
    if ( ddn_server_running() ) {
      if ( (i_am_dd2tcp_producer) || (i_am_dd2tcp_consumer) ) 
	return ddnc_close();
      ddnc_exit();
      return 0;
    }
  }
  
  if ( dd_dcom_lock() ){
    sprintf(dd_log_msg,"ddu_close: error in dd_dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( my_own_fifo >= 0 ) {
    if (fifo_perm2empty(my_own_fifo,&iperm)){
      sprintf(dd_log_msg," ddu_close: error in fifo_perm2rem\n");
      dd_log(DD_ERROR, dd_log_msg);
      if ( dd_dcom_unlock() ){
	sprintf(dd_log_msg,"ddu_close: error in dd_dcom_lock\n");
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
      }
      dd_cleanup();
      return -1;
    }
    
    if (iperm){
      if (fifo_set_status(my_own_fifo,FSTAT_IDLE) ){
	sprintf(dd_log_msg,"ddu_close: cannot set my fifo to idle\n");
	dd_log(DD_ERROR, dd_log_msg);
	if ( dd_dcom_unlock() ){
	  sprintf(dd_log_msg,"ddu_close: error in dcom_lock\n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	dd_cleanup();
	return -1;
      }
      
      if (dbfi_empty_fifo(my_own_fifo) ){
	sprintf(dd_log_msg," ddu_close: cannot empty my_own_fifo \n");
	dd_log(DD_ERROR, dd_log_msg);
	/*	dd_dcom->dd_status  = DD_STATUS_FKDUP; */
	if ( dd_dcom_unlock() ){
	  sprintf(dd_log_msg,"ddu_close: error in dcom_lock\n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	dd_error_inv();
	dd_cleanup();
	return -1;
      }
    }
    
    if (fifo_remove(my_own_fifo) ){
      sprintf(dd_log_msg," ddu_close: cannot remove my_own_fifo \n");
      dd_log(DD_ERROR, dd_log_msg);
      /*      dd_dcom->dd_status = DD_STATUS_FKDUP; */
      if ( dd_dcom_unlock() ){
	sprintf(dd_log_msg,"ddu_close: error in dcom_lock\n");
	dd_log(DD_ERROR, dd_log_msg);
	return -1;
      }
      dd_error_inv();
      dd_cleanup();
      return -1; 
      
    }
    my_own_fifo = -1;
  }
  
#if defined(__sun) || defined(__linux)
  /*
   * Add this line to close the shared lib with user routine
   * providing event selection logic. CTCTCT
   */
  dlclose(ddu_filter_handle);
#endif
  
  if ( dd_dcom_remproc() ){
    sprintf(dd_log_msg," ddu_close: error in dd_dcom_remproc \n");
    dd_log(DD_ERROR, dd_log_msg);
    /*    dd_dcom->dd_status = DD_STATUS_FKDUP; */
    if ( dd_dcom_unlock() ){
      sprintf(dd_log_msg,"ddu_close: error in dcom_lock\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
    dd_error_inv();
    dd_cleanup();
    return -1;
  }
  
  if ( dd_dcom_unlock() ){
    sprintf(dd_log_msg,"ddu_close: error in dcom_lock\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  dd_dcom = dd_dcom_detach();
  
  return 0;
}

int ddu_start()
{
  dd_active = TRUE;
  
  if ( ! ddu_attached() ) {
    sprintf(dd_log_msg,"ddu_start: dd system not attached\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( (i_am_dd2tcp_producer) || (i_am_dd2tcp_consumer) )
    return ddnc_start();
  
  if ( my_own_fifo != INPUT_FIFO ) {
    if (fifo_set_status(my_own_fifo,FSTAT_ACTIVE) ){
      sprintf(dd_log_msg,"ddu_start: cannot change fifo status to active \n");
      dd_log(DD_ERROR, dd_log_msg);
     dd_cleanup();
      return -1;
    }
  }
  
  dd_dcom_start();
  
  return 0;
}

int ddu_stop()
{
  if ( ! ddu_attached() ) {
    sprintf(dd_log_msg,"ddu_stop: dd system not attached\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( (i_am_dd2tcp_producer) || (i_am_dd2tcp_consumer) )
    return ddnc_stop();
  
  if ( my_own_fifo != INPUT_FIFO ){
    struct fifo_header fhdr;

    dd_dcom_lock();
    if ( fifo_get_header(my_own_fifo,&fhdr) ) {
      sprintf(dd_log_msg,"ddu_stop: error in fifo_get_header\n");
      dd_log(DD_ERROR, dd_log_msg);
    }
    else {
      if ( fhdr.fnproc_att == 1 ) {
	if (fifo_set_status(my_own_fifo,FSTAT_IDLE) ){
	  sprintf(dd_log_msg,"ddu_stop: cannot change fifo status to idle \n");
	  dd_log(DD_ERROR, dd_log_msg);
	  dd_cleanup();
	  dd_dcom_unlock();
	  return -1;
	}
      }
    }
    dd_dcom_unlock();
  }
  
  dd_dcom_stop();
  
  dd_active = FALSE;
  
  return 0;
}


int ddu_set_brc(int brc_flag)
{
  int status = 0;
  struct fifo_header fhdr;
  
  if ( ! ddu_attached() ) {
    sprintf(dd_log_msg,"ddu_set_brc: dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( (i_am_dd2tcp_producer) || (i_am_dd2tcp_consumer) )
    return ddnc_set_brc(brc_flag);
  
  if ( my_own_fifo < 0) {
    if ( brc_flag == DD_BRC_OFF )
      return 0;
    else{
      sprintf(dd_log_msg,"ddu_set_brc: no fifo attached\n");
      dd_log(DD_ERROR, dd_log_msg);
      return -1;
    }
  }
  
  dd_dcom_lock();
  if ( (brc_flag == DD_BRC_USMAIL) && (my_own_fifo != INPUT_FIFO) ){
    if ( (status = fifo_get_header(my_own_fifo,&fhdr)) ) {
      sprintf(dd_log_msg,"ddu_set_brc: error in fifo_get_header\n");
      dd_log(DD_ERROR, dd_log_msg);
    }
   else{
      dd_dcom->dd_proc[my_own_ddpid].brc_flag = brc_flag;
      if ( fhdr.fnproc_att != 1 )
	sprintf(dd_log_msg,"WARNING: broadcast fev received not garanteed!!!!\n");
    }
  }
  else
    dd_dcom->dd_proc[my_own_ddpid].brc_flag = brc_flag;
  
  dd_dcom_unlock();
  
  return status;
}

int ddu_set_brccmd(int cmd, DD_BRCCMD_FCN brc_cmd_fcn)
/*
 * User function to define the pointer to the function for 
 * the broadcast command cmd
 * This commands does not work over the network.
 */
{
  if ( ! ddu_attached() ) {
    sprintf(dd_log_msg,"ddu_set_brccmd:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( i_am_dd2tcp_producer || i_am_dd2tcp_consumer ){
    sprintf(dd_log_msg,"ddu_set_brccmd: cannot be called remotely\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( (cmd > DD_BRCCMD_MAXCMD) || (cmd < 0) ) {
    sprintf(dd_log_msg, "ddu_set_brccmd: only %d commands allowed - requested cmd %d\n",
	    DD_BRCCMD_MAXCMD, cmd);
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  dd_dcom->dd_proc[my_own_ddpid].brc_cmd[cmd] = brc_cmd_fcn;

  return 0;
}

int ddu_set_protection(int protection)
{
  if ( ! ddu_attached() ) {
    sprintf(dd_log_msg,"ddu_set_req_protection:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( i_am_dd2tcp_producer){
    sprintf(dd_log_msg,"producer cannot set protection\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  if ( i_am_dd2tcp_consumer)
    return ddnc_set_protection(protection);
  
  if ( my_own_fifo < 0 ) {
    sprintf(dd_log_msg,"ddu_set_protection: no fifo attached\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }
  
  return (fifo_set_protection(my_own_fifo, protection));
}

int ddu_set_dd_wait(int mode)
{
  
  if ( ! ddu_attached() ) {
    sprintf(dd_log_msg,"ddu_set_dd_wait: dd system not attached\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( (i_am_dd2tcp_producer) || (i_am_dd2tcp_consumer) )
    return ddnc_set_dd_wait(mode);

  dd_dcom_lock();
  dd_dcom->dd_proc[my_own_ddpid].dd_wait = mode;
  dd_dcom_unlock();

  return 0;
}

int ddu_set_netswap(int swap_flag)
{
  
  if ( ! ddu_attached() ) {
    sprintf(dd_log_msg,"ddu_set_netswap: dd system not attached\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  ddn_set_dataswap(swap_flag);

  if ( (i_am_dd2tcp_producer) || (i_am_dd2tcp_consumer) )
	 return ddnc_set_netswap(swap_flag);

  return 0;
}



int ddu_set_reqcnt(int cnt)
{
  if ( ! dd_attached ) {
    sprintf(dd_log_msg,"ddu_set_req_cnt:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_producer){
    sprintf(dd_log_msg,"producer cannot set reqcnt\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_consumer)
    return ddnc_set_reqcnt(cnt);

  if ( my_own_fifo < 0 ) {
    sprintf(dd_log_msg,"ddu_set_reqcnt: no fifo attached\n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  return (fifo_set_reqcnt(my_own_fifo, cnt));
}


int ddu_get_free_buffers()
{
  if ( ! dd_attached ) {
    sprintf(dd_log_msg,"ddu_get_free_buffers:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_producer || i_am_dd2tcp_consumer )
    return ddnc_get_free_buffers();

  return ( dbfi_get_free_buffers() );
}


int ddu_fifo_exist(char *fifoname)
{
/*
 * returns fifo_number if fifo exist, 0 if not.
 * can be called by anyone, who is attached (test on dd2tcp only)
 */
  if ( ! dd_attached ) {
    sprintf(dd_log_msg,"ddu_fifo_exist:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd2tcp )
    return ddnc_fifo_exist(fifoname);

  return ( dbfi_fifo_exist(fifoname) );
}

int ddu_fifo_active(char *fifoname)
{
/*
 * returns 0 if fifo is active, -1 if fifo doesn't exist, +1 if 
 * fifo exists but is not active.
 */
  if ( ! dd_attached ) {
    sprintf(dd_log_msg,"ddu_fifo_active:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( dd2tcp )
    return ddnc_fifo_active(fifoname);

  return ( dbfi_fifo_active(fifoname) );
}




int ddu_req_fev(int size, struct fifo_entry *p2fev)
{
  int status;
  struct fifo_entry fev1;

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"ddu_req_fev:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_producer ){
    if ( ( status = ddnc_req_fev(size, &fev1)) ){
      if ( status < 0 ) {
	sprintf(dd_log_msg,"ddu_req_fev: error in ddnc_req_fev\n");
	dd_log(DD_ERROR, dd_log_msg);
      }
      return status;
    }
  }
  else if ( i_am_dd2tcp_consumer ){
    sprintf(dd_log_msg,"ddu_req_fev: called as a consumer!!\n");
    dd_log(DD_ERROR, dd_log_msg);
    status = -1;
  }
  else{
    if ( (status = dbfi_req_fev(size,&fev1)) ){
      if (status < 0){
	if ( ! ddu_attached() ) {
	  sprintf(dd_log_msg,"ddu_req_fev: dd system not attached \n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	sprintf(dd_log_msg," ddu_req_fev: error in dbfi_req_fev\n");
	dd_log(DD_ERROR, dd_log_msg);
	dd_error_inv();
      }
      return status;
    }
  }
  
  *p2fev = fev1;

  return status;
}

int ddu_creq_fev(int size, struct fifo_entry *p2fev, int min_cnt)
{
  return ( ddu_get_free_buffers() < min_cnt ) ? 1 : ddu_req_fev(size, p2fev);

}


int ddu_get_fev(struct fifo_entry *fev)
/*
 * gets a fifo event from the dd
 * (at the moment no test on fifo status when reading....)
 */
{
  int status;
  struct fifo_entry fev1;

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"ddu_get_fev:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_producer ){
    sprintf(dd_log_msg,"ddu_get_fev: ddu_get_fev called as producer!!\n");
    dd_log(DD_ERROR, dd_log_msg);
  }
  else if ( i_am_dd2tcp_consumer){
    if ( ( status = ddnc_get_fev(&fev1)) ){
      if (status < 0) {
	sprintf(dd_log_msg," ddu_get_fev: error in ddnc_get_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
      }
      return status;
    }
  }
  else{
    if ( (status = dbfi_get_fev(&fev1)) ){
      if (status < 0){
	sprintf(dd_log_msg," ddu_get_fev: error in dbfi_get_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
	if ( ! ddu_attached() ) {
	  sprintf(dd_log_msg,"ddu_get_fev: dd system not attached \n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
        sprintf(dd_log_msg," ddu_get_fev: error in dbfi_get_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
	dd_error_inv();
      }
      return status;
    }
  }
  
  *fev = fev1;
  return status;
}


int ddu_put_fev(struct fifo_entry fev)
/*
 * Gets the next fifo and puts the fifo event fev there.
 */
{
  int status; 

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"ddu_put_fev:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_producer ){
    if ( (status = ddnc_put_fev(fev)) ){
      if (status < 0) {
	sprintf(dd_log_msg,"ddu_put_fev: error in ddnc_put_fev\n");
	dd_log(DD_ERROR, dd_log_msg);
      }
      return status;
    }
  }
  else if ( i_am_dd2tcp_consumer ){
    return 0;
  }
  else{
    if ( (status = dbfi_put_fev(fev)) ){
      if (status < 0){
	if ( ! ddu_attached() ) {
	  sprintf(dd_log_msg,"ddu_put_fev: dd system not attached \n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	sprintf(dd_log_msg," ddu_put_fev: error in dbfi_put_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
	dd_error_inv();
      }
      return status;
    }
  }
  return status;
}

int ddu_ins_fev(struct fifo_entry fev)
/*
 * inserts an fev into the DD system
 */
{
  int status; 

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"ddu_ins_fev:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_producer ){
    if ( (status = ddnc_ins_fev(fev)) ){
      if (status < 0) {
	sprintf(dd_log_msg,"ddu_ins_fev: error in ddnc_ins_fev\n");
	dd_log(DD_ERROR, dd_log_msg);
      }
      return status;
    }
  }
  else if ( i_am_dd2tcp_consumer ){
    return 0;
  }
  else{
    if ( (status = dbfi_ins_fev(fev)) ){
      if (status < 0){
	if ( ! ddu_attached() ) {
	  sprintf(dd_log_msg,"ddu_ins_fev:  dd system not attached \n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	sprintf(dd_log_msg," ddu_ins_fev: error in dbfi_ins_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
	dd_error_inv();
      }
      return status;
    }
  }
  return status;
}


int ddu_brc_fev(struct fifo_entry fev, int wait_flag)
/*
 * Broadcasts the fifo event fev over the dd system. 
 * If wait_flag = 0 return immediately, if != 0 wait until every 
 * process attached at the dd has received the event.
 */
{
  int status, status1, status2; 

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"ddu_brc_fev:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_producer ){
    if ( (status = ddnc_brc_fev(fev, wait_flag)) ){
      if (status < 0) {
	sprintf(dd_log_msg,"ddnc_brc_fev: error in ddnc_brc_fev\n");
      dd_log(DD_ERROR, dd_log_msg);
      }
      return status;
    }
  }
  else if ( i_am_dd2tcp_consumer ){
    return 0;
  }
  else{
    if ( (status1 = dbfi_brc_fev(fev, wait_flag)) ){
      if (status1 < 0){
	if ( ! ddu_attached() ) {
	  sprintf(dd_log_msg,"ddu_brc_fev: dd system not attached \n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	sprintf(dd_log_msg," dbfi_brc_fev: error in dbfi_brc_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
	dd_error_inv();
      }
    }
    if ( (status2 = dbfi_rel_fev(fev)) ){
      if (status2 < 0){
	if ( ! ddu_attached() ) {
	  sprintf(dd_log_msg,"ddu_brc_fev: dd system not attached \n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	sprintf(dd_log_msg," dbfi_rel_fev: error in dbfi_rel_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
	dd_error_inv();
      }
    }
    status = status1 | status2;
  }
  
  return status;
}


int ddu_rel_fev(struct fifo_entry fev)
/*
 * Releases an event aquired with ddu_req_fev, 
 * i.e. puts it directly back into INPUT_FIFO.
 * If the process in a dd2tcp_producer, then no action is 
 * necessary, as the network transfer is done in ddu_put_fev.
 */
{
  int status; 

  if ( ! dd_attached() ) {
    sprintf(dd_log_msg,"ddu_rel_fev:  dd system not attached \n");
    dd_log(DD_ERROR, dd_log_msg);
    return -1;
  }

  if ( i_am_dd2tcp_producer ){
	 return 0;
  }
  else if ( i_am_dd2tcp_consumer ){
	 return 0;
  }
  else{
    if ( (status = dbfi_rel_fev(fev)) ){
      if (status < 0){
	if ( ! ddu_attached() ) {
	  sprintf(dd_log_msg,"ddu_brc_fev: dd system not attached \n");
	  dd_log(DD_ERROR, dd_log_msg);
	  return -1;
	}
	sprintf(dd_log_msg," dbfi_rel_fev: error in dbfi_rel_fev \n");
	dd_log(DD_ERROR, dd_log_msg);
	dd_error_inv();
      }
      return status;
    }
  }
  return status;
}


