
/*
 * dd_server     : Main server loop for dd2tcp.
 * Author        :  C.Witzig
 * Date          :  Jan 2, 1996
 */

/*
 * DD server program:
 * ------------------
 * The DD server gets spawned on a host (e.g. by the script 
 * start_dd_server and then waits for a request
 * 
 * See W.Steven's book "UNIX NETWORK PROGRAMMING" page 284ff
 * for details
 */

/*
 * Mods          :
 *
 */

#include <stdio.h>
#include <syslog.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <time.h>

#include <dd_user.h>

/* Added by CTCTCT in order to die gracefully */
extern  int h_errno;
static int go_on = 1;

static void
sig_handler (int sig)
{
/*
  printf("Going thru sig_handler for dd_server.\n");
  printf("signal %d received\n", sig);
  fflush(stdout);
*/
  go_on = 0;
  return;
}


main()
{
  int sockfd;
  struct sockaddr_in client_addr, server_addr;
  int client_addr_len = sizeof(client_addr);
  time_t now;

  /* Added by CTCTCT in order to die gracefully */
  signal(SIGHUP,  sig_handler);
  signal(SIGINT,  sig_handler);
  signal(SIGQUIT, sig_handler);
  signal(SIGTERM, sig_handler);

  time(&now);
  fprintf(stdout ,"DD_server: pid %d LOGFILE \n",getpid() );
  fprintf(stdout ,"---------------------------------------\n");
  fprintf(stdout ,"DD_server: pid %d starts at %s",getpid(), ctime(&now));
  fflush(stdout);

  if ( (sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0 ) {
    now = time(NULL);
    printf("DD_server: pid %d cannot open socket at %s ",getpid(),ctime(&now)) ;
    printf("DD_server: pid %d exits at %s",getpid(),ctime(&now));
    exit(1);
  }

  bzero( (char*)&server_addr, sizeof(server_addr));
  server_addr.sin_family = AF_INET;
  server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
  server_addr.sin_port = htons(DD_SERVER_TCP_PORT);
  if ( bind(sockfd, (struct sockaddr*) &server_addr, sizeof(server_addr)) < 0 ) {
    now = time(NULL);
    printf("DD_server: pid %d cannot bind socket %s at %s",getpid(), strerror(errno),ctime(&now));
    printf("DD_server: pid %d exits at %s",getpid(),ctime(&now));
    exit(1);
  }
 
  listen(sockfd, 5);

  while (go_on) {
    pid_t pid;
    struct hostent *hp_new;
    void *p2v;

    dd_sockfd = accept(sockfd, (struct sockaddr *)&client_addr, &client_addr_len);
    if ( dd_sockfd < 0 ) {
	printf("DD_server: pid %d error in accept socket\n",getpid());
	go_on = 0;
    }
    
    now = time(NULL);
    p2v = (void*) &client_addr.sin_addr.s_addr;
    hp_new = gethostbyaddr(p2v, client_addr_len, AF_INET);
    /* add some debug info here, CTCTCT */
    if (hp_new == NULL) {
	printf("gethostbyaddr in dd_server: ");
	switch (h_errno) {
	    case HOST_NOT_FOUND : printf("HOST_NOT_FOUND\n");break;
	    case TRY_AGAIN :      printf("TRY_AGAIN\n");break;
	    case NO_RECOVERY :    printf("NO_RECOVERY\n");break;
	    case NO_DATA :        printf("NO_DATA or NO_ADDRESS\n");break;
	}
    }

    printf("dd_server: %d: connection established to %08x at %s",
	    getpid(), client_addr.sin_addr.s_addr/*hp_new->h_name*/,ctime(&now));
    fflush(stdout);

    if ( ( pid = fork() ) < 0 ) {
	printf("DD_server: pid %d error in fork \n",getpid());
	go_on = 0;
    }
    else if ( pid == 0 ) {
	close(sockfd);
	dd2tcp_server();
	exit(0);
    }
    
    close(dd_sockfd);
    fflush(stdout);
  }

  now = time(NULL);
  printf("DD_server: %d exit at %s",getpid(), ctime(&now));

}


