//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG database interface to b-tree library (Thread Safe)
//
// Author:  Jie Chen
//
// Revision History:
//   $Log: cmlogDatabase.h,v $
//   Revision 1.2  2000/03/01 16:56:45  chen
//   Cocurrent control + db patches
//
//   Revision 1.1.1.1  1999/09/07 15:29:15  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:34:37  bickley
// Added cmlog to application development system.
//
//
//
#ifndef _CMLOG_DATABASE_H
#define _CMLOG_DATABASE_H

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <fcntl.h>
#include <cdevData.h>
#include <sys/types.h>
#include <db.h>
#include <cmlogConfig.h>

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
#include <cpSynch.h>
#endif

typedef double cmlogKey;

// this class will handle interface to b+tree based database implementation.
// data/key are cdevData/double. When caller insert a cdevData, the
// cdevData must have a key in this data with tag name 'cmlogTime'.

// This interface will not provide two iterators at the same time.
// This is the limitation by low level database library

// All pointers to key must be the memory location provided by callers

class cmlogDatabaseLocker;
class cmlogDatabaseWLocker;

class cmlogDatabase
{
public:
  // constructor
  // default constructor, users have to call open explicitly
  cmlogDatabase (void);
  // constructor: name is a file name, flags are the same flags as
  // standard open routine. 
  cmlogDatabase (char* name, int flags, int mode = 0666);
  // destructor
  ~cmlogDatabase (void);

  // operation

  // open a database
  int open      (char* name, int flags, int mode = 0666);
  // close connection to the database
  int close     (void);
  // get a cdevData
  int get       (cdevData& data, cmlogKey* key);
  // insert a cdevData which must have a key inside
  int put       (cdevData& data);
  // flush all internal buffer to disk
  int flush     (void);

  // return database name
  char* database (void) const;
  // return file descriptor for the opened database
  int   fd       (void);
  // check to see the database is open or not
  int   opened   (void) const;

  // iterator for sequential access to database

  // init cursor to the position pointed by key 'key',
  // and return data 'data' if successful
  // if first = 1: just put cursor to the first element
  // Note: the returned key is not necessarily an exact match for
  //       the specified key. The returned key is the smallest key
  //       greater than or equal to the specified key, permitting
  //       partial ket matches and range searches
  int cursorInit (cdevData& data, cmlogKey* key, int first = 0);

  // move cursor to the next position. 
  // return CDEV_SUCCESS on success, CDEV_ERROR: check errno
  // return CDEV_INVALIDOBJ: no more
  int cursorNext (cdevData& data, cmlogKey* key);

  // move cursor to the previous position
  // return CDEV_SUCCESS on success, CDEV_ERROR: check errno
  // return CDEV_INVALIDOBJ: no more
  int cursorPrev (cdevData& data, cmlogKey* key);

  const char* className (void) const {return "cmlogDatabase";}

  // very small number for double value comparision
  static double tinyValue;

protected:
  // convert DBT to cdevData
  static int convertData (cdevData& data, const DBT* res);
  // convert cdevData to DBT and a key
  // Make sure to free memory pointed by out->data, key->data 
  static int convertData (DBT* out, DBT* key, cdevData& data);
  // copy data from database back to memory
  static void copyKeyData (cmlogKey* key, DBT* data);

private:
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutex         mutex_;
#endif

  // open flag and mode
  int       flag_;
  int       mode_;
  // database name
  char*     name_;

  // b+tree access option: different site can change option
  BTREEINFO b_;
  // internal DB pointer
  DB*       dbp_;

  // keep track of size of data that have been written to a database
  int       wsize_;

  // deny access to copy and assignment operator
  cmlogDatabase (const cmlogDatabase& dbase);
  cmlogDatabase& operator = (const cmlogDatabase& dbase);

  // friend class
  friend class cmlogDatabaseLocker;
  friend class cmlogDatabaseWLocker;
};

class cmlogDatabaseLocker
{
public:
  // constructor and destructor
  cmlogDatabaseLocker  (cmlogDatabase* dbase);
  ~cmlogDatabaseLocker (void);

private:
  cmlogDatabase* dbase_;
  // deny access to assignment and copy operations
  cmlogDatabaseLocker  (const cmlogDatabaseLocker& );
  cmlogDatabaseLocker& operator = (const cmlogDatabaseLocker& );
};


class cmlogDatabaseWLocker
{
public:
  // constructor and destructor
  cmlogDatabaseWLocker  (cmlogDatabase* dbase);
  ~cmlogDatabaseWLocker (void);

private:
  cmlogDatabase* dbase_;
  // deny access to assignment and copy operations
  cmlogDatabaseWLocker  (const cmlogDatabaseWLocker& );
  cmlogDatabaseWLocker& operator = (const cmlogDatabaseWLocker& );
};

#undef INLINE
#ifdef __INLINE__
#define INLINE inline
#include "cmlogDatabase.i"
#else
#define INLINE
#endif


#endif  
  
  
  
