//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of cmlogXuiDisp Class
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogXuiDisp.cc,v $
//   Revision 1.12  2001/12/11 15:38:53  chen
//   Fix a minor compile problem for solaris CC 5.2
//
//   Revision 1.11  2001/07/25 15:15:52  chen
//   64 BIT Initial Port
//
//   Revision 1.10  2000/12/13 16:44:14  chen
//   add action and bell
//
//   Revision 1.9  2000/06/20 19:36:37  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.8  2000/02/14 17:32:43  chen
//   fix a bug when -c -hu when update string empty
//
//   Revision 1.7  2000/01/04 14:18:41  chen
//   improve performance for rapid incoming data
//
//   Revision 1.6  1999/12/13 21:24:06  chen
//   Fix a saving width problem from title bar area
//
//   Revision 1.5  1999/11/12 17:32:04  chen
//   beta 2.0
//
//   Revision 1.4  1999/10/29 17:40:07  chen
//   support spaces and quotes in cmlogrc file
//
//   Revision 1.3  1999/09/21 20:27:56  chen
//   Add dialog box for save settings
//
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.2  1998/10/07  17:03:42  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.1  1997/08/01  15:31:19  bickley
// Added cmlog to application development system.
//
//
#include <cmlogMsg.h>
#include <XcodaErrorDialog.h>
#include <XcodaInfoDialog.h>
#include <cmlogXuiMenuWin.h>
#include <cmlogXuiConfig.h>
#include <cmlogXuiTopW.h>
#include <cmlogXuiQueryProg.h>
#include <cmlogXuiQueryCancelDialog.h>
#include <cmlogXuiUpdate.h>
#include <cmlogXuiQuery.h>
#include <cmlogXuiLogicSup.h>
#include <cmlogXuiCodeConverter.h>
#include "cmlogXuiTitleBar.h"
#include "cmlogXuiDisp.h"

const char* cmlogXuiDisp::loggingData_ = "loggingData";
int   cmlogXuiDisp::initDone_ = 0;
int   cmlogXuiDisp::hasExposeEvHandle_ = 0;

cmlogXuiDisp::cmlogXuiDisp (Widget parent, const char* name,
			    int numcol, Dimension* colwidths,
			    char** tags, int ubufsize,
			    cmlogXuiMenuWin* menuwin,
			    cmlogXuiBrowser& browser)
:XcodaUi (name), XcodaTimer (parent), parent_ (parent), 
 list_ (0), num_cols_ (numcol),
 browser_ (browser), counter_ (0), updateBufsize_ (ubufsize), num_msgs_ (0),
 errDialog_ (0), titleBar_ (0),
 type_ (CMLOG_QUERY), numitems_ (200), getall_ (0), qstr_ (0),
 timestamp_ (0.0), start_ (0.0), end_ (0.0), numfound_ (0), menuwin_ (menuwin),
 progDialog_ (0), cancelQueryDialog_ (0), fdialog_ (0),queryInProg_ (0),
 doCodeConversion_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("          Create cmlogXuiDisp Class Object\n");
#endif
  int i;
  for (i = 0; i < num_cols_; i++) {
    col_widths_[i] = colwidths[i];
    tags_[i] = new char[strlen (tags[i]) + 1];
    strcpy (tags_[i], tags[i]);
  }
  for (i = num_cols_; i < _CMLOG_XUI_MAX_COLUMNS; i++) {
    col_widths_[i] = 0;
    tags_[i] = 0;
  }
}

cmlogXuiDisp::~cmlogXuiDisp (void)
{
#ifdef _TRACE_OBJECTS
  printf ("          Delete cmlogXuiDisp Class Object\n");
#endif
  for (int i = 0; i < num_cols_; i++) 
    delete []tags_[i];

  if (qstr_)
    delete []qstr_;
}

void
cmlogXuiDisp::init (void)
{
  Arg arg[20];
  int ac = 0;
  cmlogXuiConfig* config = cmlogXuiConfig::config();

  XtSetArg (arg[ac], XmNdefaultColumns, num_cols_); ac++;
  XtSetArg (arg[ac], XmNcolumnWidth, 50); ac++;
  XtSetArg (arg[ac], XmNforceColumns, True); ac++;
  XtSetArg (arg[ac], XmNverticalList, True); ac++;
  XtSetArg (arg[ac], XmNseparatorWidth, 4); ac++;
  XtSetArg (arg[ac], XmNseparatorType, XmSHADOW_ETCHED_OUT); ac++;
  XtSetArg (arg[ac], XmNpreferredWidths, col_widths_); ac++;
  XtSetArg (arg[ac], XmNmaxSelectable, 0); ac++;
  XtSetArg (arg[ac], XmNwidth, config->windowWidth()); ac++;
  XtSetArg (arg[ac], XmNheight, config->windowHeight()); ac++;
  list_ = MultiListCreateScrolledList (parent_, (char *)("mlist"), arg, ac);
  ac = 0;
  XtManageChild (list_);
  
  _w = XtParent (list_);

  // handle moveing separator callback
  XtAddCallback (list_, XmNsepMoveCallback,
		 (XtCallbackProc)&(cmlogXuiDisp::sepMoveCallback),
		 (XtPointer)this);

  if (!cmlogXuiDisp::hasExposeEvHandle_) {
    XtAddEventHandler (list_, ExposureMask, 1,
		       (XtEventHandler)&(cmlogXuiDisp::exposeEventHandler), 
		       (XtPointer)this);
    cmlogXuiDisp::hasExposeEvHandle_ = 1;
  }

  // handle pointer motion
  XtAddCallback (list_, XmNpointerMotionCallback,
		 (XtCallbackProc)&(cmlogXuiDisp::pointerMotionCallback),
		 (XtPointer)this);

  // handle destroy callback
  installDestroyHandler ();

  // find out default background and foreground
  XtSetArg (arg[ac], XtNbackground, &defBg_); ac++;
  XtSetArg (arg[ac], XtNforeground, &defFg_); ac++;
  XtGetValues (list_, arg, ac);
  ac = 0;
}

void
cmlogXuiDisp::configure (int numcols, Dimension* colwidths, char** tags)
{
  int i;

  // free old information
  for (i = 0; i < num_cols_; i++)
    delete []tags_[i];
  
  num_cols_ = numcols;
  for (i = 0; i < num_cols_; i++) {
    col_widths_[i] = colwidths[i];
    tags_[i] = new char[strlen (tags[i]) + 1];
    strcpy (tags_[i], tags[i]);
  }
  for (i = num_cols_; i < _CMLOG_XUI_MAX_COLUMNS; i++) {
    col_widths_[i] = 0;
    tags_[i] = 0;
  }

  Arg arg[10];
  int ac = 0;

  XtSetArg (arg[ac], XmNdefaultColumns, num_cols_); ac++;
  XtSetArg (arg[ac], XmNpreferredWidths, col_widths_); ac++;
  XtSetValues (list_, arg, ac);
  ac = 0;

  // update configuration to the configuration object
  // update changes to configuration object
  // this  routine is called by title bar seperator move callback
  cmlogXuiConfig* config = cmlogXuiConfig::config ();
  
  config->changeConfiguration (num_cols_, col_widths_,
			       titleBar_->titles (),
			       tags_, updateBufsize_);  
}

void
cmlogXuiDisp::configure (int numcols, Dimension* colwidths)
{
  int i;

  assert (num_cols_ == numcols);
  num_cols_ = numcols;

  for (i = 0; i < num_cols_; i++) 
    col_widths_[i] = colwidths[i];

  for (i = num_cols_; i < _CMLOG_XUI_MAX_COLUMNS; i++) 
    col_widths_[i] = 0;


  Arg arg[10];
  int ac = 0;

  XtSetArg (arg[ac], XmNdefaultColumns, num_cols_); ac++;
  XtSetArg (arg[ac], XmNpreferredWidths, col_widths_); ac++;
  XtSetValues (list_, arg, ac);
  ac = 0;
  // since the preferred width pointer is not changed, refresh
  MultiListRefresh (list_);

  // update configuration to the configuration object
  // update changes to configuration object
  // this  routine is called by title bar seperator move callback
  cmlogXuiConfig* config = cmlogXuiConfig::config ();
  
  config->changeConfiguration (num_cols_, col_widths_,
			       titleBar_->titles (),
			       tags_, updateBufsize_);
}

void
cmlogXuiDisp::display (cmlogMsg** msgs, int num, int movetobottom)
{
  // all display string
  char temp[256];
  char codestr[32];
  char* swap;
  MultiListItem* inputs;
  cdevData* data;
  cmlogXuiConfig* config = cmlogXuiConfig::config();
  int       i = 0, j = 0, k = 0, l =0;
  int       numitems = num * num_cols_;

  // allocate memory for all items
  inputs = new MultiListItem[numitems];

  // initialize this array
  for (i = 0; i < numitems; i++) {
    inputs[i].sensitive = 1;
    inputs[i].highlighted = 0;
    inputs[i].type = MULTILIST_CELL_NORMAL;
    inputs[i].color = 0;
    inputs[i].string = 0;
    inputs[i].altstring = 0;
    inputs[i].blink = 0;
  }

  for (i = 0; i < num; i++) {
    cmlog_cdevMessage& idata = (*msgs[i]);
    data = idata.getData ();
    if (data) {
      cmlogXuiCodeConverter* converters[10];
      int                    csize = 10;
      int                    status = 0;
      status = config->codeConverter (*data, converters, &csize);

      // first get key value
      double key;
      time_t t = 0;
      if (data->get ((char *)("cmlogTime"), &key) == CDEV_SUCCESS) {
	timestamp_ = key;
	t = (time_t)key;
      }

      for (j = 0; j < num_cols_; j++) {
	if (strcmp (tags_[j], "cmlogTime") == 0) {
	  if (t != 0)
	    strncpy (temp, ctime (&t), sizeof (temp));
	  else
	    strcpy (temp, "N/A");
	}
	else {
	  if (!csize) {
	    if (data->get (tags_[j], temp, sizeof (temp) - 1) 
		!= CDEV_SUCCESS)
	      strcpy (temp, "N/A");
	  }
	  else {
	    int itag;
	    int datatype;
	    long code;
	    Pixel cellColor;
	    int   blink;
	    int   found;
	    // the following call must be successful since we have checked tas_[i]
	    cdevData::tagC2I (tags_[j], &itag);
	    datatype = data->getType (itag);
	    if (datatype == CDEV_INVALID) 
	      strcpy (temp, "N/A");
	    else if (datatype < CDEV_STRING) {
	      data->get (itag, &code);
	      // check whether tag matches conversion tag and whether type is right
	      found = 0;
	      for (l = 0; l < csize; l++) {
		if (itag == converters[l]->conversionTag()) {
		  // do color coding in both modes
		  if (converters[l]->find (code,temp,sizeof (temp), &cellColor,&blink)) {
		    inputs[k].color = cellColor > 0 ? cellColor : defBg_;
		    inputs[k].blink = blink;
		    inputs[k].type = MULTILIST_CELL_DRAW_BG;
		    // save code into alternative string
		    sprintf (codestr, "%d", code);
		    inputs[k].altstring = new char[strlen(codestr) + 1];
		    strcpy (inputs[k].altstring, codestr);
		    found = 1;
		  }
		  // only do bell and action in updating mode
		  if (type_ == CMLOG_UPDATING) {
		    char action[128];
		    if (converters[l]->doBell (code))
		      XBell (XtDisplay (list_), 50);
		    if (converters[l]->hasAction (code, action, sizeof (action) - 1))
		      doAction (action);
		  }
		  break;
		}
	      }
	      if (!found) 
		data->get (itag, temp, sizeof (temp) - 1); 
	    }
	    else if (datatype == CDEV_STRING) {
	      data->get (itag, temp, sizeof (temp) - 1);
	      for (l = 0; l < csize; l++) {
		if (itag == converters[l]->conversionTag()) {
		  if (converters[l]->find (temp, &cellColor, &blink)) {
		    inputs[k].color = cellColor > 0 ?  cellColor: defBg_;
		    inputs[k].blink = blink;
		    inputs[k].type = MULTILIST_CELL_DRAW_BG;
		  }
		  // only do bell and action in updating mode
		  if (type_ == CMLOG_UPDATING) {
		    char action[128];
		    if (converters[l]->doBell (temp))
		      XBell (XtDisplay (list_), 50);
		    if (converters[l]->hasAction (temp, action, sizeof (action) - 1))
		      doAction (action);
		  }		  
		  break;
		}
	      }
	    }
	    else 
	      data->get (itag, temp, sizeof (temp) - 1);
	  }
	}
	inputs[k].string = new char[strlen (temp) + 1];
	strcpy (inputs[k].string, temp);
	// if there is converter but doConversion is off, swap code and string
	if (!doCodeConversion_ && csize && inputs[k].altstring) {
	  swap = inputs[k].string;
	  inputs[k].string = inputs[k].altstring;
	  inputs[k].altstring = swap;
	}
	k++;
      }
    }
  }

  if (movetobottom) { // update mode: scroll to bottom
    if (updateBufsize_ == 0 || counter_ + numitems < updateBufsize_ * num_cols_) {
      MultiListAddListItems (list_, inputs,
			     numitems, 0, counter_, False);
      if (MultiListItemIsVisibleV (list_, counter_) != MULTILIST_YES)
	MultiListSetPos (list_, counter_);
      
      counter_ += numitems;
    }
    else {
      MultiListAddListItemsFixedBuf (list_,  numitems, inputs,
				     0, False);

      if (MultiListItemIsVisibleV (list_, counter_) != MULTILIST_YES)
 	MultiListSetPos (list_, counter_ - 1);
    }
  }
  else {
    MultiListAddListItems (list_, inputs, 
			   numitems,
			   0, counter_, False);
    counter_ += numitems;
  }


  // free input memory, but not string memories since they are moved into
  // widget directly
  delete []inputs;
}
  

void
cmlogXuiDisp::append (cmlogPacket* packet)
{
  // update number of found
  numfound_ += packet->numberOfData ();

  cmlogMsg** msgs = packet->messages ();
  display (msgs, packet->numberOfData (), 0);

  for (int i = 0; i < packet->numberOfData (); i++) 
    delete msgs[i];
  delete []msgs;
}

void
cmlogXuiDisp::getQueryMiscInfo (cmlogPacket* packet, double& key)
{
  cdevData* data;
  int       i = 0;

  cmlogMsg** msgs = packet->messages ();
  cmlog_cdevMessage& idata = (*msgs[0]);

  data = idata.getData ();
      
  if (data) {
    // first get key value
    if (data->get ((char *)("cmlogTime"), &key) != CDEV_SUCCESS) 
      key = 0.0;
  }
   
  for (i = 0; i < packet->numberOfData (); i++) {
    delete msgs[i];
  }
  delete []msgs;

}


void
cmlogXuiDisp::reportErrorMsg (const char* msg)
{
  if (!errDialog_) {
    errDialog_ = new XcodaErrorDialog (_w, "cmlogXuiError", "Error Dialog");
    errDialog_->init ();
  }
  errDialog_->setMessage (msg);
  errDialog_->popup ();
}

void
cmlogXuiDisp::reportInfoMsg (const char* msg)
{
  if (!fdialog_) {
    fdialog_ = new XcodaInfoDialog (_w, "cmlogXuiInfo", "Info Dialog");
    fdialog_->init ();
  }
  fdialog_->setMessage (msg);
  fdialog_->popup ();
}

void
cmlogXuiDisp::popupCancelQueryDialog ()
{
  if (!cancelQueryDialog_) {
    cancelQueryDialog_ = new cmlogXuiQueryCancelDialog (this, 
							"cmlogXuiQueryCancel",
							"Cancel Query Dialog",
							browser_);
    cancelQueryDialog_->init ();
  }
  cancelQueryDialog_->popup ();
}

void
cmlogXuiDisp::popdownCancelQueryDialog ()
{
  if (cancelQueryDialog_ && cancelQueryDialog_->isMapped ()) 
    cancelQueryDialog_->popdown ();
}

void
cmlogXuiDisp::titleBar (cmlogXuiTitleBar *titlebar)
{
  titleBar_ = titlebar;
}


void
cmlogXuiDisp::sepMoveCallback (Widget w, XtPointer data,
			       MultiListSepMoveStruct* val)
{
  cmlogXuiDisp *obj = (cmlogXuiDisp *)data;

  if (obj->titleBar_) 
    obj->titleBar_->configure (val->num, val->widths);

  // update changes to configuration object
  cmlogXuiConfig* config = cmlogXuiConfig::config ();
  
  config->changeConfiguration (obj->num_cols_, val->widths,
			       obj->titleBar_->titles (),
			       obj->tags_, obj->updateBufsize_);
}

void
cmlogXuiDisp::pointerMotionCallback (Widget w, XtPointer data,
				     MultiListPositionStruct* pstruct)
{
  cmlogXuiDisp* disp = (cmlogXuiDisp *)data;
  disp->menuwin_->pointedMsgNumber (pstruct->row + 1);
}


int
cmlogXuiDisp::type (void) const
{
  return type_;
}

void
cmlogXuiDisp::enableUpdating (void)
{
  type_ = CMLOG_UPDATING;
}

void
cmlogXuiDisp::disableUpdating (void)
{
  type_ = CMLOG_QUERY;
}

void
cmlogXuiDisp::enableBlinkOn (void)
{
  Arg arg[10];
  int ac = 0;

  XtSetArg (arg[ac], XmNblink, True); ac++;
  XtSetValues (list_, arg, ac);
  ac = 0;
}

void
cmlogXuiDisp::disableBlinkOn (void)
{
  Arg arg[10];
  int ac = 0;

  XtSetArg (arg[ac], XmNblink, False); ac++;
  XtSetValues (list_, arg, ac);
  ac = 0;
}

void
cmlogXuiDisp::queryCallback (int status, void* arg, cmlogPacket* data)
{
  cmlogXuiDisp* disp = (cmlogXuiDisp *)arg;
  cdevData* res = 0;
  cmlogXuiTopW* topw = (cmlogXuiTopW *)disp->menuwin_->topLevelWindow ();
  int       numfound = 0;
  char      message[128];
  cmlogXuiConfig* config = cmlogXuiConfig::config();

  // first reset cusror
  disp->menuwin_->undefineCursor ();

  if (status == CMLOG_ERROR || status == CMLOG_NOTFOUND) {
    // popdown cancel query dialog box
    disp->popdownCancelQueryDialog ();
    
    // reset query in progress flag
    disp->queryInProgress (0);

    if (disp->numfound_ > 0) {
      // this must be the last step
      sprintf (message, 
	       "Found no more messages: %d messages have been found.",
	       disp->numfound_);
      disp->reportErrorMsg (message);
    }
    else
      disp->reportErrorMsg ((char *)("Cannot find anything !!!"));
    disp->menuwin_->normalConfig ();
    disp->numfound_ = 0;

    // if we are still in initialiation step, go into updating mode
    if (!cmlogXuiDisp::initDone_) {
      disp->menuwin_->update_->startUpdating (config->updateSelMessage());      
      cmlogXuiDisp::initDone_ = 1;
    }
  }
  else if (status == CMLOG_INCOMPLETE) {
    disp->append (data);
    // update motif display
    topw->forceProcessX ();
  }
  else if (status == CMLOG_PAUSED) {
    // reset query in progress flag
    disp->queryInProgress (0);

    // get time stamp and so on
    disp->getQueryMiscInfo (data, disp->timestamp_);
    if (disp->numfound_ > 0)
      sprintf (message, "Found %d items, do you want more?", disp->numfound_);
    else
      sprintf (message, "Found nothing so far, do you want to continue?");

    // set time stamp to the window
    disp->menuwin_->setEndTime ((time_t)disp->lastDataTimeStamp ());
    if (!(disp->progDialog_)) {
      disp->progDialog_ = new cmlogXuiQueryProg (disp, 
			 (char *)("QueryProgressDialog"),
			 (char *)("Query Progress Report"),
			 disp->browser_);
      disp->progDialog_->init ();
    }
    disp->progDialog_->setMessage (message);
    disp->progDialog_->popup ();
  }
  else if (status == CMLOG_QUERYMSG_ERR) {
    // popdown cancel query dialog box
    disp->popdownCancelQueryDialog ();

    // reset query in progress flag
    disp->queryInProgress (0);

    disp->reportErrorMsg ((char *)("Query Message Syntax Error"));
    disp->menuwin_->normalConfig ();
    disp->numfound_ = 0;

    // if we are still in initialiation step, go into updating mode
    if (!cmlogXuiDisp::initDone_) {
      disp->menuwin_->update_->startUpdating (config->updateSelMessage());    
      cmlogXuiDisp::initDone_ = 1;
    }
  }
  else {
    // popdown cancel query dialog box
    disp->popdownCancelQueryDialog ();

    // reset query in progress flag
    disp->queryInProgress (0);

    if (disp->numfound_ > 0)
      sprintf (message, "Query Process finished: found %d items", 
	       disp->numfound_);
    else
      sprintf (message, "Query Process finished: found nothing");
    disp->numfound_ = 0;
    
    disp->menuwin_->normalConfig ();
    disp->menuwin_->setEndTime ((time_t)disp->endTime ());

    // report query information
    disp->reportInfoMsg (message);

    // if we are still in initialiation step, go into updating mode
    if (!cmlogXuiDisp::initDone_) {
      char* selstr = 0;
      if (config->updateSelMessage())
	selstr = cmlogXuiQueryString (config->updateSelMessage());
      disp->menuwin_->update_->startUpdating (selstr);
      cmlogXuiDisp::initDone_ = 1;
    }
  }
}

void
cmlogXuiDisp::cleanup (void)
{
  Arg arg[10];
  int ac = 0;
  
  XtSetArg (arg[ac], XmNnumberStrings, 0); ac++;
  XtSetValues (list_, arg, ac);
  ac = 0;
  
  counter_ = 0;
  numfound_ = 0;
  start_ = 0.0;
  end_ = 0.0;
  timestamp_ = 0.0;
}

cmlogXuiMenuWin*
cmlogXuiDisp::menuWindow (void) const
{
  return menuwin_;
}

int
cmlogXuiDisp::numberItemsToGet (void) const
{
  return numitems_;
}

void
cmlogXuiDisp::numberItemsToGet (int num)
{
  numitems_ = num;
}

int
cmlogXuiDisp::getAll (void) const
{
  return getall_;
}

void
cmlogXuiDisp::enableGetAll (void)
{
  getall_ = 1;
}

void
cmlogXuiDisp::disableGetAll (void)
{
  getall_ = 0;
}


void
cmlogXuiDisp::setEndTime (double end)
{
  end_ = end;
}

double
cmlogXuiDisp::endTime (void) const
{
  return end_;
}

void
cmlogXuiDisp::setStartingTime (double st)
{
  start_ = st;
}

double
cmlogXuiDisp::startingTime (void) const
{
  return start_;
}


double
cmlogXuiDisp::lastDataTimeStamp (void) const
{
  return timestamp_;
}

void
cmlogXuiDisp::queryMessage (char* msg)
{
  if (qstr_)
    delete []qstr_;

  if (!msg) 
    qstr_ = 0;
  else {
    qstr_ = new char[strlen (msg) + 1];
    strcpy (qstr_, msg);
  }
}

char *
cmlogXuiDisp::queryMessage (void) const
{
  return qstr_;
}

void
cmlogXuiDisp::defineCursor (unsigned int cursor)
{
  menuwin_->defineCursor (cursor);
}

void
cmlogXuiDisp::undefineCursor (void)
{
  menuwin_->undefineCursor ();
}

void
cmlogXuiDisp::scrollToBottom (void)
{
  int lastline = counter_ - num_cols_;
  if (MultiListItemIsVisibleV (list_, lastline) != MULTILIST_YES) 
    MultiListSetPos (list_, lastline);
}

int
cmlogXuiDisp::startUpdating (char* selection)
{
  char message[128];
  cdevData data;

  if (selection) 
    data.insert (cmlogUtil::CMLOG_QUERYMSG_TAG, selection);

  strcpy (message, "monitorOn ");
  strcat (message, cmlogXuiDisp::loggingData_);

  if (browser_.queryCallback (message, data,
			      &(cmlogXuiDisp::monitorDataCallback),
			      (void *)this) != CMLOG_SUCCESS) {
    reportErrorMsg ((char *)("Cannot send updating command to the server."));
    return CMLOG_ERROR;
  }
  return CMLOG_SUCCESS;
}

int
cmlogXuiDisp::stopUpdating (void)
{
  cdevData noused;

  char message[128];
  strcpy (message, "monitorOff ");
  strcat (message, cmlogXuiDisp::loggingData_);
  if (browser_.queryCallback (message, noused, 
			      &(cmlogXuiDisp::monitorDataCallback),
			      (void *)this) != CMLOG_SUCCESS) {
    reportErrorMsg((char *)("Failed to send monitorOff command to the server."));
    return CMLOG_ERROR;
  }
  return CMLOG_SUCCESS;
}

void
cmlogXuiDisp::monitorDataCallback (int status, 
				   void* arg, cmlogPacket* data)
{
  cmlogXuiDisp* obj = (cmlogXuiDisp *)arg;
  int           i, j;

  if (status == CMLOG_SUCCESS) {
    cmlogMsg** msgs = data->messages ();
    for (i = 0; i < data->numberOfData (); i++) {
      if (obj->num_msgs_ == 0) 
	obj->arm (500);
      if (obj->num_msgs_ >= CMLOG_PACKET_LIST_SIZE) {
	obj->display (obj->mbuffer_, obj->num_msgs_, 1);

	// free memory
	for (j = 0; j < obj->num_msgs_; j++)
	  delete obj->mbuffer_[j];

	obj->num_msgs_ = 0;
      }
      obj->mbuffer_[obj->num_msgs_ ++] = msgs[i];
    }
    // free memory: individual message has been released in timer or the above
    delete []msgs;
  }
  else if (status == CMLOG_CBK_FINISHED) {
    char message[128];

    strcpy (message, cmlogXuiDisp::loggingData_);
    if (obj->removeMonitorCallback (message,
			 &(cmlogXuiDisp::monitorDataCallback),
			 (void *)obj) == CMLOG_SUCCESS)
      ;
  }
}

int
cmlogXuiDisp::removeMonitorCallback (char* attr, cmlogBrCallback callback, 
				     void* arg)
{
  return browser_.removeMonitorCallback (attr, callback, arg);
}

void
cmlogXuiDisp::numberItemsFound (int num)
{
  numfound_ = num;
}

int
cmlogXuiDisp::numberItemsFound (void) const
{
  return numfound_;
}

void
cmlogXuiDisp::updatingBufSize (int size)
{
  int numremove = 0;

  if (size != 0 && counter_ > size * num_cols_) {
    // we have to remove the some items from the beginning of the list
    numremove = counter_ - size * num_cols_;
    MultiListDeleteItems (list_, 0, numremove);
    counter_ = size * num_cols_;
  }
  updateBufsize_ = size;

  // update buffer size information inside configuration
  cmlogXuiConfig* config = cmlogXuiConfig::config();
  config->updateBufsize (size);
}

int
cmlogXuiDisp::updatingBufSize (void) const
{
  return updateBufsize_;
}

void
cmlogXuiDisp::refresh (void)
{
  MultiListRefresh (list_);
}


void
cmlogXuiDisp::removeAllMessages (void)
{
  cleanup ();
}

void
cmlogXuiDisp::queryInProgress (int flag)
{
  queryInProg_ = flag;
}

void
cmlogXuiDisp::exposeEventHandler (Widget w, XtPointer data, XEvent *,
				  Boolean *)
{
  cmlogXuiDisp *disp = (cmlogXuiDisp *)data;
  cmlogXuiConfig* config = cmlogXuiConfig::config();
  int      autoconnect = 0;

  // process code conversion color string
  config->colorStringsToPixels ();

  if (config->autoConnection() || config->autoUpdatingMode() ||
      config->autoQueryAndUpdate() || config->autoQueryMode ())
    autoconnect = 1;

  if (autoconnect) {
    if (disp->browser_.connect() != CMLOG_SUCCESS) {
      cmlogXuiDisp::initDone_ = 1;
      disp->reportErrorMsg ((char *)("Cannot connect to a CMLOG server"));
    }
    else {
      if (config->autoUpdatingMode()) {
	char* selstr = 0;
	if (config->updateSelMessage())
	  selstr = cmlogXuiQueryString (config->updateSelMessage());
	cmlogXuiDisp::initDone_ = 1;
	disp->menuwin_->update_->startUpdating (selstr);
      }
      else if (config->autoQueryAndUpdate ()) {
	char *qstr = 0;
	char *selstr = 0;
	if (config->queryMessage())
	  qstr = cmlogXuiQueryString (config->queryMessage());
	if (config->updateSelMessage())
	  selstr = cmlogXuiQueryString (config->updateSelMessage());

	// check start and end time and make sure they are ok
	if (config->queryStartTime() < 0 || config->queryEndTime() < 0 ||
	    config->queryStartTime() >= config->queryEndTime() ||
	    disp->menuwin_->query_->doQuery (config->queryStartTime(),
					     config->queryEndTime(), 0, 1,
					     qstr) != CMLOG_SUCCESS) {
	  cmlogXuiDisp::initDone_ = 1;
	  disp->menuwin_->update_->startUpdating(selstr);
	}
      }
      else if (config->autoQueryMode ()) {
	cmlogXuiDisp::initDone_ = 1;

	char *qstr = 0;
	char *selstr = 0;
	if (config->queryMessage())
	  qstr = cmlogXuiQueryString (config->queryMessage());

	// check start and end time and make sure they are ok
	if (config->queryStartTime() < 0 || config->queryEndTime() < 0 ||
	    config->queryStartTime() >= config->queryEndTime() ||
	    disp->menuwin_->query_->doQuery (config->queryStartTime(),
					     config->queryEndTime(), 0, 1,
					     qstr) != CMLOG_SUCCESS) 
	  disp->reportErrorMsg ((char *)("Cannot perform user specified query request"));
      }
      else
	cmlogXuiDisp::initDone_ = 1;
    }
  }
  else
    cmlogXuiDisp::initDone_ = 1;

  // remove expose event handler
  XtRemoveEventHandler (w, ExposureMask, 1,
			&(cmlogXuiDisp::exposeEventHandler), disp);
}

void
cmlogXuiDisp::timer_callback (void)
{
  if (num_msgs_ > 0) {
    display (mbuffer_, num_msgs_, 1);
    for (int i = 0; i < num_msgs_; i++)
      delete mbuffer_[i];
    num_msgs_ = 0;
  }
}

unsigned short
cmlogXuiDisp::windowWidth (void)
{
  Arg arg[10];
  int ac = 0;
  unsigned short w;

  XtSetArg (arg[ac], XmNwidth, &w); ac++;
  XtGetValues (XtParent(list_), arg, ac);
  ac = 0;

  return w;
}

unsigned short
cmlogXuiDisp::windowHeight (void)
{
  Arg arg[10];
  int ac = 0;
  unsigned short h;

  XtSetArg (arg[ac], XmNheight, &h); ac++;
  XtGetValues (XtParent(list_), arg, ac);
  ac = 0;

  return h;
}

void
cmlogXuiDisp::toggleCodeConvDisp (void)
{
  MultiListToggleDisplay (list_);
}

int
cmlogXuiDisp::numberMessagesInBuffer (void)
{
  Arg arg[20];
  int ac = 0;
  int num;
  
  XtSetArg (arg[ac], XmNnumberStrings, &num); ac++;
  XtGetValues (list_, arg, ac); 
  ac = 0;

  cmlogXuiConfig* config = cmlogXuiConfig::config();
  return num/config->numColumns();
}

int
cmlogXuiDisp::saveCurrentScreen (char* filename)
{
  FILE *fd = fopen (filename, "w");
  if (!fd)
    return -1;

  cmlogXuiConfig* config = cmlogXuiConfig::config();
  MultiListItem* items;
  int            numitems;
  int            i;
  char*          tmp;
  char**         cols = config->columnTitles();

  // printf out header information
  fprintf (fd, "#");
  for (i = 0; i < config->numColumns(); i++) 
    fprintf (fd, "%-s      ", cols[i]);
  fprintf (fd, "\n");

  numitems = MultiListAllVisibleItems (list_, &items);
  for (i = 0; i < numitems; i++) {
    if (i != 0 && i % config->numColumns() ==0 )
      fprintf (fd, "\n");
    if ((tmp = strrchr (items[i].string, '\n')) != 0)
      *tmp = '\0';
    fprintf (fd, "%-s      ", items[i].string);
  }
  fclose (fd);
  return 0;
}

int
cmlogXuiDisp::saveMessages (char* filename, int from, int end)
{
  FILE *fd = fopen (filename, "w");
  if (!fd)
    return -1;

  cmlogXuiConfig* config = cmlogXuiConfig::config();
  MultiListItem* items;
  int            numitems;
  int            i;
  int            index, size;
  char*          tmp;
  char**         cols = config->columnTitles();

  // printf out header information
  fprintf (fd, "#");
  for (i = 0; i < config->numColumns(); i++) 
    fprintf (fd, "%-s      ", cols[i]);
  fprintf (fd, "\n");

  index = (from - 1)*config->numColumns ();
  size = (end - from + 1)*config->numColumns();
  if (size <= 0)
    return -1;

  if (MultiListGetItems (list_, index, &items, &size) != 0)
    return -1;

  for (i = 0; i < size; i++) {
    if (i != 0 && i % config->numColumns() ==0 )
      fprintf (fd, "\n");
    if ((tmp = strrchr (items[i].string, '\n')) != 0)
      *tmp = '\0';
    fprintf (fd, "%-s      ", items[i].string);
  }
  fclose (fd);
  return 0;
}

void
cmlogXuiDisp::doAction (char* action)
{
  ::system (action);
}


