//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG XUI Browser Class
//
//      Unix environment variable CMLOG_PORT changes
//      broadcasting destination port number
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogXuiBrowser.cc,v $
//   Revision 1.3  2001/07/25 15:15:52  chen
//   64 BIT Initial Port
//
//   Revision 1.2  2000/06/20 19:36:36  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:12  chen
//   CMLOG version 2.0
//
// Revision 1.2  1998/10/07  17:03:41  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.1  1997/08/01  15:30:54  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogXuiTopW.h>
#include <cmlogXuiDisp.h>
#include "cmlogXuiBrowser.h"

const char* cmlogXuiBrowser::tagTable_ = "allTags";
cmlogSlist  cmlogXuiBrowser::wlist_;
int         cmlogXuiBrowser::first_ = 1;

cmlogXuiBrowser::cmlogXuiBrowser (XtAppContext context)
  :XcodaInput (context), XcodaTimer (context), browser_ (),
   defaultServerHost_ (0), defaultServerTcpPort_ (0), defaultServerUdpPort_ (0)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogXuiBrowser Class Object\n");
#endif
  // empty
}

cmlogXuiBrowser::~cmlogXuiBrowser (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogXuiBrowser Class Object\n");
#endif
  if (browser_.connected ()) {
    browser_.disconnect ();
    removeInput ();
  }
  // all windows will be deleted by destroy callback mechanism
}

void
cmlogXuiBrowser::setServerTcpInfo (char* host, unsigned short port)
{
  if (defaultServerHost_)delete []defaultServerHost_;
  defaultServerHost_ = new char[::strlen (host) + 1];
  ::strcpy (defaultServerHost_, host);
  defaultServerTcpPort_ = port;
}

void
cmlogXuiBrowser::setServerUdpInfo (char* host, unsigned short port)
{
  if (defaultServerHost_)delete []defaultServerHost_;
  defaultServerHost_ = new char[::strlen (host) + 1];
  ::strcpy (defaultServerHost_, host);
  defaultServerUdpPort_ = port;
}


int
cmlogXuiBrowser::connect (int num)
{
  int status = CMLOG_SUCCESS;

  if (defaultServerHost_ && defaultServerTcpPort_ != 0) 
    status = browser_.tcpConnect (defaultServerHost_, defaultServerTcpPort_);
  else
    status = browser_.connect (num);

  if (status == CMLOG_SUCCESS) {
#ifdef _CMLOG_DEBUG
    printf ("Connect to the cmlogServer\n");
#endif
    // configure the appearance of the window to reflect the connection
    connectionConfig ();

    // XcodaInput method to register my fd to X event loop
    addInput (browser_.getFd (), (XtPointer)XtInputReadMask);
    

    if (cmlogXuiBrowser::first_) {
      cmlogXuiBrowser::first_ = 0;

      // construct a command
      char qcommand[256];
      cdevData unused;

      strcpy (qcommand, "monitorOn ");
      strcat (qcommand, cmlogXuiBrowser::tagTable_);
    
      if (browser_.queryCallback (qcommand, unused, 
				  &(cmlogXuiBrowser::monitorTagCallback),
				  (void *)this) != CMLOG_SUCCESS) {
	fprintf (stderr, "Cannot monitor on allTags of the server\n");
	return CMLOG_ERROR;
      }
    
      // register disconnect callback
      if (browser_.disconnectCallback (&(cmlogXuiBrowser::discCallback),
					(void *)this) != CMLOG_SUCCESS) {
	fprintf (stderr, "Cannot register disconnect callback to the server\n");
	return CMLOG_ERROR;
      }
    }
  }
  else
    status = CMLOG_ERROR;

  return status;
}

int
cmlogXuiBrowser::connected (void) const
{
  return browser_.connected ();
}

void
cmlogXuiBrowser::input_callback (void)
{
#ifndef __hpux10
  browser_.pendIO (0.01);
#else
  if (browser_.read_input () < 0)
    browser_.handle_close (0);
#endif
}

int
cmlogXuiBrowser::disconnect (void)
{
  int status;
  if ((status = browser_.disconnect ()) == CMLOG_SUCCESS) {
    // remove this from x event loop
    removeInput ();
    // clean up callbacks
    browser_.cleanupCallbacks ();
    // config display to refelect the change of connection status
    disconnectConfig ();
  }
  return status;
}

int
cmlogXuiBrowser::numberOfWindows (void)
{
  return cmlogXuiBrowser::wlist_.count ();
}

int
cmlogXuiBrowser::addTopLevelWindow (cmlogXuiTopW* disp)
{
  cmlogSlistIterator ite (cmlogXuiBrowser::wlist_);
  cmlogXuiTopW* d = 0;

  for (ite.init (); !ite; ++ite) {
    d = (cmlogXuiTopW *) ite ();
    if (d == disp)
      return -1;
  }
  cmlogXuiBrowser::wlist_.add ((void *)disp);
  return 0;
}

int
cmlogXuiBrowser::removeTopLevelWindow (cmlogXuiTopW* disp)
{
  cmlogSlistIterator ite (cmlogXuiBrowser::wlist_);
  cmlogXuiTopW* d = 0;

  for (ite.init (); !ite; ++ite) {
    d = (cmlogXuiTopW *) ite ();
    if (d == disp) {
      cmlogXuiBrowser::wlist_.remove ((void *)disp);
      return 0;
    }
  }
  return -1;
}


void
cmlogXuiBrowser::monitorTagCallback (int status, void* arg, cmlogPacket* data)
{
  cmlogXuiBrowser* browser = (cmlogXuiBrowser *)arg;
  cdevData* res;
  int i = 0;

  if (status == CMLOG_SUCCESS) {
    if (data) {
      cmlogMsg** msgs = data->messages ();
      for (i = 0; i < data->numberOfData (); i++) {
	cmlog_cdevMessage& idata = (*msgs[i]);
	res = idata.getData ();

	if (res) {
	  // tag 1 represents integer tag value
	  int*     tagvalues;
	  char**   tagnames;
	  unsigned count;

	  if (res->getElems (1, &count) == CDEV_SUCCESS) {
	    // disable tag change callback since the following tags
	    // are from server. We do not want to send those info
	    // back to the server again
	    browser->disableTagmapNotify ();

	    tagvalues = new int[count];
	    tagnames = new char*[count];
	    res->get (1, tagvalues);
	    res->get (2, tagnames);
	    for (i = 0; i < count; i++) 
	      cdevData::insertTag (tagvalues[i], tagnames[i]);

	    // free memory
	    for (i = 0; i < count; i++)
	      delete []tagnames[i];
	    delete []tagvalues;
	    delete []tagnames;

	    // re-enable tag change callback
	    browser->enableTagmapNotify ();
	  }
	}
	  
      }
      for (i = 0; i < data->numberOfData (); i++) 
	delete msgs[i];
      delete []msgs;
    }   
  }
}

void
cmlogXuiBrowser::serverCrashConfig (void)
{
  cmlogXuiTopW* disp = 0;
  cmlogXuiBrowser* br = 0;
  cmlogSlistIterator ite (cmlogXuiBrowser::wlist_);

  for (ite.init (); !ite; ++ite) {
    disp = (cmlogXuiTopW *) ite ();
    disp->serverCrashConfig ();
    br = disp->browser ();
    br->removeInput ();
  }
}

void
cmlogXuiBrowser::disconnectConfig (void)
{
  if (window_)
    window_->disconnectConfig ();
}

void
cmlogXuiBrowser::connectionConfig (void)
{
  if (window_)
    window_->connectionConfig ();
}

void
cmlogXuiBrowser::updateTimeToWindows (time_t t)
{
  cmlogXuiTopW* disp = 0;
  cmlogSlistIterator ite (cmlogXuiBrowser::wlist_);

  for (ite.init (); !ite; ++ite) {
    disp = (cmlogXuiTopW *) ite ();
    disp->updateTime (t);
  }
}


void
cmlogXuiBrowser::discCallback (int status, void* arg, cmlogPacket* data)
{
  cmlogXuiBrowser* obj = (cmlogXuiBrowser *)arg;

  obj->serverCrashConfig ();
}

void
cmlogXuiBrowser::disableTagmapNotify (void)
{
  browser_.disableTagmapNotify ();
}

void
cmlogXuiBrowser::enableTagmapNotify (void)
{
  browser_.enableTagmapNotify ();
}

void
cmlogXuiBrowser::timer_callback (void)
{
  struct timeval tv;
  int    current_time;
    
  gettimeofday (&tv, 0);
  current_time = tv.tv_sec;

  time_t temp = (time_t)current_time;
  updateTimeToWindows (temp);
}

int
cmlogXuiBrowser::query (cmlogXuiDisp* disp,
			double from, double to,
			int numitems, int getall,
			char* qstr)
{
  char message[128];
  cdevData data;

  strcpy (message, "query");
  data.insert ((char *)("start"), from);
  data.insert ((char *)("end"), to);

  if (!getall) 
    data.insert ((char *)("numberItems"), numitems);

  if (qstr) 
    data.insert ((char *)("queryMsg"), qstr);

  if (browser_.queryCallback (message, data, 
			      &(cmlogXuiDisp::queryCallback),
			      (void *)disp) != CMLOG_SUCCESS) {
    disp->reportErrorMsg ("Cannot send query command to the server.");
    return CMLOG_ERROR;
  }

  // if this query is a getall which is possible long query,
  // popup a dialog box to allow user to cancel this query
  if (getall)
    disp->popupCancelQueryDialog ();

  return CMLOG_SUCCESS;  
}

int
cmlogXuiBrowser::queryCallback (char* message, cdevData& data, 
			     cmlogBrCallback callback, void* arg)
{
  return browser_.queryCallback (message, data, callback, arg);
}


int
cmlogXuiBrowser::removeMonitorCallback (char* attr, 
					cmlogBrCallback callback, void* arg)
{
  return browser_.removeMonitorCallback (attr, callback, arg);
}



				
	
