//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      cmlog genetic Server/Client IO Handler
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogIO.cc,v $
//   Revision 1.3  2001/07/25 14:30:19  chen
//   64 BIT Initial Port
//
//   Revision 1.2  2000/06/20 19:35:16  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.2  1998/03/27  13:59:02  chen
// implement logServerMsg and put updatingLoggingVariable here
//
// Revision 1.1  1997/08/01  15:29:05  bickley
// Added cmlog to application development system.
//
//
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <cmlogProtocol.h>
#include <cmlogAcceptor.h>
#include <cmlogUtil.h>
#include <cmlogData.h>
#include <cmlogDataManager.h>
#include "cmlogIO.h"

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogIO::cmlogIO (ACE_Reactor& r, 
		  cmlogDataQueue* queue)
:ACE_Event_Handler (), sockStream_ (), acceptor_ (0), 
 reactor_ (r), clientHost_ (0), dataQueue_ (queue), 
 connected_ (0), refcount_ (1), lock_ ()
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogIO class object\n");
#endif
}
#else
cmlogIO::cmlogIO (ACE_Reactor& r, 
		  cmlogDataQueue* queue,
		  cmlogDataQueue* comdqueue)
:ACE_Event_Handler (), sockStream_ (), acceptor_ (0), 
 reactor_ (r), clientHost_ (0), dataQueue_ (queue), comdQueue_ (comdqueue),
 connected_ (0), refcount_ (1)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogIO class object\n");
#endif
}
#endif

cmlogIO::~cmlogIO (void)
{
  sockStream_.close ();
  if (clientHost_)
    delete []clientHost_;
  connected_ = 0;
}

cmlogIO::operator ACE_SOCK_Stream & (void)
{
  return sockStream_;
}

int
cmlogIO::peerAddr (const ACE_INET_Addr& addr)
{
  if (clientHost_)
    delete []clientHost_;
  clientHost_ = new char[::strlen (addr.get_host_name ()) + 1];
  ::strcpy (clientHost_, addr.get_host_name ());
  return 0;
}

int
cmlogIO::handle_close (int, ACE_Reactor_Mask)
{
  // deallocate all the resources since this one is created dynamically.
  // Reactor will only remove it from its array and calls this handle_close

  // check all callback lists and remove all related callbacks

  // remove this client from client list
  assert (acceptor_);
  acceptor_->remove (this);
  return 0;
}

int
cmlogIO::get_handle (void) const
{
  return sockStream_.get_handle ();
}

void
cmlogIO::manager (cmlogAcceptor* mana)
{
  acceptor_ = mana;
}

int
cmlogIO::setSockOption (int level, int option, void *optval, int optlen) const
{
  return sockStream_.set_option (level, option, optval, optlen);
}

void
cmlogIO::enable (int signum)
{
  sockStream_.enable (signum);
}

void
cmlogIO::disable (int signum)
{
  sockStream_.disable (signum);
}

void
cmlogIO::cleanup (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  if (--refcount_ == 0) {
#ifdef _CMLOG_DEBUG
    printf ("Clean up IO object\n");
#endif
    delete this;
  }
}

void
cmlogIO::deactivate (void)
{
  // this will call handle_close
  reactor_.remove_handler (this, ACE_Event_Handler::READ_MASK);
}
  

int
cmlogIO::connected (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  return connected_;
}

void
cmlogIO::connectionFlag (int flag)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif

  if (flag > 0)
    connected_ = 1;
  else
    connected_ = 0;
}

void
cmlogIO::ref (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  refcount_ ++;
}

void
cmlogIO::deref (void)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  refcount_ --;
}

void
cmlogIO::sendToPeer (int command, cdevData& data, long id)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  cmlogPacket packet;
  cmlogMsg msg (command, data, id);
  int      n;

  if (packet.insert (msg) != -1) {
    n = sockStream_ << packet;
    if (n <= 0) 
      cmlogUtil::serverPrintf ("Fatal error: Cannot send result back to connected peer\n");
  }
  else
    cmlogUtil::serverPrintf ("Singal cmlogMsg overflow a packet\n");
  
  packet.empty ();
}

void
cmlogIO::sendResult (cmlogPacket& packet)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  int n = sockStream_ << packet;

  if (n <= 0) 
    cmlogUtil::serverPrintf ("Fatal: Cannot send result message back to caller\n");
}

void
cmlogIO::sendErrorBack (short code, cmlog_cdevMessage& message)
{
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cpMutexGuard guard (lock_);
#endif
  unsigned qid = message.getTransIndex ();
  unsigned cbkid = message.getForeignDataIndex ();
  char*    msg   = message.getMessage ();

  cdevData error;
  error.insert (cmlogUtil::CMLOG_RESULT_TAG, code);
  
  cmlog_cdevMessage ret (0, qid, 0, 0, cbkid, 0, 0, 0, msg, &error, 0, 0);
  cmlogMsg    retmsg (ret);
  
  // create a cmlogPacket
  cmlogPacket packet;
  
  packet.insert (retmsg);
  
  int n = sockStream_ << packet;

  if (n <= 0) 
    cmlogUtil::serverPrintf ("Fatal: Cannot send error message back to browser\n");
}

void
cmlogIO::updateLoggingDataVar (cdevData& data)
{
  cmlogData* ldata = 0;
  if (dataManager->findData ((char *)("loggingData"), ldata) == 0)
    (*ldata) = data;
}


#ifdef solaris
extern "C" int gethostname (char* name, int namelen);
#endif

void
cmlogIO::logServerMsg (char* msg)
{
  // get host name
  char hostname[256];
  int  namelen = sizeof (hostname);

  if (gethostname (hostname, namelen) != 0)
    strcpy (hostname, "unknown");

  cdevData data;

  data.insert (cmlogUtil::CMLOG_TEXT_TAG, msg);
  data.insert (cmlogUtil::CMLOG_HOST_TAG, hostname);
  data.insert (cmlogUtil::CMLOG_FACILITY_TAG, (char *)("cmlogServer"));
  data.insert (cmlogUtil::CMLOG_KEY_TAG, cmlogUtil::currentTime ());

  cmlogMsg sndmsg (CMLOG_ADD_DATA, data, 0);

  cmlogPacket packet;

  if (packet.insert (sndmsg) != 0)
    return;

  updateLoggingDataVar (data);

  // remember to free all msgs ansg msgs[] somewhere
  cmlogMsg** msgs = packet.messages ();

  for (int i = 0; i < packet.numberOfData (); i++) {
    // this client id is actually a pointer to this object
    msgs[i]->channelId ((void *)this);

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
    // msgs[i] memory will be deleted by service threads
    dataQueue_->enqueue ((void *)msgs[i]);
#else
    dataQueue_->enqueue (*msgs[i]);
    delete msgs[i];
#endif
  }
  packet.empty ();  // do not delete messages inside the packet
  delete []msgs;
}





