//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation of cmlog database file name handler
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogDbaseHandler.cc,v $
//   Revision 1.2  2001/07/25 14:30:19  chen
//   64 BIT Initial Port
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:29:02  bickley
// Added cmlog to application development system.
//
//
#include <cmlogUtil.h>
#include "cmlogDbaseHandler.h"

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogDbaseHandler::cmlogDbaseHandler (ACE_Reactor& reactor,
				      cmlogDatabase& dbase,
				      cmlogDatabase& cxtdbase,
				      cmlogClientAcceptor& clientm)
:reactorTimer (reactor), dbase_ (dbase), cxtdbase_ (cxtdbase), 
 cacceptor_ (clientm)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogDbaseHandler Class Object\n");
#endif
  // open database file
  char filename[256];
  char cxtfile [256];
  
  if (cmlogDbaseHandler::dbaseFileNames (filename, cxtfile) == 0) {
    // open a database
    if (dbase_.open  (filename, O_CREAT | O_RDWR) != CDEV_SUCCESS)
      cmlogUtil::serverPrintf ("Fatal: cmlog server cannot open file %s\n", 
			       filename);
    if (cxtdbase_.open  (cxtfile, O_CREAT | O_RDWR) != CDEV_SUCCESS)
      cmlogUtil::serverPrintf ("Fatal: cmlog server cannot open file %s\n",
			       cxtfile);
  }

  // register the file name timer (in msecs)
  auto_arm (cmlogUtil::DBASE_CHANGE_INTERVAL*1000*60, 
	    cmlogUtil::DBASE_CHANGE_INTERVAL*1000*60); 

  // get tag table file and read it and insert all tag to the tag table
  if (updateTagTableFromFile () != 0)
    writeToTagTableFile ();
}
#else
cmlogDbaseHandler::cmlogDbaseHandler (ACE_Reactor& reactor,
				      cmlogDataQueueWriter& wq,
				      cmlogClientAcceptor& clientm)

:reactorTimer (reactor), queue_ (wq), cacceptor_ (clientm)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogDbaseHandler Class Object\n");
#endif
  // open database file
  char filename[256];
  char cxtfile [256];
  
  if (cmlogDbaseHandler::dbaseFileNames (filename, cxtfile) == 0) {

    // create databases first
    cmlogDatabase dbase;
    cmlogDatabase cxtdbase;

    if (dbase.open  (filename, O_CREAT | O_RDWR) != CDEV_SUCCESS)
      cmlogUtil::serverPrintf ("Fatal: cmlog server cannot open file %s\n", 
			       filename);
    if (cxtdbase.open  (cxtfile, O_CREAT | O_RDWR) != CDEV_SUCCESS)
      cmlogUtil::serverPrintf ("Fatal: cmlog server cannot open file %s\n", 
			       cxtfile);
    dbase.close ();
    cxtdbase.close ();

    // write to shared memory segement to let other process to handle
    // all databases
    queue_.writeDbaseName (filename);
    queue_.writeCxtdbaseName (cxtfile);

    // pass change file name information to other processes
    cdevData unused;
    cmlogMsg msg (CMLOG_DBASE, unused, 0);
    queue_.enqueue (msg);
  }
  // register the file name timer (in msecs)
  auto_arm (cmlogUtil::DBASE_CHANGE_INTERVAL*1000*60, 
	    cmlogUtil::DBASE_CHANGE_INTERVAL*1000*60);  

  // get tag table file and read it and insert all tag to the tag table
  if (updateTagTableFromFile () != 0)
    writeToTagTableFile ();
}
#endif

cmlogDbaseHandler::~cmlogDbaseHandler (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogDbaseHandler Class Object\n");
#endif
  dis_arm ();
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  // close database
  dbase_.flush ();
  dbase_.close ();

  // close cxt database
  cxtdbase_.flush ();
  cxtdbase_.close ();
#endif
}

void
cmlogDbaseHandler::timer_callback (void)
{
  char filename[256];
  char cxtfile [256];
  
  if (cmlogDbaseHandler::dbaseFileNames (filename, cxtfile) == 0) {
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
    // since cmlogDatabase interface is thread safe, it is ok
    // to change dbase here

    // close old database
    dbase_.flush ();
    dbase_.close ();
    // close old cxt database
    cxtdbase_.flush ();
    cxtdbase_.close ();
    // reopen a new database
    if (dbase_.open  (filename, O_CREAT | O_RDWR) != CDEV_SUCCESS)
      cmlogUtil::serverPrintf ("Fatal: cmlog server cannot open file %s\n",
			       filename);
    if (cxtdbase_.open  (cxtfile, O_CREAT | O_RDWR) != CDEV_SUCCESS)
      cmlogUtil::serverPrintf ("Fatal: cmlog server cannot open file %s\n", 
			       cxtfile);
    else {
      // dump all current connection context to this file for easy access
      cacceptor_.writeAllClientContext (cxtdbase_);
    }
#else
    queue_.writeDbaseName (filename);
    queue_.writeCxtdbaseName (cxtfile);
    // dump all current connection contexts to this file for easy access
    cacceptor_.writeAllClientContext (cxtfile);

    // pass change file name information to other processes
    cdevData unused;
    cmlogMsg msg (CMLOG_DBASE, unused, 0);
    queue_.enqueue (msg);
#endif
  }
}

int
cmlogDbaseHandler::dbaseFileNames (char* filename, char* cxtfilename)
{
  struct timeval tp;
  gettimeofday (&tp, 0);

  char filetag[80];
  sprintf (filetag, "%u", tp.tv_sec);
  // construct file name
  sprintf (filename, cmlogUtil::DATABASE_NAME, filetag);
  cmlogUtil::serverPrintf ("New database Filename is %s\n", filename);
  // construct context database file name
  sprintf (cxtfilename, cmlogUtil::CXTDBASE_NAME, filetag);
  cmlogUtil::serverPrintf ("New Context filename is %s\n", cxtfilename);
  return 0;
}

int
cmlogDbaseHandler::writeToTagTableFile (void)
{
  FILE *fd = ::fopen (cmlogUtil::TAGTABLE_NAME, "w");

  if (fd) {
    cdevGlobalTagTable::tagTable ()->asciiDump (fd); 
    fclose (fd);
    return 0;
  }
  return -1;
}

int
cmlogDbaseHandler::updateTagTableFromFile (void)
{
  char** tags;
  int*   tagvalues;
  int    num = 0, i = 0;

  FILE *fd = ::fopen (cmlogUtil::TAGTABLE_NAME, "r");

  if (fd) {
    char buffer[128], tag[80];
    int  tagval = 0;

    // first check how many tags: not efficient, 
    // but how many tags we are talking about
    while (!feof (fd)) {
      ::fgets (buffer, sizeof (buffer), fd);
      if (::sscanf (buffer, "%d %s", &tagval, tag) == 2)
	num ++;
    }
    ::fseek (fd, 0L, SEEK_SET);  // reset to the begining of the file

    if (num > 0) {
      // allocate space for tag/tagvalue array
      tags = new char*[num];
      tagvalues = new int[num];

      i = 0;
      while (!feof (fd)) {
	::fgets (buffer, sizeof (buffer), fd);
	if (::sscanf (buffer, "%d %s", &tagval, tag) == 2) {
	  tags[i] = new char[::strlen (tag) + 1];
	  ::strcpy (tags[i], tag);
	  tagvalues[i++] = tagval;
	}
      }
      assert (i == num);

    }
    ::fclose (fd);
  
    for (i = 0; i < num; i++) {
      cdevData::insertTag (tagvalues[i], tags[i]);
      delete []tags[i];
    }
    if (num > 0) {
      delete []tags;
      delete []tagvalues;
    }
    return 0;
  }
  return -1;
}
