//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      cmlog Server to Admin IO handler
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogAdminIO.cc,v $
//   Revision 1.3  2001/07/25 14:30:17  chen
//   64 BIT Initial Port
//
//   Revision 1.2  1999/12/13 21:24:38  chen
//   Add more error checking using BAD_PTR
//
//   Revision 1.1.1.1  1999/09/07 15:29:11  chen
//   CMLOG version 2.0
//
// Revision 1.2  1998/10/07  17:03:22  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.1  1997/08/01  15:28:42  bickley
// Added cmlog to application development system.
//
//
#include <sys/ioctl.h>
#include <sys/time.h>
#include <cmlogClntCxt.h>
#include <cmlogUtil.h>
#include <cmlogDataManager.h>
#include <cmlogData.h>
#include <cmlogAdminAcceptor.h>
#include "cmlogAdminIO.h"

extern sig_atomic_t    finished;

//==========================================================================
//         Implementation of cmlogAdminIO class
//==========================================================================
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
cmlogAdminIO::cmlogAdminIO (ACE_Reactor& r, 
			    cmlogDataQueue* q)
:cmlogIO (r, q)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogAdminIO Class Object\n");
#endif
}
#else
cmlogAdminIO::cmlogAdminIO (ACE_Reactor& r, 
			    cmlogDataQueue* q,
			    cmlogDataQueue* cq)
:cmlogIO (r, q, cq)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogAdminIO Class Object\n");
#endif
}
#endif

cmlogAdminIO::~cmlogAdminIO (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogAdminIO Class Object\n");
#endif
}

int
cmlogAdminIO::handle_close (int fd, ACE_Reactor_Mask mask)
{
  // call default handle close
  int st = cmlogIO::handle_close (fd, mask);

  // no other threads referencing this io object, it is safe to delete
  delete this;
  return st;
}

int
cmlogAdminIO::handle_input (int fd)
{
  return reader ();
}


int
cmlogAdminIO::reader (void)
{
  cmlogPacket packet;
  int         n = 0;

  n = sockStream_ >> packet;
#if 0
  printf ("Admin IO received %d bytes\n", n);
#endif
  if (n <= 0) {
#ifdef _CMLOG_DEBUG
    printf ("Admin connection closed\n");
#endif
    packet.reset ();
    return -1;
  }

  return processData (packet);
}

int
cmlogAdminIO::processData (cmlogPacket& packet)
{
  unsigned    opcode;
  int         needp = 1;
  char*       attribute;
  cmlogData*  mdata = 0;

  // remember to free all msgs ansg msgs[] somewhere
  cmlogMsg** msgs = packet.messages ();
  if (msgs == (cmlogMsg **)CMLOG_BAD_PTR) {
    // log server msg
    char tmp[80];
    sprintf (tmp, "Admin %s processing message error. Closing connection\n", 
	     clientHost_);
    logServerMsg (tmp);
    printf (tmp);
    
    return -1;
  }

  for (int i = 0; i < packet.numberOfData (); i++) {
    cmlog_cdevMessage& idata = (*msgs[i]);
    opcode = msgs[i]->type ();

    switch (opcode) {
    case CMLOG_GET_ATTR:
      attribute = idata.getMessage ();
      // try to find this attribute

      if (dataManager->findData (attribute, mdata) != 0) { // not found
	sendErrorBack (CMLOG_NOTFOUND, idata);
	continue;
      }
      // update client pointer
      msgs[i]->channelId ((void *)this);
      // get value and send back
      mdata->getValue (&idata);
      // we do not put this on the queue
      needp = 0;
      // free memory
      delete msgs[i];
      break;
    case CMLOG_EXIT:
      needp = 0;
      {
	// ask manager to shutdown all client daemons
	cmlogAdminAcceptor* adminM = (cmlogAdminAcceptor *)acceptor_;
	adminM->shutdownAllClientD ();
      }
      delete msgs[i];
      // send terminate signal
      kill (getpid (), SIGQUIT);
      break;
    default:
#ifdef _CMLOG_DEBUG
      printf ("No idea what this protocl: %d  stands for \n", opcode);
#endif
      needp = 0;
      delete msgs[i];
      break;
    }
    
    if (needp) { // need service thread to process
      msgs[i]->channelId ((void *)this);
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
      // msgs[i] memory will be deleted by service threads
      dataQueue_->enqueue ((void *)msgs[i]);
#else
      dataQueue_->enqueue (*msgs[i]);
      comdQueue_->enqueue (*msgs[i]);
      delete msgs[i];
#endif
    }
  }
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  packet.empty ();  // do not delete messages inside the packet
#else
  packet.empty ();  // do delete messages inside tha packet
#endif
  delete []msgs;
}
