//-----------------------------------------------------------------------------
// Copyright (c) 1991,1992 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
// Email: coda@cebaf.gov  Tel: (804) 249-7101  Fax: (804) 249-7363
//-----------------------------------------------------------------------------
// 
// Description:
//	 Simple FIFO with void * as its content
//	
// Author:  Jie Chen
//       CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogDataQueue.h,v $
//   Revision 1.1.1.1  1999/09/07 15:29:10  chen
//   CMLOG version 2.0
//
// Revision 1.1  1997/08/01  15:27:20  bickley
// Added cmlog to application development system.
//
//
#ifndef _CMLOG_DATA_QUEUE_H
#define _CMLOG_DATA_QUEUE_H

#include <cmlogConfig.h>

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)

#include <cmlogSlist.h>
typedef void* cmlogQItem; 

class cmlogDataQueue;

#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)

// multiple waiters of t he data queue. These waiters will 
// follow strict FIFO behaviour to dequeue the data queue

class cmlog_dq_waiter{
public:
  cmlog_dq_waiter   *next_;
  int               wakeup_;
  cpConditionMutex  cv_;
  cp_thread_t       thr_;
private:
  // constructor
  cmlog_dq_waiter   (cpMutex& lock, cp_thread_t t);
  ~cmlog_dq_waiter  (void);

  friend class cmlogDataQueue;
};
#endif

//======================================================================
//	class cmlogDataQueue
//		cmlogDataQueue class: queue implmented doubleEndedList
//              Note: this class has mutex a inherited from cmlogSlist
//======================================================================
class cmlogDataQueue: public cmlogDoubleEndedSlist
{
public:
  // constructor and destructor
  cmlogDataQueue                   (void);

  // operations
  cmlogQItem        dequeue        (void);
  void              enqueue        (cmlogQItem value);
  cmlogQItem        front          (void);
  unsigned int      numElements    (void);
 
protected:
#if defined (CMLOG_USE_THREAD) && defined (_REENTRANT)
  cmlog_dq_waiter* waiters_head_;
  cmlog_dq_waiter* waiters_tail_;
  
  unsigned int     num_waiters_;
#endif

private:
  // deny access to assignment operation
  cmlogDataQueue   (const cmlogDataQueue &v);
  cmlogDataQueue& operator = (const cmlogDataQueue& q);
};

#endif

#if !defined (CMLOG_USE_THREAD) || !defined (_REENTRANT)
#include <ace/SV_Shared_Memory.h>
#include <ace/SV_Semaphore_Complex.h>
#include <cmlogMsg.h>
#include <cmlogDatabase.h>

//======================================================================
//      class cmlogQueueHandler
//               create all semaphores and shared memory
//======================================================================
class cmlogQueueHandler
{
public:
  cmlogQueueHandler    (void);
  ~cmlogQueueHandler   (void);

  // initialize the shared memory and so on
  // this routine must be called before one can do anything
  void init (void);

  // return shared memory address
  char* shmemPtr (void);

private:
  // all synchronization mechanism for data to database shared memory
  ACE_SV_Semaphore_Complex dlock_;     // lock for access shared memory
  ACE_SV_Semaphore_Complex disEmpty_;  // empty condition 
  ACE_SV_Semaphore_Complex disFull_;   // full condition
  ACE_SV_Shared_Memory     dshmem_;    // shared memory for data queue

  cmlogQueueHandler (const cmlogQueueHandler&);
  cmlogQueueHandler& operator = (const cmlogQueueHandler &);
};

//======================================================================
//	class cmlogShmHeader
//		Header Information kept in the shared memory
//              Every process gets all information such as
//              number of elements, head/tail of queue from
//              this shared memory header
//======================================================================
class cmlogShmHeader
{
public:
  // constructor and destructor
  cmlogShmHeader  (char* shmhead, unsigned int size);
  ~cmlogShmHeader (void);

  // all operations must be called with lock held

  // get header information from shared memory
  void            getHeaderInfo (void);
  // dump header information to the shared memory
  void            putHeaderInfo (void);
  // beginning of queue
  char*           beginningOfQueue (void);
  // end of queue
  char*           endOfQueue       (void);
  // overflow data queue: must be called afetr getHeaderInfo
  int             overflow         (cmlogMsg& msg);
  // round address if this address is over the end of queue
  // must be called after getHeaderInfo
  void            roundUpAddress   (char* &address);

  char            dbaseName_[CMLOG_DBASENAME_SIZE];
  char            cxtdbaseName_[CMLOG_DBASENAME_SIZE];
  unsigned int    numElements_;
  unsigned int    full_;
  char*           head_;
  char*           tail_;

private:
  char*                 shmem_;  // header of shared memroy block
  unsigned int          size_;   // size of the memory block
};

//======================================================================
//	class cmlogDataQueue
//		cmlogDataQueue class: generic class which is
//              a wrapper on the shared memory FIFO
//======================================================================
class cmlogDataQueue
{
public:
  enum {TODBASE = 1001, QUERY};
  // destructor
  ~cmlogDataQueue                       (void);

  // caller should release memory
  cmlogMsg*         front               (void);
  unsigned int      numElements         (void);

  // attach shared memory: default to attach
  // or not to attach again just figure out the header
  int               attach              (char* mem = 0);
  // return shared memory header
  char*             shmemHeader         (void);
  // return size in bytes
  unsigned int      shmemSize           (void);

  virtual void      dequeue             (cmlogMsg& value);
  virtual void      enqueue             (cmlogMsg& value);

protected:
  // constructor
  cmlogDataQueue                        (int type);

  ACE_SV_Semaphore_Complex lock_;     // lock for access shared memory
  ACE_SV_Semaphore_Complex isEmpty_;  // empty condition
  ACE_SV_Semaphore_Complex isFull_;   // full condition
  ACE_SV_Shared_Memory     shmem_;    // shared memory for data/command queue

  // internal routine to return number of elements
  // must be called with lock held
  unsigned int             numElements_i      (void);

  int                      type_;
  char*                    shmheader_;
  unsigned int             shmsize_;

private:
  // deny access to assignment operation
  cmlogDataQueue   (const cmlogDataQueue &v);
  cmlogDataQueue& operator = (const cmlogDataQueue& q);
};


//======================================================================
//  class cmlogDataQueueWriter
//        Parent process using this class to enqueue data items
//======================================================================
class cmlogDataQueueWriter: public cmlogDataQueue
{
public:
  // constructor
  cmlogDataQueueWriter                  (int type);

  virtual void      enqueue             (cmlogMsg& value);

  // write data base name to header
  void              writeDbaseName      (char* name);
  void              writeCxtdbaseName   (char* name);
};

//======================================================================
//  class cmlogDataQueueToDbase
//        All service processes using this class to dequeue data items
//        And to put to database file
//======================================================================
class cmlogDataQueueToDbase: public cmlogDataQueue
{
public:
  // constructor
  cmlogDataQueueToDbase                  (void);

  // create new process
  static  void* readerProcess            (void *);

  virtual void      dequeue              (cmlogMsg& value);

  // read database name from header
  void              readDbaseName        (char* namebuffer, int bufsize);
  void              readCxtdbaseName     (char* namebuffer, int bufsize);

  // register signal handler
  static int registerSignalHandlers (void);
  // all signal handlers 
  static void  signalFunc   (int signo);

  // global object database
  static cmlogDatabase* dbase;

private:
  // all signals we want to catch
  static int signals[];
  static int numSignals;
};
#endif

#endif

