//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      CMLOG User logging client Class
//
// Author:  
//      Jie Chen & danjin wu
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogClient.cc,v $
//   Revision 1.7  2001/07/25 14:25:37  chen
//   64 BIT Initial Port
//
//   Revision 1.6  2001/06/21 18:19:10  chen
//   Port to RedHat 7.1
//
//   Revision 1.5  2000/06/20 19:31:53  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.4  2000/03/17 19:14:40  chen
//   add cmlogParentPid and minor code change
//
//   Revision 1.3  2000/02/07 16:16:20  chen
//   to work with protocol 2.0
//
//   Revision 1.2  1999/09/20 17:01:31  chen
//   put coda fix
//
//   Revision 1.1.1.1  1999/09/07 15:29:09  chen
//   CMLOG version 2.0
//
// Revision 1.10  1999/01/12  14:28:13  chen
// fix a bug on connect (context)
//
// Revision 1.9  1998/11/19  17:36:53  chen
// Port to pc486 board
//
// Revision 1.8  1998/10/20  15:23:12  chen
// use _CMLOG_USE_THREAD to build thread safe client for Unix
//
// Revision 1.7  1998/07/01  16:42:17  chen
// Fix a bug introduced by wrong client syntax
//
// Revision 1.6  1998/06/16  17:31:10  chen
// fix bug on disconnect for vxworks
//
// Revision 1.5  1997/12/05  16:21:17  chen
// fix a small bug
//
// Revision 1.4  1997/10/20  14:01:08  chen
// allow client to wait for clientD a little longer
//
// Revision 1.3  1997/09/30  19:00:09  chen
// fix a potential problem when client daemon is shutdown
//
// Revision 1.2  1997/09/05  13:06:22  chen
// handle errLogAddListener and small bug fix on vxworks
//
// Revision 1.1  1997/08/01  15:26:35  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogProtocol.h>
#include "cmlogClient.h"


#define CMLOG_CODA (char *)("CODA")

#define CMLOG_DEFAULT_MSG (char *)("Wrong Client Syntax")

#ifdef solaris
extern "C" int gethostname (char* name, int namelen);
#endif

#ifdef __vxworks
#include <cpSynch.h>

extern "C" unsigned short cmlogVxClientId (void);
extern "C" int            cmlogVxFifoFd   (void);
extern "C" int            cmlogVxFifoFull (void);
extern "C" int            cmlogVxIncDroppedMsgsCount (void);

extern     int            cmlog_clientd_running;
extern cpMutex*           cmlog_rlock;
extern cpConditionMutex*  cmlog_rcond;

cmlogClient* cmlogClient::gclient_ = 0;

#if CPU_FAMILY == I80X86
#include <remLib.h>
#endif

#endif

//=======================================================================
//  Implementation of cmlogClientTagCallback
//=======================================================================
cmlogClientTagCallback::cmlogClientTagCallback (cmlogClient* client)
:cdevTagTableCallback (), client_ (client)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogClientTagCallback Class Object\n");
#endif
  // empty
}

cmlogClientTagCallback::~cmlogClientTagCallback (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogClientTagCallback Class Object\n");
#endif
  // empty
}

void
cmlogClientTagCallback::callback (int tag, char* name)
{
  client_->sendTagMapInfo (tag, name);
}


//========================================================================
//     Implementation of cmlogClient
//========================================================================
static int cmlogUdpBackOff[]={
  1, 2, 4, 8, 16, 32
};

/* try to connect to local client daemon in maximum number of times      */
/* when you change the following number, make sure to change above array */
#define  CMLOG_MAX_UDP_RETRIES 6

cmlogClient::cmlogClient (char* progname)
:connected_ (0), clientId_ (-1), fd_ (0), fifo_ (0),
 filter_ (CMLOG_MAX_SEVERITY, CMLOG_MAX_VERBOSITY, CMLOG_MAX_LIST_SIZE)
{
#ifdef _TRACE_OBJECTS
  printf ("Create cmlogClient Class Object\n");
#endif

#ifdef _CMLOG_USE_THREAD
  cmlog_rlock = &lock_;
#endif

  if (progname) {
    progname_ = new char[::strlen (progname) + 1];
    ::strcpy (progname_, progname);
  }
  else {
    progname_ = new char[::strlen ("UNKNOWN") + 1];
    ::strcpy (progname_, "UNKNOWN");
  }
  
  // setup all tags
  cmlogUtil::setTags ();

  // register tag table callback
  tagCbk_ = new cmlogClientTagCallback (this);
  cdevData::addTagCallback (tagCbk_);

  // get all misc user program information
  getUserInfo ();
}

cmlogClient::~cmlogClient (void)
{
#ifdef _TRACE_OBJECTS
  printf ("Delete cmlogClient Class Object\n");
#endif
  delete []progname_;
  delete []username_;
  delete []hostname_;
  delete []displayname_;

  if (connected_) {
#ifndef __vxworks
    close (fifo_);
#endif
  }

  fifo_ = 0;

  // delete tag table callback object
  delete tagCbk_;
}

int
cmlogClient::connect (int numConnectionRetries) 
{
  if (!connected_) {
    int st;
#ifndef __vxworks

#ifdef _CMLOG_USE_THREAD
    cpMutexGuard guard (lock_);
#endif
    
    if (udpToClntDOpen (CMLOG_CLNT_PORT) != 0) 
      return CMLOG_ERROR;

    if ((st = talkToClntD (numConnectionRetries)) == 0) 
      return CMLOG_CONN_TIMEOUT;
    else if (st == -1) 
      return CMLOG_ERROR;
    else {
      close (fd_);
      fd_ = -1;
#ifdef _CMLOG_DEBUG
      printf ("Get client id %d\n", clientId_);
#endif
      if (clientId_ < 1234)  /* Daemon says no connection */
	return CMLOG_ERROR;

      if (openNamedPipe () == -1)
	return CMLOG_ERROR;
      if (sendConnectionInfo () == -1)
	return CMLOG_ERROR;
      connected_ = 1;
    }
#else  /* not vxworks */
    while (!cmlog_rcond) {
      printf ("cmlogClientD is not loaded yet\n");
      taskDelay (sysClkRateGet ());
    }

    /* wait for the client daemon to start */
    cmlog_rlock->acquire ();
    while (!cmlog_clientd_running) {
      printf ("Waiting for cmlogClientD to start up\n");
      cmlog_rcond->wait ();
    }

    /* now let us talk to the daemon       */
    clientId_ = cmlogVxClientId ();
#ifdef _CMLOG_DEBUG
    printf ("Get client id %d\n", clientId_);
#endif
    if (clientId_ < 1234) {
      cmlog_rlock->release ();
      return CMLOG_ERROR;
    }

    if (openNamedPipe () == -1) {
      cmlog_rlock->release ();
      return CMLOG_ERROR;
    }
    if (sendConnectionInfo () == -1) {
      cmlog_rlock->release ();
      return CMLOG_ERROR;
    }
    connected_ = 1;      
    cmlog_rlock->release ();
#endif /* vxworks */
  }
  return CMLOG_SUCCESS;
}  

int
cmlogClient::connect (const cdevData& context, int numConnectionRetries) 
{
  if (!connected_) {
    int st;
#ifndef __vxworks

#ifdef _CMLOG_USE_THREAD
    cpMutexGuard guard (lock_);
#endif

    if (udpToClntDOpen (CMLOG_CLNT_PORT) != 0)
      return CMLOG_ERROR;
  
    if ((st = talkToClntD (numConnectionRetries)) == 0)
      return CMLOG_CONN_TIMEOUT;
    else if (st == -1)
      return CMLOG_ERROR;
    else {
      close (fd_);
      fd_ = -1;
#ifdef _CMLOG_DEBUG
      printf ("Get client id %d\n", clientId_);
#endif
      if (openNamedPipe () == -1)
	return CMLOG_ERROR;
      if (sendConnectionInfo ((cdevData *)&context) == -1) 
	return CMLOG_ERROR;
      connected_ = 1;
    }
#else  /* not vxworks */
    while (!cmlog_rcond) {
      printf ("cmlogClientD is not loaded yet\n");
      taskDelay (sysClkRateGet ());
    }

    /* wait for the client daemon to start */
    cmlog_rlock->acquire ();
    while (!cmlog_clientd_running) {
      printf ("Waiting for cmlogClientD to start up\n");
      cmlog_rcond->wait ();
    }
    /* now let us talk to the daemon       */
    clientId_ = cmlogVxClientId ();
#ifdef _CMLOG_DEBUG
    printf ("Get client id %d\n", clientId_);
#endif
    if (clientId_ < 1234) {
      cmlog_rlock->release ();
      return CMLOG_ERROR;
    }

    if (openNamedPipe () == -1) {
      cmlog_rlock->release ();
      return CMLOG_ERROR;
    }

    if (sendConnectionInfo ((cdevData *)&context) == -1) {
      cmlog_rlock->release ();
      return CMLOG_ERROR;
    }
    connected_ = 1;    
    cmlog_rlock->release ();
#endif
  }
  return CMLOG_SUCCESS;
}  

int
cmlogClient::connected (void) const
{
#ifdef __vxworks
  cpMutexGuard guard (*cmlog_rlock);
#endif
  return connected_;
}

int
cmlogClient::disconnect (void)
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif

#ifndef __vxworks
  if (connected_) {
    sendDisconnInfo ();

    connected_ = 0;
    close (fifo_);
    fifo_ = 0;
  }
#endif

  return CMLOG_SUCCESS;
}

int
cmlogClient::setThrottle (char* tag, int size, int limit, double delta)
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif

  return filter_.setThrottle (tag, size, limit, delta);
}

int
cmlogClient::getThrottle (char** tag, int& size, int& limit, double& delta)
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif
  return filter_.getThrottle (tag, size, limit, delta);
}

int
cmlogClient::setSeverityThreshold (int th)
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif
  return filter_.setSeverityThreshold (th);
}

int 
cmlogClient::getSeverityThreshold (void) const
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif
  return filter_.getSeverityThreshold ();
}

int 
cmlogClient::getVerbosityThreshold (void) const
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif
  return filter_.getVerbosityThreshold ();
}

int 
cmlogClient::setVerbosityThreshold (int thresh)
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif
  return filter_.setVerbosityThreshold (thresh);
}

int
cmlogClient::postData (cdevData& data)
{
  // check if it is in the interrupt service routines
  CMLOG_ISR_CHECK ();

#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif

  if (connected_) {
    if (filter_.filter (data) != CMLOG_FILTERED) {
#ifndef __vxworks
      // currently standard vxworks has no timestamp capability
      // check if a time stamp is present or it has wrong type of value
      cdevDataTypes datatype = data.getType (cmlogUtil::CMLOG_KEY_TAG);
      if (datatype == CDEV_INVALID ||
	  datatype >= CDEV_STRING ||
	  datatype <= CDEV_UINT16) {
	data.insert  (cmlogUtil::CMLOG_KEY_TAG, cmlogUtil::currentTime ());
      }
#endif

      cmlog_cdevMessage message (clientId_, 0, 0, 0, 0,
				 CMLOG_ADD_DATA, 0, 0, 0, 
				 &data, 0, 0);
      cmlogMsg msg (message);

#ifdef __vxworks
      if (cmlogVxFifoFd () == 0) {
	// client daemon must have gone
	printf ("Client daemon is gone\n");
	connected_ = 0;
	return CMLOG_NOTCONSERVER;
      }

      if (cmlogVxFifoFull ()) {
	cmlogVxIncDroppedMsgsCount ();
	return CMLOG_FILTERED;
      }
#endif
      
      if (fifo_ << msg > 0)
	return CMLOG_SUCCESS;
      return CMLOG_ERROR;
    }
    return CMLOG_FILTERED;
  }
  else {
    char output[256];

    if (data.get (cmlogUtil::CMLOG_TEXT_TAG, output, sizeof (output))
	== CDEV_SUCCESS) 
      fprintf (stderr, "%s\n", output);
    
    return CMLOG_NOTCONSERVER;
  }
}

int
cmlogClient::postStaticData (cdevData& data)
{
  // check if it is in the interrupt service routines
  CMLOG_ISR_CHECK ();

#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif

  if (connected_) {
    if (filter_.filter (data) != CMLOG_FILTERED) {
#ifndef __vxworks
      // insert time stamp value
      // currently standard vxworks has no timestamp capability
      data.insert  (cmlogUtil::CMLOG_KEY_TAG, cmlogUtil::currentTime ());
#endif

      cmlog_cdevMessage message (clientId_, 0, 0, 0, 0,
				 CMLOG_ADD_DATA, 0, 0, 0, 
				 0, 0, 0);
      message.setData (&data, 1); // the 2nd argument denote permnant data

      cmlogMsg msg (message);

#ifdef __vxworks
      if (cmlogVxFifoFd () == 0) {
	// client daemon must have gone
	printf ("Client daemon is gone\n");
	connected_ = 0;
	return CMLOG_NOTCONSERVER;
      }

      if (cmlogVxFifoFull ()) {
	cmlogVxIncDroppedMsgsCount ();
	return CMLOG_FILTERED;
      }
#endif
      
      if (fifo_ << msg > 0)
	return CMLOG_SUCCESS;
      return CMLOG_ERROR;
    }
    return CMLOG_FILTERED;
  }
  else {
    char output[256];

    if (data.get (cmlogUtil::CMLOG_TEXT_TAG, output, sizeof (output))
	== CDEV_SUCCESS) 
      fprintf (stderr, "%s\n", output);
    
    return CMLOG_NOTCONSERVER;
  }
}

int
cmlogClient::postError (int verbosity, int severity, 
			int code, char* facility,
			char* format, ...)
{
  CMLOG_ISR_CHECK ();

  cdevData data;
  
  // all tags are registered
  data.insert (cmlogUtil::CMLOG_VERBOSITY_TAG, verbosity);
  data.insert (cmlogUtil::CMLOG_SEVERITY_TAG, severity);
  data.insert (cmlogUtil::CMLOG_CODE_TAG, code);
  data.insert (cmlogUtil::CMLOG_FACILITY_TAG, facility);
  data.insert (cmlogUtil::CMLOG_TEXT_TAG, CMLOG_DEFAULT_MSG);
  
  va_list arp;

  char *p;
  char strtmp[256], tagName[64];
  int ival, tag, i = 0;
  double dval;
  char cval;


  va_start(arp, format);
  
  for (p = format; *p; p++) {
    if (*p == ' ' || *p == '\t' || *p == '\r' || *p == '\n')
      continue;
    if (*p == '%') {
      switch(*++p) {
      case 'd': 
      case 'i':
      case 'x':
      case 'X':
      case 'o':
      case 'u':
	ival = va_arg(arp, int);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, ival);
	break;
      case 'f':
      case 'g':
      case 'F':
      case 'G':
	dval = va_arg(arp, double);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, dval);
	break;
      case 's':
	::strncpy(strtmp, va_arg(arp, char *), sizeof (strtmp) - 1);
	strtmp[sizeof (strtmp) - 1] = '\0';

	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, strtmp);
	break; 
      case 'c':
#if (__GNUC_MAJOR__ >2) || (__GNUC_MINOR__ >= 96) 
	cval = va_arg(arp, int);
#else
	cval = va_arg(arp, char);
#endif
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, cval);
	break;
      default:
	printf ("The argument type % %c not supported  \n", *p);
	break;
      }
      i = 0;
    }   //if (*p == '%')
    else if ((tagName[i++] = *p) == '=') {
      i--;
      tagName[i] = '\0'; 
    }
  }
  va_end(arp);  

  return postData (data);
}

int
cmlogClient::logMsg (char* format, ...)
{
  CMLOG_ISR_CHECK ();

  cdevData data;

  va_list arp;

  char *p;
  char strtmp[256], tagName[64];
  int ival, tag, i = 0;
  double dval;
  char cval;

  data.insert (cmlogUtil::CMLOG_TEXT_TAG, CMLOG_DEFAULT_MSG);
  va_start(arp, format);
  
  for (p = format; *p; p++) {
    if (*p == ' ' || *p == '\t' || *p == '\r' || *p == '\n')
      continue;
    if (*p == '%') {
      switch(*++p) {
      case 'd': 
      case 'i':
      case 'x':
      case 'X':
      case 'o':
      case 'u':
	ival = va_arg(arp, int);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, ival);
	break;
      case 'f':
      case 'g':
      case 'F':
      case 'G':
	dval = va_arg(arp, double);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, dval);
	break;
      case 's':
	::strncpy(strtmp, va_arg(arp, char *), sizeof (strtmp) - 1);
	strtmp[sizeof (strtmp) - 1] = '\0';

	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, strtmp);
	break; 
      case 'c':

#if (__GNUC_MAJOR__ >2) || (__GNUC_MINOR__ >= 96) 
	cval = va_arg(arp, int);
#else
	cval = va_arg(arp, char);
#endif
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, cval);
	break;
      default:
	printf ("The argument type % %c not supported  \n", *p);
	break;
      }
      i = 0;
    }   //if (*p == '%')
    else if ((tagName[i++] = *p) == '=') {
      i--;
      tagName[i] = '\0';
    }
  }
  va_end(arp);  

  return postData (data);
}

int
cmlogClient::getFd (void) const
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif
  return fifo_;
}

void
cmlogClient::getUserInfo (void)
{
#ifndef __vxworks
  // get user name and process id number
  struct passwd* pwsd = 0;
  if ((pwsd = getpwuid (getuid()) ) != 0) {
    username_ = new char[::strlen(pwsd->pw_name) + 1];
    ::strcpy (username_,pwsd->pw_name);
  }
  else {
    char buff[80];
    ::sprintf (buff, "userid_%d",getuid ());
    username_ = new char[::strlen (buff) + 1];
    ::strcpy (username_, buff);
  }
  taskid_ = ::getpid ();

  // get hostname and if possible X-window display information

  // get host name
  char host[128];
  int  len = sizeof (host);
  if (::gethostname (host, len) != 0)
    strcpy (host, "unknown");
  hostname_ = new char[::strlen (host) + 1];
  ::strcpy (hostname_, host);

  // get X window display name
  char* tmp = ::getenv ("DISPLAY");
  char  fullname[128];
  if (!tmp) {
    strcat (host, ":0.0");
    strcpy (fullname, host);
  }
  else {
    if (::strcmp (tmp, ":0.0") == 0) {
      ::strcpy (fullname, host);
      ::strcat (fullname, tmp);
    }
    else
      strcpy (fullname, tmp);
  }
  displayname_ = new char[::strlen (fullname) + 1];
  ::strcpy (displayname_, fullname);
#else
  BOOT_PARAMS params;

#if CPU_FAMILY==I80X86
  char usern[128], pass[128];
  ::remCurIdGet (usern, pass);
  username_ = new char[::strlen (usern) + 1];
  ::strcpy (username_, usern);

  char host[128];
  int len = sizeof (host);
  if (::gethostname (host, len) != 0)
    strcpy (host, "unknown");
  hostname_ = new char[::strlen (host) + 1];
  ::strcpy (hostname_, host);
#else
  nvramStore (&params);  // get all boot parameters

  username_ = new char[::strlen(params.usr) + 1];
  ::strcpy (username_,params.usr);

  hostname_ = new char[::strlen (params.targetName) + 1];
  ::strcpy (hostname_, params.targetName);
#endif

  taskid_ = taskIdSelf ();

  displayname_ = new char[::strlen ("VXWORKS") + 1];
  ::strcpy (displayname_, "VXWORKS");

#endif  /* vxworks */
}  

int
cmlogClient::udpToClntDOpen (unsigned short port)
{
  struct sockaddr_in my_addr;
  struct hostent*    hp = 0;

  // initialize the server inet address structure
  ::memset (&udp_addr_, 0, sizeof (udp_addr_));
  udp_addr_.sin_family = AF_INET;
  udp_addr_.sin_port = htons (port);

#ifndef __vxworks
  char   hostname[128];
  int    len = sizeof (hostname);

  gethostname (hostname, len);
  hp = gethostbyname (hostname);

  if (hp == 0) {
    fprintf (stderr, "Cannot find ip address of local host\n");
    return -1;
  }
  ::memcpy (&udp_addr_.sin_addr, hp->h_addr, hp->h_length);
#else
  int targetAddr = 0;
  if ((targetAddr = hostGetByName (hostname_)) != ERROR) 
    ::memcpy (&udp_addr_.sin_addr, &targetAddr, sizeof (targetAddr));
  else {
    fprintf (stderr, "Cannot find local ip addr for %s\n", hostname_);
    exit (1);
  }
#endif

  // open a socket
  if ((fd_ = socket (AF_INET, SOCK_DGRAM, 0)) < 0) {
    fprintf (stderr, "udp_open: cannot create socket\n");
    fd_ = -1;
    return -1;
  }

  // bind local address for us
  ::memset (&my_addr, 0, sizeof (my_addr));
  my_addr.sin_family = AF_INET;
  my_addr.sin_addr.s_addr = htonl (INADDR_ANY);
  my_addr.sin_port = htons (0);
  if (bind (fd_, (struct sockaddr *)&my_addr, sizeof (my_addr)) < 0) {
    fprintf (stderr, "udp error: bind error\n");
    close (fd_);
    fd_ = -1;
    return -1;
  }

  return 0;
}

int
cmlogClient::talkToClntD (int numConnRetries)
{
  char outbuff[1024];
  int  len = sizeof (outbuff);

  // find server port
  cdevData pdata;
  unsigned short port = findServerUdpPort ();
  pdata.insert (cmlogUtil::CMLOG_VALUE_TAG, port);

  cmlogMsg msg (CMLOG_USERLKCLNT, pdata, 0);

  if (cmlogMsg::msgToBuffer (msg, outbuff, &len) == -1)
    return -1;

  return cmlogClient::udpSendRecv (fd_, outbuff, len, 
				   (char *)&clientId_, sizeof (clientId_),
				   (struct sockaddr *)&udp_addr_,
				   sizeof (udp_addr_),
				   numConnRetries);
}

int
cmlogClient::udpSendRecv (int fd, 
			  char* outbuff, int outbytes,
			  char* inbuff,  int inbytes,
			  struct sockaddr*  destaddr, int destlen,
			  int numRetries)
{
  int n = 0, i = 0, nfound = 0;
  struct timeval tv;
  fd_set         readfd;
  int            realNumRetries;
  int            dstarted = 0;

  realNumRetries = 
    (numRetries < CMLOG_MAX_UDP_RETRIES) ? numRetries : CMLOG_MAX_UDP_RETRIES;
  if (realNumRetries <= 0)
    realNumRetries = 3;

  while (i < realNumRetries) {
    // send to server something
    if (sendto (fd, outbuff, outbytes, 0, destaddr, destlen) != outbytes) {
      fprintf (stderr, "Send error on udp socket %d\n", fd);
      return -1;
    }
    
    FD_ZERO (&readfd);
    FD_SET  (fd, &readfd);
    tv.tv_sec = cmlogUdpBackOff[i];
    tv.tv_usec = 0;

    do {
#ifndef __hpux
      nfound = select (fd + 1, &readfd, 0, 0, &tv);
#else

#ifndef __hpux10
      nfound = select (fd + 1, (int *)&readfd, (int *)0, (int *)0, &tv);
#else
      nfound = select (fd + 1, &readfd, 0, 0, &tv);
#endif

#endif
    }while (nfound == -1 && errno == EINTR);

    if (nfound == -1) { // something is wrong with file descriptor
      fprintf (stderr, "bad fd number %d\n", fd);
      return -1;
    }
    else if (nfound == 0) { // timeout
#ifndef __vxworks
      if (i >= realNumRetries - 1 && !dstarted) {
	cmlogClient::startClientD ();
	dstarted = 1;
	i = 0;
	// let child process to have a chance to run first
	sleep (1);
      }
#endif
      i++;
    }
    else {
      // receive from server
#ifdef __linux
      n = recvfrom (fd, inbuff, inbytes, 0, (struct sockaddr *)0, 
		    (socklen_t *)0);
#else
      n = recvfrom (fd, inbuff, inbytes, 0, (struct sockaddr *)0, (int *)0);
#endif
      if (n < 0) {
#ifndef __vxworks
	if (i >= realNumRetries - 1 && !dstarted) {
	  cmlogClient::startClientD ();
	  dstarted = 1;
	  i = 0;
	  // let child process to have a chance to run first
	  sleep (1);
	}
	i++;
#else
	return -1;
#endif
      }
      else
	return n;
    }
  }
  return 0;
}

int
cmlogClient::openNamedPipe (void)
{
  // now try to open named pipe
  unsigned long ipaddr = cmlogUtil::hostIpAddr ();
  char          pipepath[128];
  sprintf       (pipepath, CMLOG_CLNT_PIPE, ipaddr);

#ifndef __vxworks
  fifo_ = open (pipepath, O_WRONLY);
  if (fifo_ < 0) {
    fprintf (stderr, "Cannot open named pipe : %s\n", pipepath);
    return -1;
  }
#else
  // use fifo fd opened by client daemon
  fifo_ = cmlogVxFifoFd ();
#endif
  return 0;
}

int
cmlogClient::sendConnectionInfo (cdevData* cxt)
{
  // first get tag map information
  char**     tagnames;
  int*       tagvalues;
  int        ntags = 0;
  cdevData   tagmap, ccxt;

  cdevData::readTagTable (tagvalues, tagnames, ntags);
  if (ntags == 0) 
    fprintf (stderr, "no tags found in the tag table\n");
  else {
    // integer values are inserted with tag value 1
    tagmap.insert (1, tagvalues, ntags);
    // names of tags are inserted with tag value 2
    tagmap.insert (2, tagnames, ntags);
    // free memory
    delete []tagvalues;
    delete []tagnames;
  }

  if (cxt == 0) { // use default context
    ccxt.insert (cmlogUtil::CMLOG_HOST_TAG, hostname_);
    ccxt.insert (cmlogUtil::CMLOG_PID_TAG,  taskid_);
    ccxt.insert (cmlogUtil::CMLOG_NAME_TAG, progname_);
    ccxt.insert (cmlogUtil::CMLOG_USER_TAG, username_);
    ccxt.insert (cmlogUtil::CMLOG_DISP_TAG, displayname_);
    ccxt.insert (cmlogUtil::CMLOG_CLASS_TAG, (char *)("UNKNOWN"));
    ccxt.insert (cmlogUtil::CMLOG_DOMAIN_TAG, (char *)("UNKNOWN"));
    ccxt.insert (cmlogUtil::CMLOG_FACILITY_TAG, (char *)("UNKNOWN"));
    cxt = &ccxt;
  }
  else {
    // check whether this context has pid info
    int cpid;
    if (cxt->get (cmlogUtil::CMLOG_PID_TAG, &cpid) != CDEV_SUCCESS)
      cxt->insert (cmlogUtil::CMLOG_PID_TAG, taskid_);
  }

  cmlog_cdevMessage message (clientId_, 0, 0, 0, 0,
			     CMLOG_CONN_INFO, 0, 0, 0, 
			     cxt, 0, &tagmap);

  // create cmlogMsg
  cmlogMsg sendmsg (message);

  return fifo_ << sendmsg;
}

int
cmlogClient::sendTagMapInfo (int tagvalue, char* tagname)
{
#if defined (__vxworks) || defined (_CMLOG_USE_THREAD)
  cpMutexGuard guard (*cmlog_rlock);
#endif

  cdevData   tagmap;
  
  // integer values are inserted with tag value 1
  tagmap.insert (1, tagvalue);
  // names of tags are inserted with tag value 2
  tagmap.insert (2, tagname);

  cmlog_cdevMessage message (clientId_, 0, 0, 0, 0,
		       CMLOG_TAGMAP, 0, 0, 0, 
		       0, 0, &tagmap);

  // create cmlogMsg
  cmlogMsg sendmsg (message);

  return fifo_ << sendmsg;
}

int
cmlogClient::sendDisconnInfo (void)
{
  cmlog_cdevMessage message (clientId_, 0, 0, 0, 0,
			     CMLOG_CLOSE_CONN, 0, 0, 0,
			     0, 0, 0);
  
  cmlogMsg sendmsg (message);
  return fifo_ << sendmsg;
}

unsigned short
cmlogClient::findServerUdpPort (void)
{
  // find udp server port to broadcast to
  char* tmp = ::getenv ("CMLOG_PORT");
  int   tmpport = 0;
  if (tmp)
    sscanf (tmp, "%d", &tmpport);
  else
    tmpport = CMLOG_PORT;
  
  unsigned short brdPort = (unsigned short)tmpport;
  return brdPort;
}


#ifdef __vxworks

static char fbuf[] = "verbosity=%d severity=%d code=%d facility=%s";

/* The following code are provided by Paul Banta */
int
cmlogClient::nvramStore (BOOT_PARAMS* params)
{
#define BOOT_LINE_LEN	1024
#ifndef	NV_RAM_SIZE
#define	NV_RAM_SIZE	254
#endif	/* NV_RAM_SIZE */

  char boot_line [BOOT_LINE_LEN];
  
  if (sysNvRamGet (boot_line, NV_RAM_SIZE, 0) == -1 ||
      boot_line [0] == EOS)
    {
      printf ("Unable to get from NV Ram\n");
      exit (1);
    }

  bootStringToStruct (boot_line, params);

#if 0
  printf ("      bootDev: %s\n", params->bootDev);
  printf ("     hostName: %s\n", params->hostName);
  printf ("   targetName: %s\n", params->targetName);
  printf ("          ead: %s\n", params->ead);
  printf ("          bad: %s\n", params->bad);
  printf ("          had: %s\n", params->had);
  printf ("          gad: %s\n", params->gad);
  printf ("     bootFile: %s\n", params->bootFile);
  printf ("startupScript: %s\n", params->startupScript);
  printf ("          usr: %s\n", params->usr);
  printf ("       passwd: %s\n", params->passwd);
  printf ("        other: %s\n", params->other);
  printf ("      procNum: %x\n", params->procNum);
  printf ("        flags: %x\n", params->flags);
#endif 
  return 0;
}

int
cmlogClient::logMsgI (char* format, int arg0, int arg1, int arg2,
		      int     arg3, int arg4, int arg5, int arg6,
		      int     arg7, int arg8, int arg9)
{
  if (!intContext ())
    return CMLOG_ERROR;

  packet_.set (clientId_, CMLOG_ISR_DATA, format, arg0, arg1, arg2, arg3,
	       arg4, arg5, arg6, arg7, arg8, arg9);

  size_t len = 0;
  int    wlen = 0;
  int    header;

  // the reason to write twice to the pipe::
  // VxWorks pipe device is a message based device instead of stream based
  // device such as socket. If receiver side of the pipe reads the pipe
  // twice, the remaining part of the message will be discarded.
  len = packet_.size ();
  if (len > CMLOG_CLNT_PIPE_MAX_BYTES - sizeof(int)) {
    ::logMsg (format, arg0, arg1, arg2, arg3, arg4, arg5);
    return CMLOG_DROPPED;
  }

  // write out header information
  header = cmlogMsg::setMsgHeader (len);
  wlen = write (fifo_, (char *)&header, sizeof(int));
  if (wlen != sizeof (int)) {
    ::logMsg (format, arg0, arg1, arg2, arg3, arg4, arg5);
    return CMLOG_DROPPED;
  }  

  if (packet_.streamOut (sbuf_, len, CMLOG_CLNT_PIPE_MAX_BYTES) == 0) {
    if ((wlen = write (fifo_, sbuf_, len)) == len) 
      return CMLOG_SUCCESS;
    else 
      ::logMsg (format, arg0, arg1, arg2, arg3, arg4, arg5);
  }
  else 
    ::logMsg (format, arg0, arg1, arg2, arg3, arg4, arg5);
  
  return CMLOG_DROPPED;
}

int
cmlogClient::postErrorI (int verbosity, int severity, 
			 int code, char* facility,
			 char* format, int arg0, int arg1,
			 int   arg2,   int arg3, int arg4,
			 int   arg5)
{
  if (!intContext ())
    return CMLOG_ERROR;

  int  i = 0;

  bcopy (fbuf, format_, strlen (fbuf));
  i += (strlen (fbuf));

  bcopy (" ", &(format_[i]), 1);
  i += 1;

  if (i + strlen (format) + 1> CMLOG_CLNT_ISR_FORMAT_STR_LEN)
    return CMLOG_DROPPED;

  bcopy (format, &(format_[i]), strlen (format) + 1);
  i += (strlen (format) + 1);

  return logMsgI (format_, verbosity, severity, code, (int)facility,
		  arg0, arg1, arg2, arg3, arg4, arg5);
}

int
cmlog_logmsgI (cmlog_client_t client,
	       int verbosity, int severity, 
	       int code, char* facility,
	       char* format, int arg0, int arg1,
	       int   arg2,   int arg3, int arg4,
	       int   arg5)
{
  int  i;
  cmlogClient* cl;

  i = 0;
  cl = (cmlogClient *)client;

  if (client == 0)
    return CMLOG_ERROR;

  if (!intContext ())
    return CMLOG_ERROR;


  bcopy (fbuf, cl->format_, strlen (fbuf));
  i += (strlen (fbuf));

  bcopy (" ", &(cl->format_[i]), 1);
  i += 1;

  if (i + strlen (format) + 1> CMLOG_CLNT_ISR_FORMAT_STR_LEN)
    return CMLOG_DROPPED;

  bcopy (format, &(cl->format_[i]), strlen (format) + 1);
  i += (strlen (format) + 1);

  return cl->logMsgI (cl->format_, verbosity, severity, code, (int)facility,
		      arg0, arg1, arg2, arg3, arg4, arg5);
}  

int
cmlog_logtextI (cmlog_client_t client,
		char* format, 
		int arg0, int arg1,
		int   arg2,   int arg3, int arg4,
		int   arg5,   int arg6, int arg7,
		int   arg8,   int arg9)
{
  size_t len;
  int    wlen;
  cmlogClient* cl;

  len = 0;
  wlen = 0;
  cl = (cmlogClient *)client;

  if (client == 0)
    return CMLOG_ERROR;

  if (!intContext ())
    return CMLOG_ERROR;


  cl->packet_.set (cl->clientId_, 
		   CMLOG_ISR_DATA, format, arg0, arg1, arg2, arg3,
		   arg4, arg5, arg6, arg7, arg8, arg9);


  // the reason to write twice to the pipe::
  // VxWorks pipe device is a message based device instead of stream based
  // device such as socket. If receiver side of the pipe reads the pipe
  // twice, the remaining part of the message will be discarded.
  len = cl->packet_.size ();
  if (len > CMLOG_CLNT_PIPE_MAX_BYTES - sizeof(int)) {
    ::logMsg (format, arg0, arg1, arg2, arg3, arg4, arg5);
    return CMLOG_DROPPED;
  }

  // write out header
  int header = cmlogMsg::setMsgHeader (len);
  wlen = write (cl->fifo_, (char *)&header, sizeof (int));
  if (wlen != sizeof (int)) {
    ::logMsg (format, arg0, arg1, arg2, arg3, arg4, arg5);
    return CMLOG_DROPPED;
  }  

  if (cl->packet_.streamOut (cl->sbuf_, len, CMLOG_CLNT_PIPE_MAX_BYTES) == 0) {
    if ((wlen = write (cl->fifo_, cl->sbuf_, len)) == len) 
      return CMLOG_SUCCESS;
    else 
      ::logMsg (format, arg0, arg1, arg2, arg3, arg4, arg5);
  }
  else 
    ::logMsg (format, arg0, arg1, arg2, arg3, arg4, arg5);
  
  return CMLOG_DROPPED;
}

cmlogClient*
cmlogClient::logClient (void)
{
  if (cmlogClient::gclient_ == 0) 
    cmlogClient::gclient_ = new cmlogClient ("cmlogVxLogClient");
  return cmlogClient::gclient_;
}
  
#else
void
cmlogClient::startClientD (void)
{
  char* port = 0;
  if ((port = getenv ("CMLOG_PORT")) != 0) {
    char fullcomd[128];
    sprintf (fullcomd, "cmlog_activate -s %s", port);
    // system ("cmlog_activate -s port -l /tmp/cmlogClietD.log");
    system (fullcomd);
  }
  else {
    // system ("cmlog_activate -l /tmp/cmlogClientD.log");
    system ("cmlog_activate");
  }
}
    
#endif


/****************************************************************************
 *            Implementation of C interface                                 *
 ***************************************************************************/
cmlog_client_t
cmlog_open (char* progname)
{
#ifndef __vxworks
  cmlogClient* client = new cmlogClient (progname);
#else
  cmlogClient* client = cmlogClient::logClient ();
#endif
  
  if (client->connect () == CMLOG_SUCCESS)
    return (cmlog_client_t)client;
  else
    return 0;
}

void
cmlog_close (cmlog_client_t client)
{
  if (client != 0) {
    cmlogClient* cl = (cmlogClient *)client;
    cl->disconnect ();
  }
}

int
cmlog_logmsg (cmlog_client_t client,
	      int verbosity,
	      int severity, 
	      int code, 
	      char* facility,
	      char* format, ...)
{
  if (client == 0)
    return CMLOG_ERROR;

  CMLOG_ISR_CHECK ();

  cmlogClient* cl = (cmlogClient *)client;
  cdevData data;


  // all tags are registered
  data.insert (cmlogUtil::CMLOG_VERBOSITY_TAG, verbosity);
  data.insert (cmlogUtil::CMLOG_SEVERITY_TAG, severity);
  data.insert (cmlogUtil::CMLOG_CODE_TAG, code);
  data.insert (cmlogUtil::CMLOG_FACILITY_TAG, facility);
  data.insert (cmlogUtil::CMLOG_TEXT_TAG, CMLOG_DEFAULT_MSG);
  
  va_list arp;

  char *p;
  char strtmp[256], tagName[64];
  int ival, tag, i = 0;
  double dval;
  char cval;

  va_start(arp, format);
  
  for (p = format; *p; p++) {
    if (*p == ' ' || *p == '\t' || *p == '\r' || *p == '\n')
      continue;
    if (*p == '%') {
      switch(*++p) {
      case 'd': 
      case 'i':
      case 'x':
      case 'X':
      case 'o':
      case 'u':
	ival = va_arg(arp, int);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, ival);
	break;
      case 'f':
      case 'g':
      case 'F':
      case 'G':
	dval = va_arg(arp, double);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, dval);
	break;
      case 's':
	::strncpy(strtmp, va_arg(arp, char *), sizeof (strtmp) -1);
	strtmp[sizeof (strtmp) -1] = '\0';

	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, strtmp);
	break; 
      case 'c':
#if (__GNUC_MAJOR__ >2) || (__GNUC_MINOR__ >= 96) 
	cval = va_arg(arp, int);
#else
	cval = va_arg(arp, char);
#endif	
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, cval);
	break;
      default:
	printf ("The argument type % %c not supported  \n", *p);
	break;
      }
      i = 0;
    }   //if (*p == '%')
    else if ((tagName[i++] = *p) == '=') {
      i--;
      tagName[i] = '\0'; 
    }
  }
  va_end(arp);  

  return cl->postData (data);
}

int
cmlog_logtext (cmlog_client_t client, char* format, ...)
{
  if (client == 0)
    return CMLOG_ERROR;

  CMLOG_ISR_CHECK ();

  cmlogClient* cl = (cmlogClient *)client;

  cdevData data;

  va_list arp;

  char *p;
  char strtmp[256], tagName[64];
  int ival, tag, i = 0;
  double dval;
  char cval;

  data.insert (cmlogUtil::CMLOG_TEXT_TAG, CMLOG_DEFAULT_MSG);
  va_start(arp, format);
  
  for (p = format; *p; p++) {
    if (*p == ' ' || *p == '\t' || *p == '\r' || *p == '\n')
      continue;
    if (*p == '%') {
      switch(*++p) {
      case 'd': 
      case 'i':
      case 'x':
      case 'X':
      case 'o':
      case 'u':
	ival = va_arg(arp, int);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, ival);
	break;
      case 'f':
      case 'g':
      case 'F':
      case 'G':
	dval = va_arg(arp, double);
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, dval);
	break;
      case 's':
	::strncpy(strtmp, va_arg(arp, char *), sizeof (strtmp) -1);
	strtmp[sizeof (strtmp) -1] = '\0';

	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, strtmp);
	break; 
      case 'c':
#if (__GNUC_MAJOR__ >2) || (__GNUC_MINOR__ >= 96) 
	cval = va_arg(arp, int);
#else
	cval = va_arg(arp, char);
#endif
	if (cdevData::tagC2I(tagName, &tag) == CDEV_SUCCESS) 
	  data.insert(tag, cval);
	break;
      default:
	printf ("The argument type % %c not supported  \n", *p);
	break;
      }
      i = 0;
    }   //if (*p == '%')
    else if ((tagName[i++] = *p) == '=') {
      i--;
      tagName[i] = '\0';
    }
  }
  va_end(arp);  

  return cl->postData (data);
}

int
cmlog_set_severity_threshold (cmlog_client_t client,
			      int threshold)
{
  if (client != 0) {
    cmlogClient* cl = (cmlogClient *)client;
    
    return cl->setSeverityThreshold (threshold);
  }
  return CMLOG_ERROR;
}

int
cmlog_get_severity_threshold (cmlog_client_t client,
			      int *threshold)
{
  if (client != 0) {
    cmlogClient* cl = (cmlogClient *)client;
    
    *threshold = cl->getSeverityThreshold ();
    return CMLOG_SUCCESS;
  }
  *threshold = 0;
  return CMLOG_ERROR;
}

int
cmlog_set_verbosity_threshold (cmlog_client_t client,
			       int threshold)
{
  if (client != 0) {
    cmlogClient* cl = (cmlogClient *)client;
    
    return cl->setVerbosityThreshold (threshold);
  }
  return CMLOG_ERROR;
}

int
cmlog_get_verbosity_threshold (cmlog_client_t client,
			       int *threshold)
{
  if (client != 0) {
    cmlogClient* cl = (cmlogClient *)client;
    
    *threshold = cl->getVerbosityThreshold ();
    return CMLOG_SUCCESS;
  }
  *threshold = 0;
  return CMLOG_ERROR;
}

int
cmlog_set_throttle (cmlog_client_t client,
		    char* tag, int valuerange,
		    int limit, double interval)
{
  if (client != 0) {
    cmlogClient* cl = (cmlogClient *)client;
    
    return cl->setThrottle (tag, valuerange, limit, interval);
  }
  return CMLOG_ERROR;
}

int
cmlog_get_throttle (cmlog_client_t client,
		    char** tag, int* valuerange,
		    int* limit, double* interval)
{
  if (client != 0) {
    cmlogClient* cl = (cmlogClient *)client;
    
    return cl->getThrottle (tag, *valuerange, *limit, *interval);
  }
  return CMLOG_ERROR;
}


  

#ifdef _CODA_DALOG
/**************************************************************************
 *               Implementation for CODA logging facility                 *
 *************************************************************************/
static cmlogClient* daclient = 0;
static char * object_class = "Unknown";
static char * object_process = "Unknown";
static char * object_domain = "Unknown";

#ifdef __vxworks
int cmlog_printf_flag = 1;  /* if flag=1 then issue printf along with message
                               otherwise suppress the printf */
#endif

int 
daLogOpen(char *name,char *type)
{
  char *session;

  if (daclient == 0) {
#ifdef __vxworks
    daclient = cmlogClient::logClient();
#else
    daclient = new cmlogClient (name);
#endif
    if (daclient->connect () != CMLOG_SUCCESS)
      return 1;
  }
  object_class = new char [strlen(type)+1];
  strcpy(object_class,type);

  object_process = new char [strlen(name)+1];
  strcpy(object_process,name);

  if (session = getenv("SESSION")) {
    object_domain = new char [strlen(session)+1];
    strcpy(object_domain,session);
  }

  return (0);
}


static int sevToInt(char *severity)
{
  if (!strcmp(severity,"INFO")) {
    return CDEV_SEVERITY_INFO;
  } else if (!strcmp(severity,"ERROR")) {
    return CDEV_SEVERITY_ERROR;
  } else if (!strcmp(severity,"WARN")) {
    return CDEV_SEVERITY_WARN;
  } else if (!strcmp(severity,"SEVERE")) {
    return CDEV_SEVERITY_SEVERE;
  } else return 100;
  
}

#ifdef __vxworks
void
daLogMsg (char *sev, char *fmt,
	  int a,int b,int c,int d,int e,int f,int g,
	  int h,int i)
{
  int severity;
  char        temp[1000];

  severity = sevToInt (sev);
  sprintf (temp, fmt, a, b, c, d, e, f, g, h, i);

  if (daclient == 0) {
    daclient = cmlogClient::logClient();
    if (daclient->connect () != CMLOG_SUCCESS) {
      printf("daLogMsg: %s\n", temp);
      return;
    }
  }

  daclient->logMsg ("process = %s class = %s status = %s severity = %d facility = %s domain = %s text = %s",
		    object_process,object_class, sev, severity, CMLOG_CODA, object_domain, temp);
  
  if (cmlog_printf_flag)
    printf("daLogMsg: %s\n", temp);
}

#else

void
daLogMsg (char *sev,char *fmt,...)
{
  va_list ap;
  int severity;
  static char temp[1000];
  va_start(ap, fmt);

  severity = sevToInt (sev);
  vsprintf (temp, fmt, ap);

  if (daclient == 0) {
    daclient = new cmlogClient ();
    if (daclient->connect () != CMLOG_SUCCESS) {
      printf("daLogMsg: %s\n", temp);
      return;
    }
  }

  daclient->logMsg ("process = %s class = %s status = %s severity = %d facility = %s domain = %s text = %s",
		    object_process,object_class, sev, severity, CMLOG_CODA, object_domain, temp);

  printf("daLogMsg: %s\n", temp);
}
#endif

void
daLogMsgI (char *sev, char *fmt,
	  int a,int b,int c,int d,int e,int f,int g,
	  int h,int i)
{
  int severity;
  char        temp[1000];

  if (daclient == 0) {
      return;
  }

  severity = sevToInt (sev);
  sprintf (temp, fmt, a, b, c, d, e, f, g, h, i);
  
#ifdef __vxworks
  daclient->logMsgI ("process = %s class = %s status = %s severity = %d facility = %s text = %s",
		    (int) object_process, (int) object_class, (int) sev, severity, (int) CMLOG_CODA, (int) temp, 0, 0, 0, 0);
#else
  daclient->logMsg ("process = %s class = %s status = %s severity = %d facility = %s text = %s",
		    object_process,object_class, sev, severity, CMLOG_CODA, temp);
#endif
}

void
daLogMsg2 (char* name, char *sev, char *fmt,
	  int a,int b,int c,int d,int e,int f,int g,
	  int h,int i)
{
  int severity;
  char        temp[1000];

  if (daclient == 0) {
#ifdef __vxworks
    daclient = cmlogClient::logClient();
#else
    daclient = new cmlogClient (name);
#endif
    if (daclient->connect () != CMLOG_SUCCESS)
      return;
  }

  severity = sevToInt (sev);
  sprintf (temp, fmt, a, b, c, d, e, f, g, h, i);
  
  daclient->logMsg ("process = %s class = %s status = %s severity = %d facility = %s text = %s",
		    object_process,object_class, sev, severity, CMLOG_CODA, temp);
}

#endif
