// We need this to get the status of ACE_NTRACE...

#include "ace/config.h"
#include "ace/Trace.h"

// Turn off tracing for the duration of this file.
#if defined (ACE_NTRACE)
#undef ACE_NTRACE
#endif /* ACE_NTRACE */
#define ACE_NTRACE 1

// This must come first to avoid "order of include" problems...

#if !defined (ACE_HAS_INLINED_OSCALLS) && !defined (ACE_WIN32)
#define ACE_HAS_INLINED_OSCALLS
#include "ace/ACE.h"
#undef ACE_HAS_INLINED_OSCALLS
#else
#include "ace/ACE.h"
#endif /* !ACE_HAS_INLINED_OSCALLS */

#include "ace/Log_Msg.h"

#if !defined (__INLINE__)
#include "ace/Trace.i"
#endif /* __INLINE__ */

class ACE_Trace_State
  // = TITLE
  //    This is a thread-specific Singleton.
  //
  // = DESCRIPTION
  //    Instances of this class are stored in thread-specific storage
  //    (TSS) and used to keep track of the depth of the Tracing and
  //    whether tracing is being invoked recursively from within an
  //    ACE_ERROR or ACE_DEBUG macro (to avoid infinite recursion).
{
public:
  static ACE_Trace_State *instance (void);
  // Singleton interface.

  ACE_Trace_State (void): depth_ (0), trace_active_ (0) {}

  // = Nesting depth increment and decrement.
  int inc (void) { return this->depth_++; }
  int dec (void) { return --this->depth_; }

  // Get/set trace active status.
  int trace_active (void) { return this->trace_active_; }
  void trace_active (int value) { this->trace_active_ = value; }

private:
  int depth_;
  // Depth of the nesting for printing.

  int trace_active_;
  // Are we already within an ACE_Trace constructor call?
};

class ACE_Trace_TSS
  // = TITLE
  //    This class is a "lightweight" implementation of the ACE_TSS
  //    "smart pointer" that avoids all calls to methods in ACE that
  //    may have tracing enabled.  
  //
  // = DESCRIPTION
  //    This class goes here since we don't need to make it visible
  //    externally.
{
public:
  // = Initialization and termination
  ACE_Trace_TSS (void);
  ~ACE_Trace_TSS (void);

  ACE_Trace_State *operator->();
  // Retrieve thread-specific storage via the delegation operator.

  operator ACE_Trace_State *();
  // Retrieve thread-specific storage.

private:
  int once_;
  // "First time in" flag.

  thread_key_t key_;
  // Key for the thread-specific error data.

  mutex_t keylock_;
  // Avoid race conditions during initialization.

  static void cleanup (void *ptr);
  // "Destructor" that deletes internal TYPE * when thread exits.
};

// Static initialization.
int ACE_Trace::nesting_indent_ = ACE_Trace::DEFAULT_INDENT;
int ACE_Trace::enable_tracing_ = ACE_Trace::DEFAULT_TRACING;

#if !(defined (ACE_HAS_THREADS) && defined (ACE_HAS_THREAD_SPECIFIC_STORAGE))
typedef ACE_Trace_State ACE_TRACE_TSS_INSTANCE;
#define ACE_TRACE_TSS_GET(INSTANCE) (INSTANCE)
#else
typedef ACE_Trace_TSS ACE_TRACE_TSS_INSTANCE;
#define ACE_TRACE_TSS_GET(INSTANCE) ((INSTANCE)->operator ACE_Trace_State * ())
#endif /* !(defined (ACE_HAS_THREADS) && defined (ACE_HAS_THREAD_SPECIFIC_STORAGE)) */

ACE_ALLOC_HOOK_DEFINE(ACE_Trace)

void
ACE_Trace::dump (void) const
{
}

ACE_Trace_State *
ACE_Trace_State::instance (void)
{
  // Singleton implementation.
  static ACE_TRACE_TSS_INSTANCE *instance_ = 0;

  if (instance_ == 0)
    instance_ = new ACE_TRACE_TSS_INSTANCE;

  return ACE_TRACE_TSS_GET (instance_);
}

// Enable the tracing facility.

void 
ACE_Trace::start_tracing (void) 
{ 
  ACE_Trace::enable_tracing_ = 1;
}

// Disable the tracing facility. 

void 
ACE_Trace::stop_tracing (void)  
{ 
  ACE_Trace::enable_tracing_ = 0; 
}

// Change the nesting indentation level. 

void 
ACE_Trace::set_nesting_indent (int indent) 
{ 
  ACE_Trace::nesting_indent_ = indent; 
}

// Perform the first part of the trace, which prints out the string N,
// the LINE, and the ACE_FILE as the function is entered.

ACE_Trace::ACE_Trace (const char *n, int line, const char *file)
{
  if (ACE_Trace::enable_tracing_ 
      && ACE_Trace_State::instance ()->trace_active () == 0)
    {
      ACE_Trace_State::instance ()->trace_active (1);
      ACE_DEBUG ((LM_DEBUG, "%*s(%t) calling %s in file `%s' on line %d\n",
		  ACE_Trace::nesting_indent_ * ACE_Trace_State::instance ()->inc (),
		  "", this->name_ = n, file, line));
      ACE_Trace_State::instance ()->trace_active (0);
    }
}

// Perform the second part of the trace, which prints out the NAME as
// the function is exited.

ACE_Trace::~ACE_Trace (void) 
{
  if (ACE_Trace::enable_tracing_ 
      && ACE_Trace_State::instance ()->trace_active () == 0)
    {
      ACE_Trace_State::instance ()->trace_active (1);
      ACE_DEBUG ((LM_DEBUG, "%*s(%t) leaving %s\n", 
		  ACE_Trace::nesting_indent_ * ACE_Trace_State::instance ()->dec (),
		  "", this->name_));
      ACE_Trace_State::instance ()->trace_active (0);
    }
}

#if defined (ACE_HAS_THREADS) && defined (ACE_HAS_THREAD_SPECIFIC_STORAGE)

ACE_Trace_TSS::ACE_Trace_TSS (void): once_ (0), key_ (0) 
{ 
  ACE_OS::mutex_init (&this->keylock_, USYNC_THREAD, 0, 0);
}

ACE_Trace_TSS::~ACE_Trace_TSS (void)
{ 
  ACE_OS::mutex_destroy (&this->keylock_);
}

/* static */
void
ACE_Trace_TSS::cleanup (void *ptr)
{
  // This cast is necessary to invoke the destructor (if necessary).
  ACE_Trace_State *trace_ptr = (ACE_Trace_State *) ptr;
  delete trace_ptr;
}

ACE_Trace_TSS::operator ACE_Trace_State * ()
{
  // Create and initialize thread-specific ts_obj.
  if (this->once_ == 0)
    {
      // Insure that we are serialized!
      ACE_OS::mutex_lock (&this->keylock_);

      // Make sure we only create the key once!
      if (this->once_ == 0)
	{
	  if (ACE_OS::thr_keycreate (&this->key_,
				     &ACE_Trace_TSS::cleanup) != 0)
	    {
	      ACE_OS::mutex_unlock (&this->keylock_);
	      return 0; // Major problems, this should *never* happen!
	    }

	  // This *must* come last to avoid race conditions!
	  this->once_ = 1;
	}

      ACE_OS::mutex_unlock (&this->keylock_);
    }

  ACE_Trace_State *ts_obj = 0;

  // Get the ts_obj from thread-specific storage.  Note that no locks
  // are required here...
  if (ACE_OS::thr_getspecific (this->key_, (void **) &ts_obj) == -1)
    return 0; // This should not happen!

  // Check to see if this is the first time in for this thread.
  if (ts_obj == 0)
    {
      // Allocate memory off the heap and store it in a pointer in
      // thread-specific storage (on the stack...).

      ts_obj = new ACE_Trace_State;

      if (ts_obj == 0)
	return 0;

      // Store the dynamically allocated pointer in thread-specific
      // storage.
      if (ACE_OS::thr_setspecific (this->key_, (void *) ts_obj) != 0)
	return 0; // Major problems, this should *never* happen!
    }

  return ts_obj;
}

ACE_Trace_State *
ACE_Trace_TSS::operator-> ()
{
  return this->operator ACE_Trace_State *();
}
#endif /* !(defined (ACE_HAS_THREADS) && defined (ACE_HAS_THREAD_SPECIFIC_STORAGE)) */

