#include "ace/Thread_Manager.h"

#if !defined (__INLINE__)
#include "ace/Thread_Manager.i"
#endif /* __INLINE__ */

ACE_ALLOC_HOOK_DEFINE(ACE_Thread_Control)
ACE_ALLOC_HOOK_DEFINE(ACE_Thread_Manager)

#if defined (ACE_HAS_THREADS)

void
ACE_Thread_Manager::dump (void) const
{
  ACE_TRACE ("ACE_Thread_Manager::dump");
}

ACE_Thread_Descriptor::ACE_Thread_Descriptor (void)
  : thr_id_ (ACE_OS::NULL_thread),
    grp_id_ (0),
    thr_state_ (ACE_Thread_Descriptor::IDLE)
{
  ACE_TRACE ("ACE_Thread_Descriptor::ACE_Thread_Descriptor");
}

// Return the thread descriptor (indexed by thread_t).

int 
ACE_Thread_Manager::descriptor (thread_t thr_id,
				ACE_Thread_Descriptor &descriptor)
{
  ACE_TRACE ("ACE_Thread_Descriptor::descriptor");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  for (size_t i = 0; i < this->current_count_; i++)
    if (this->thr_table_[i].thr_id_ == thr_id)
      {
	descriptor = this->thr_table_[i];
	return 0;
      }

  return -1;
}

// Return the thread descriptor (indexed by hthread_t).

int 
ACE_Thread_Manager::descriptor (hthread_t thr_handle, 
				ACE_Thread_Descriptor &descriptor)
{
  ACE_TRACE ("ACE_Thread_Descriptor::descriptor");  
  ACE_MT (ACE_Mutex_Guard m (this->lock_));


  for (size_t i = 0; i < this->current_count_; i++)
    if (this->thr_table_[i].thr_handle_ == thr_handle)
      {
	descriptor = this->thr_table_[i];
	return 0;
      }

  return -1;
}

int
ACE_Thread_Manager::resize (size_t size)
{
  ACE_TRACE ("ACE_Thread_Manager::resize");
  ACE_Thread_Descriptor *temp = new ACE_Thread_Descriptor[size];
      
  if (temp == 0)
    {
      errno = ENOMEM;
      return -1;
    }

  for (size_t i = 0; i < this->max_table_size_; i++)
    temp[i] = this->thr_table_[i]; // Structure assignment.

  this->max_table_size_ = size;

  delete [] this->thr_table_;

  this->thr_table_ = temp;
  return 0;
}

// Create and initialize the table to keep track of the thread pool.

int
ACE_Thread_Manager::open (size_t size)
{
  ACE_TRACE ("ACE_Thread_Manager::open");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  if (this->max_table_size_ < size)
    this->resize (size);
  return 0;
}

// Initialize the synchronization variables.

ACE_Thread_Manager::ACE_Thread_Manager (size_t size)
  : zero_cond_ (lock_),
    max_table_size_ (0), 
    thr_table_ (0),
    current_count_ (0),
    grp_id_ (1)
{
  ACE_TRACE ("ACE_Thread_Manager::ACE_Thread_Manager");
  if (this->open (size) == -1)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_Thread_Manager"));
}

// Close up and release all resources.

int
ACE_Thread_Manager::close (void)
{
  ACE_TRACE ("ACE_Thread_Manager::close");
  if (this->thr_table_ != 0)
    {
      delete [] this->thr_table_;
      this->thr_table_ = 0;
      this->max_table_size_ = 0;
      this->current_count_ = 0;
    }
  return 0;
}

ACE_Thread_Manager::~ACE_Thread_Manager (void)
{
  ACE_TRACE ("ACE_Thread_Manager::~ACE_Thread_Manager");
  this->close ();
}

// Call the appropriate OS routine to spawn a thread.  Should *not* be
// called with the lock_ held...

int 
ACE_Thread_Manager::spawn_i (ACE_THR_FUNC func, 
			     void *args, 
			     long flags, 
			     thread_t *t_id, 
			     hthread_t *t_handle,
			     u_int priority,
			     void *stack, 
			     size_t stack_size)
{
  ACE_TRACE ("ACE_Thread_Manager::spawn_i");
  thread_t thr_id;
  hthread_t thr_handle = 0;

  if (t_id == 0)
    t_id = &thr_id;

  if (t_handle == 0)
    t_handle = &thr_handle;
  
  int result = ACE_Thread::spawn (func, args, flags, 
				  t_id, t_handle, priority,
				  stack, stack_size);
  if (result != 0)
    {
      errno = result;
      return -1;
    }
  else
    return this->append_thr (*t_id, *t_handle, ACE_Thread_Descriptor::SPAWNED);
}

// Create a new thread running FUNC.  *Must* be called with the lock_
// held...

int 
ACE_Thread_Manager::spawn (ACE_THR_FUNC func, 
			   void *args, 
			   long flags, 
			   thread_t *t_id, 
			   hthread_t *t_handle,
			   u_int priority,
			   void *stack, 
			   size_t stack_size)
{
  ACE_TRACE ("ACE_Thread_Manager::spawn");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  if (this->spawn_i (func, args, flags, t_id, t_handle, 
		     priority, stack, stack_size) == -1)
    return -1;
  else
    return this->grp_id_++; // Increment the group id.
}

// Create N new threads running FUNC.

int 
ACE_Thread_Manager::spawn_n (int n, 
			     ACE_THR_FUNC func, 
			     void *args, 
			     long flags,
			     u_int priority)
{
  ACE_TRACE ("ACE_Thread_Manager::spawn_n");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  for (int i = 0; i < n; i++)
    {
      // @@ What should happen if this fails?! e.g., should we try to
      // cancel the other threads that we've already spawned or what?
      if (this->spawn_i (func, args, flags, 0, 0, priority) == -1)
	return -1;
    }

  return this->grp_id_++;   // Increment the group id.
}

// Append a thread into the pool (does not check for duplicates).
// Must be called with locks held.

int
ACE_Thread_Manager::append_thr (thread_t t_id, 
				hthread_t t_handle,
				ACE_Thread_Descriptor::Thread_State thr_state)
{
  ACE_TRACE ("ACE_Thread_Manager::append_thr");
  // Try to resize the array to twice its existing size if we run out
  // of space...
  if (this->current_count_ >= this->max_table_size_ 
      && this->resize (this->max_table_size_ * 2) == -1)
    return -1;
  else
    {
      ACE_Thread_Descriptor &thr_desc = this->thr_table_[this->current_count_];

      thr_desc.thr_id_ = t_id;
      thr_desc.thr_handle_ = t_handle;
      thr_desc.grp_id_ = this->grp_id_;
      thr_desc.thr_state_ = thr_state;

      this->current_count_++;
      return 0;
    }
}

// Insert a thread into the pool (checks for duplicates and doesn't
// allow them to be inserted twice).

int
ACE_Thread_Manager::insert_thr (thread_t t_id, hthread_t t_handle)
{
  ACE_TRACE ("ACE_Thread_Manager::insert_thr");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  // Check for duplicates and bail out if they're already
  // registered...
  if (this->find (t_id) != -1)
    return -1;

  return this->append_thr (t_id, t_handle, ACE_Thread_Descriptor::SPAWNED);
}

// Remove a thread from the pool.  Must be called with locks held.

void
ACE_Thread_Manager::remove_thr (int i)
{
  ACE_TRACE ("ACE_Thread_Manager::remove_thr");
  if (this->current_count_ > 1)
    // Structure assignment.
    this->thr_table_[i] = this->thr_table_[this->current_count_ - 1];

  this->current_count_--;

  // Tell waiters when there are no more threads left in the pool.
  if (this->current_count_ == 0)
    this->zero_cond_.signal ();
}

int
ACE_Thread_Manager::resume_thr (int i)
{
  ACE_TRACE ("ACE_Thread_Manager::resume_thr");
  int result = ACE_Thread::resume (this->thr_table_[i].thr_handle_);

  if (result != 0)
    {
      this->remove_thr (i);
      result = errno;
      return -1;
    }
  else
    {
      this->thr_table_[i].thr_state_ = ACE_Thread_Descriptor::RUNNING;
      return 0;
    }
}

int
ACE_Thread_Manager::suspend_thr (int i)
{
  ACE_TRACE ("ACE_Thread_Manager::suspend_thr");
  int result = ACE_Thread::suspend (this->thr_table_[i].thr_handle_);

  if (result != 0)
    {
      this->remove_thr (i);
      errno = result;
      return -1;
    }
  else
    {
      this->thr_table_[i].thr_state_ = ACE_Thread_Descriptor::SUSPENDED;
      return 0;
    }
}

int
ACE_Thread_Manager::kill_thr (int i, int signum)
{
  ACE_TRACE ("ACE_Thread_Manager::kill_thr");
  int result = ACE_Thread::kill (this->thr_table_[i].thr_id_, signum);

  if (result != 0)
    {
      this->remove_thr (i);
      errno = result;
      return -1;
    }
  else
    return 0;
}

// Locate the index in the table associated with <t_id>.  Must be
// called with the lock held.

int 
ACE_Thread_Manager::find (thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::find");
  size_t i;

  for (i = 0; i < this->current_count_; i++)
    if (ACE_OS::thr_equal (t_id, (this->thr_table_[i].thr_id_)))
      return i;

  return -1;    
}

// Suspend a single thread.

int 
ACE_Thread_Manager::suspend (thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::suspend");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  int i = this->find (t_id);
  this->thr_table_[i].thr_state_ = ACE_Thread_Descriptor::SUSPENDED;
  return ACE_Thread::suspend (this->thr_table_[i].thr_handle_);
}

// Resume a single thread.

int 
ACE_Thread_Manager::resume (thread_t t_id)
{
  ACE_TRACE ("ACE_Thread_Manager::resume");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  int i = this->find (t_id);
  this->thr_table_[i].thr_state_ = ACE_Thread_Descriptor::RUNNING;
  return ACE_Thread::resume (this->thr_table_[i].thr_handle_);
}

// Kill a single thread.
int 
ACE_Thread_Manager::kill (thread_t t_id, int signum)
{
  ACE_TRACE ("ACE_Thread_Manager::kill");
  return ACE_Thread::kill (t_id, signum);
}

// Get group ids for a particular thread id.

int 
ACE_Thread_Manager::get_grp (thread_t t_id, int &grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::get_grp");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  int i = this->find (t_id);
  grp_id = this->thr_table_[i].grp_id_;
  return 0;
}

// Set group ids for a particular thread id.

int 
ACE_Thread_Manager::set_grp (thread_t t_id, int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::set_grp");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  int i = this->find (t_id);
  this->thr_table_[i].grp_id_ = grp_id;
  return 0;
}

// Suspend a group of threads.

int
ACE_Thread_Manager::apply_grp (int grp_id, 
			       THR_FUNC func,
			       int arg)
{
  ACE_TRACE ("ACE_Thread_Manager::apply_grp");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  int result = 0;

  for (size_t i = 0; i < this->current_count_; i++)
    if (this->thr_table_[i].grp_id_ == grp_id
	&& (this->*func)(i, arg) == -1)
      result = -1;

  return result;
}

int 
ACE_Thread_Manager::suspend_grp (int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::suspend_grp");
  return this->apply_grp (grp_id, 
			  THR_FUNC (&ACE_Thread_Manager::suspend_thr));
}

// Resume a group of threads.

int 
ACE_Thread_Manager::resume_grp (int grp_id)
{
  ACE_TRACE ("ACE_Thread_Manager::resume_grp");
  return this->apply_grp (grp_id, 
			  THR_FUNC (&ACE_Thread_Manager::resume_thr));
}

// Kill a group of threads.

int 
ACE_Thread_Manager::kill_grp (int grp_id, int signum)
{
  ACE_TRACE ("ACE_Thread_Manager::kill_grp");
  return this->apply_grp (grp_id, 
			  THR_FUNC (&ACE_Thread_Manager::kill_thr), signum);
}

int
ACE_Thread_Manager::apply_all (THR_FUNC func, int arg)
{
  ACE_TRACE ("ACE_Thread_Manager::apply_all");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  int result = 0;

  for (size_t i = 0; i < this->current_count_; i++)
    if ((this->*func)(i, arg) == -1)
      result = -1;

  return result;
}

// Resume all threads that are suspended.

int 
ACE_Thread_Manager::resume_all (void)
{
  ACE_TRACE ("ACE_Thread_Manager::resume_all");
  return this->apply_all (THR_FUNC (&ACE_Thread_Manager::resume_thr));
}

int 
ACE_Thread_Manager::suspend_all (void)
{
  ACE_TRACE ("ACE_Thread_Manager::suspend_all");
  return this->apply_all (THR_FUNC (&ACE_Thread_Manager::suspend_thr));
}

int 
ACE_Thread_Manager::kill_all (int sig)
{
  ACE_TRACE ("ACE_Thread_Manager::kill_all");
  return this->apply_all (&ACE_Thread_Manager::kill_thr, sig);
}

// Must be called when thread goes out of scope to clean up its table
// slot.

void *
ACE_Thread_Manager::exit (void *status)
{
  ACE_TRACE ("ACE_Thread_Manager::exit");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  /* Locate thread id */
  int i = this->find (ACE_Thread::self ());

  if (i == -1)
    {
      errno = ENOENT; 
      return (void *) -1;
    }
  else
    {
      this->remove_thr (i);

      ACE_MT (m.release ()); /* Note, destructor is never called... */
      ACE_Thread::exit (status);
      /* NOTREACHED */
    }
  return 0;
}

// Wait for all the threads to exit.

int
ACE_Thread_Manager::wait (ACE_Time_Value *timeout)
{
  ACE_TRACE ("ACE_Thread_Manager::wait");
  ACE_MT (ACE_Mutex_Guard m (this->lock_));

  while (this->current_count_ > 0)
    if (this->zero_cond_.wait (timeout) == -1)
      return -1;

  return 0;
}

void
ACE_Thread_Control::dump (void) const
{
  ACE_TRACE ("ACE_Thread_Control::dump");
}

// Initialize the thread controller.

ACE_Thread_Control::ACE_Thread_Control (ACE_Thread_Manager *t, 
					int insert)
  : tm_ (t)
{
  ACE_TRACE ("ACE_Thread_Control::ACE_Thread_Control");
  if (this->tm_ != 0 && insert)
    {
      hthread_t t_id;
      ACE_Thread::self (t_id);
      this->tm_->insert_thr (ACE_Thread::self (), t_id);
    }
}

// Automatically kill thread on exit.

ACE_Thread_Control::~ACE_Thread_Control (void)
{
  ACE_TRACE ("ACE_Thread_Control::~ACE_Thread_Control");
  this->exit (this->status_);
}

// Exit from thread (but clean up first).

void *
ACE_Thread_Control::exit (void *exit_status)
{
  ACE_TRACE ("ACE_Thread_Control::exit");
  if (this->tm_ != 0)
    return this->tm_->exit (exit_status);
  else
    {
      ACE_Thread::exit (exit_status);
      return 0;
    }
}

#endif /* !defined (ACE_HAS_THREADS) */
