#include "ace/SOCK.h"
#include "ace/Log_Msg.h"

ACE_ALLOC_HOOK_DEFINE(ACE_SOCK)

#if defined (ACE_WIN32)
// Static used to initialize WinSock.
ACE_SOCK ACE_SOCK::dummy_;

int ACE_SOCK::win32_initialized_ = 0;

int
ACE_SOCK::win32_init (int version_high, int version_low)
{
  WORD version_requested;
  WSADATA wsa_data;
  int err;

  version_requested = MAKEWORD (version_high, version_low);

  err = ::WSAStartup (version_requested, &wsa_data);

  if (err != 0)
    ACE_ERROR_RETURN ((LM_ERROR, 
		       "WSAStartup failed, WSAGetLastError returned %u.\n",
		       err), -1);

  ACE_SOCK::win32_initialized_ = 1;
  return 0;
}

int
ACE_SOCK::win32_fini (void)
{
  if (::WSACleanup () != 0)
    ACE_ERROR_RETURN ((LM_ERROR, "WSACleanup failed, WSAGetLastError returned %u.\n", 
		       ::WSAGetLastError ()), -1);
  ACE_SOCK::win32_initialized_ = 0;
  return 0;
}

BOOL WINAPI
DllMain (HINSTANCE hinstDLL, // DLL module handle
	 DWORD fdwReason, // Reason called
	 LPVOID lpvReserved) // Reserved
{
  switch (fdwReason)
    {
    case DLL_PROCESS_ATTACH:
      if (ACE_SOCK::win32_init (ACE_WSOCK_VERSION) != 0)
	return FALSE;
      break;

    case DLL_PROCESS_DETACH:
      if (ACE_SOCK::win32_fini () != 0)
        return FALSE;
      break;

    case DLL_THREAD_ATTACH:
    case DLL_THREAD_DETACH:
      break;

    default:
      ACE_ERROR_RETURN ((LM_ERROR, 
			 "Sock.DLL DllMain called with unknown fdwReason = %u\n.", 
			 fdwReason), FALSE);
      /* NOTREACHED */
    }

  return TRUE;

  // Keep the compiler from complaining...
  UNREFERENCED_PARAMETER (hinstDLL);
  UNREFERENCED_PARAMETER (lpvReserved);
}
#endif /* ACE_WIN32 */

void
ACE_SOCK::dump (void) const
{
  ACE_TRACE ("ACE_SOCK::dump");
}

ACE_SOCK::ACE_SOCK (void)
{
  ACE_TRACE ("ACE_SOCK::ACE_SOCK");

#if defined (ACE_WIN32)
  // Make sure that we've initialized the WinSock library before using
  // sockets!
  if (ACE_SOCK::win32_initialized_ == 0)
    ACE_SOCK::win32_init (ACE_WSOCK_VERSION);
#endif /* ACE_WIN32 */
}

// Returns information about the remote peer endpoint (if there is
// one).

int
ACE_SOCK::get_remote_addr (ACE_Addr &sa) const
{
  ACE_TRACE ("ACE_SOCK::get_remote_addr");
  int len = sa.get_size ();
  sockaddr *addr = (sockaddr *) sa.get_addr ();

  if (ACE_OS::getpeername (this->get_handle (), addr, &len) == -1)
    return -1;
  
  sa.set_size (len);
  return 0;
}

int
ACE_SOCK::get_local_addr (ACE_Addr &sa) const
{
  ACE_TRACE ("ACE_SOCK::get_local_addr");
  int len = sa.get_size ();
  sockaddr *addr = (sockaddr *) sa.get_addr ();

  if (ACE_OS::getsockname (this->get_handle (), addr, &len) == -1)
    return -1;

  sa.set_size (len);
  return 0;
}

int
ACE_SOCK::open (int type, 
		int protocol_family, 
		int protocol)
{
  ACE_TRACE ("ACE_SOCK::open");
  this->set_handle (ACE_OS::socket (protocol_family, type, protocol));
  return this->get_handle () == ACE_INVALID_HANDLE ? -1 : 0;
}

// Close down a ACE_SOCK.

int
ACE_SOCK::close (void)
{
  ACE_TRACE ("ACE_SOCK::close");
  int result = 0;

  if (this->get_handle () != ACE_INVALID_HANDLE)
    {
      result = ACE_OS::close (this->get_handle ());
      this->set_handle (ACE_INVALID_HANDLE);
    }
  return result;
}

// General purpose constructor for performing server ACE_SOCK
// creation.

ACE_SOCK::ACE_SOCK (int type, 
		    int protocol_family, 
		    int protocol)
{
  ACE_TRACE ("ACE_SOCK::ACE_SOCK");
  if (this->open (type, protocol_family, protocol) == -1)
    ACE_ERROR ((LM_ERROR, "%p\n", "ACE_SOCK::ACE_SOCK"));
}
