//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// CEBAF Data Acquisition Group, 12000 Jefferson Ave., Newport News, VA 23606
//       coda@cebaf.gov  Tel: (804) 249-7030     Fax: (804) 249-5800
//-----------------------------------------------------------------------------
//
// Description:
//      Implementation cmlogBrowser Class
//
// Author:  
//      Jie Chen
//      CEBAF Data Acquisition Group
//
// Revision History:
//   $Log: cmlogAdmin.cc,v $
//   Revision 1.3  2001/07/25 14:24:39  chen
//   64 BIT Initial Port
//
//   Revision 1.2  2000/06/20 19:37:10  chen
//   port to CC 5.0 and gcc 2.95.2
//
//   Revision 1.1.1.1  1999/09/07 15:29:02  chen
//   CMLOG version 2.0
//
// Revision 1.3  1998/10/07  17:02:03  chen
// Fix a compiler (egcs) warning about cmlog_cdevMessage& = (cmlog_cdevMessage)
//
// Revision 1.2  1998/01/21  16:03:46  chen
// take line input correctly
//
// Revision 1.1  1997/08/01  15:16:58  bickley
// Added cmlog to application development system.
//
//
//
#include <cmlogUtil.h>
#include <cmlogNetUtil.h>
#include "cmlogAdmin.h"

#define CMLOG_GETCBK_TABLE_SIZE 997
#define CMLOG_EVID_MAX          1093939

//=========================================================================
//       Implementation of cmlogAdmin
//=========================================================================
cmlogAdmin::cmlogAdmin (void)
:connected_ (0), lock_ (0), udp_fd_ (-1), tcp_fd_ (-1), 
  serverPort_ (0), serverHost_ (0), qid_ (93), evid_ (1024),
 getCbkTable_ (CMLOG_GETCBK_TABLE_SIZE, cmlogStrHashFunc)
{
#ifdef _TRACE_OBJECTS
  printf ("    Create cmlogAdmin Class Object\n");
#endif
  // setup all tags
  cmlogUtil::setTags ();
}

cmlogAdmin::~cmlogAdmin (void)
{
#ifdef _TRACE_OBJECTS
  printf ("    Delete cmlogAdmin Class Object\n");
#endif
  // delete everything from get callback list
  cmlogCallback*     cbk = 0;
  cmlogStrHashIterator site (getCbkTable_);
  for (site.init (); !site; ++site) {
    cbk = (cmlogCallback *) site ();
    delete cbk;
  }
  getCbkTable_.deleteAllValues ();  

  // delete server hostname
  if (serverHost_)
    delete []serverHost_;

  if (connected_) 
    disconnect ();
}

int
cmlogAdmin::connect (int num)
{
  if (!connected_) {
    unsigned short brd_port = cmlogNetUtil::findServerUdpPort ();

    if (cmlogNetUtil::udpOpenLocal (udp_fd_) != 0)
      return CMLOG_ERROR;
    
    if (cmlogNetUtil::udpTalkToServer (udp_fd_, brd_port, num, 
				       CMLOG_ADMINLKSVC,
				       serverPort_, serverHost_) != 0)
      return CMLOG_ERROR;

    // connect to TCP port
    // if code reach here, udp_fd should be closed, reset it to -1
    udp_fd_ = -1;

    if (cmlogNetUtil::tcpConnect (serverPort_, serverHost_, tcp_fd_) != 0)
      return CMLOG_ERROR;

    connected_ = 1;

  }
  return CMLOG_SUCCESS;
}

int
cmlogAdmin::connected (void) const
{
  return connected_;
}

int
cmlogAdmin::getFd (void) const
{
  return tcp_fd_;
}

int
cmlogAdmin::disconnect (void)
{
  if (connected_) {
    close (tcp_fd_);
    tcp_fd_ = -1;
    connected_ = 0;
  }
  return CMLOG_SUCCESS;
}

int 
cmlogAdmin::getCallback (char* attribute, 
			 cmlogBrCallback callback, void* arg)
{
  int n = 0;
  int st = 0;

  if (!connected_ || tcp_fd_ == -1)
    return CMLOG_BADIO;

  // create a new cmlogCallback
  cmlogCallback* cbk = new cmlogCallback (callback, arg);
    
  // set request id 
  if (qid_ > 65535)
    qid_ = 93;

  if (evid_ > CMLOG_EVID_MAX)
    evid_ = 1024;

  cmlog_cdevMessage msg (0, qid_++, 0, 0, evid_++, CMLOG_GET_ATTR, 
			 0, 0, attribute, 0, 0, 0);
  cmlogMsg    qmsg    (msg);
    
  cmlogPacket qpacket;
  if (qpacket.overflow (qmsg))
    return CMLOG_OVERFLOW;

  qpacket.insert (qmsg);
      
  // send over to the server
  n = tcp_fd_ << qpacket;
  if (n > 0) {
    // insert this cbk into get callback table
    cbk->setEvent (evid_ - 1);
    getCbkTable_.add (attribute, (void *)cbk);
    return CMLOG_SUCCESS;
  }
      
  return CMLOG_IOERROR;
}


int
cmlogAdmin::pendIO (void)
{
  if (lock_)
    return CMLOG_INVALID;

  if (tcp_fd_ == -1)
    return CMLOG_BADIO;

  int nfound = 0;
  fd_set             readfd;  

  FD_ZERO (&readfd);
  FD_SET  (tcp_fd_, &readfd);

  do {
#ifndef __hpux
    nfound = select (tcp_fd_ + 1, &readfd, 0, 0, 0);
#else

#ifndef __hpux10
    nfound = select (tcp_fd_ + 1, (int *)&readfd, (int *)0, (int *)0,
		     (struct timeval *)0);
#else
    nfound = select (tcp_fd_ + 1, &readfd, 0, 0, 0);
#endif

#endif
  }while (nfound == -1 && errno == EINTR);

  if (nfound == -1) { // something wrong with tcp_fd_
    fprintf (stderr, "Bad tcp fd %d\n", tcp_fd_);
    tcp_fd_ = -1;
    return CMLOG_BADIO;
  }
  else if (nfound == 0) // timeout
    return CMLOG_TIMEOUT;
  else {
    if (FD_ISSET (tcp_fd_, &readfd))
      if (handle_input (tcp_fd_) < 0) { // bad io
	close (tcp_fd_);
	tcp_fd_ = -1;
	return CMLOG_IOERROR;
      }
  }
  return CMLOG_SUCCESS;

}

int
cmlogAdmin::pendIO (double seconds)
{
  if (lock_)
    return CMLOG_INVALID;

  if (tcp_fd_ == -1)
    return CMLOG_BADIO;

  int nfound = 0;
  fd_set             readfd;  
  struct timeval     tv;

  if (seconds <= 0.000001) {
    tv.tv_sec = 0;
    tv.tv_usec = 0;
  }
  else {
    tv.tv_sec = (int)seconds;
    tv.tv_usec = (int)((seconds - tv.tv_sec)*1000000);
  }
    
  FD_ZERO (&readfd);
  FD_SET  (tcp_fd_, &readfd);

  do {
#ifndef __hpux
    nfound = select (tcp_fd_ + 1, &readfd, 0, 0, &tv);
#else

#ifndef __hpux10
    nfound = select (tcp_fd_ + 1, (int *)&readfd, (int *)0, (int *)0,
		     &tv);
#else
    nfound = select (tcp_fd_ + 1, &readfd, 0, 0, &tv);
#endif

#endif
  }while (nfound == -1 && errno == EINTR);

  if (nfound == -1) { // something wrong with tcp_fd_ 
    fprintf (stderr, "Bad tcp fd %d\n", tcp_fd_);
    tcp_fd_ = -1;
    return CMLOG_BADIO;
  }
  else if (nfound == 0) // timeout
    return CMLOG_TIMEOUT;
  else {
    if (FD_ISSET (tcp_fd_, &readfd))
      if (handle_input (tcp_fd_) < 0) { // bad io
	handle_close (tcp_fd_);
	return CMLOG_IOERROR;
      }
  }
  return CMLOG_SUCCESS;
}

int
cmlogAdmin::handle_input (int fd)
{
  int n, status = 0;
  cmlogPacket packet;
  
  // lock this event processing part to prevent recursive calls
  cmlogAdminLocker locker (this);

  n = tcp_fd_ >> packet;
  if (n <= 0) {
    fprintf (stderr, "Server is dead\n");
    status = -1;
  }
  else {
    printf ("Received %d bytes from server\n", n);
    status = processData (packet);
  }

  return status;
}

int
cmlogAdmin::processData (cmlogPacket& packet)
{
  char*       attribute = 0;
  cdevData*   data = 0;
  int         errcode = CMLOG_ERROR;
  unsigned    evid;
  int         st = 0;
  long        opcode = 0;

  // remember to free all msgs and msgs[] somewhere
  cmlogMsg** msgs = packet.messages ();

  // look at the first message which tells status of the query
  cmlog_cdevMessage& idata = (*msgs[0]);
  attribute = idata.getMessage ();
  opcode = msgs[0]->type ();

  if (opcode != CMLOG_GET_ATTR)
    errcode = CMLOG_ERROR;
  
  // get data item inside the message
  data = idata.getData ();
  if (data) {
    if (data->get (cmlogUtil::CMLOG_RESULT_TAG, &errcode) 
	!= CDEV_SUCCESS)
      errcode = CMLOG_ERROR;
  }
 
  evid = idata.getForeignDataIndex ();
  getCbkFromServer (errcode, attribute, evid, packet);

  // empty the packet
  packet.empty (); 
  
  // free memory for msgs
  for (int i = 0; i < packet.numberOfData (); i++)
    delete msgs[i];
  delete []msgs;

  return 0;
}

int
cmlogAdmin::handle_close (int)
{
  close (tcp_fd_);
  tcp_fd_ = -1;
  return 0;
}

int
cmlogAdmin::getCbkFromServer (int errcode, char* attribute,
			      unsigned evid,
			      cmlogPacket& packet)
{
  cmlogCallback* cbk = 0;
  cmlogSlist& list = getCbkTable_.bucketRef(attribute);
  cmlogSlistIterator ite (list);
  int                found = 0;

  for (ite.init(); !ite; ++ite) {
    cbk = (cmlogCallback *) ite ();
    if (cbk->getEvent() == evid) {
      found = 1;
      break;
    }
  }

  if (found) {
    cmlogBrCallback func = cbk->callbackFunction ();
    void *arg = cbk->userarg ();

    (*func)(errcode, arg, &packet);
    
    // remove get callback
    getCbkTable_.remove (attribute, (void *)cbk);
    delete cbk;      
  }
  else {
#ifdef _CMLOG_DEBUG
    printf ("Cannot find this get callback: %s\n", attribute);
#endif
  }
  return 0;
}

int
cmlogAdmin::shutdownServer (void)
{
  cdevData unused;
  cmlog_cdevMessage msg (0, 0, 0, 0, 0, CMLOG_EXIT,
			 0, 0, 0, &unused, 0, 0);
  cmlogMsg    qmsg    (msg);
    
  cmlogPacket qpacket;
  if (qpacket.overflow (qmsg))
    return CMLOG_OVERFLOW;

  qpacket.insert (qmsg);
      
  // send over to the server
  int n = tcp_fd_ << qpacket;
  printf ("Admin sent shutdown %d bytes to the server\n", n);

  if (n > 0)
    return CMLOG_SUCCESS;  
  return CMLOG_ERROR;
}

void
cmlogAdmin::lock (void)
{
  lock_ = 1;
}

void
cmlogAdmin::unlock (void)
{
  lock_ = 0;
}

//=======================================================================
//       Implementation of cmlogAdminLocker class
//=======================================================================
cmlogAdminLocker::cmlogAdminLocker (cmlogAdmin* b)
:b_ (b)
{
  b_->lock ();
}

cmlogAdminLocker::~cmlogAdminLocker (void)
{
  b_->unlock ();
}


// cmlogAdmin utility program
static void getAttrCallback (int status, void* arg, cmlogPacket* data)
{
  cdevData* res = 0;
  char* attribute;
  int i;

  if (status == CMLOG_SUCCESS) {
    if (data) {
      cmlogMsg** msgs = data->messages ();
      for (i = 0; i < data->numberOfData (); i++) {
	cmlog_cdevMessage& idata = (*msgs[i]);
	res = idata.getData ();
	attribute = idata.getMessage ();
	printf ("cmlog sever %s has value %s\n", attribute, (char *)(*res));
      }
      // free memory
      for (i = 0; i < data->numberOfData (); i++) 
	delete msgs[i];
      delete []msgs;
    }
    else
      printf ("NO value returned from the server\n");
  }
  else
    printf ("Error occured in this process\n");
}

main (int argc, char** argv)
{
  cmlogAdmin admin;
  char       line[128];
  char       command[80];
  char       attr[80];
  int        status;

  if (admin.connect () == CMLOG_SUCCESS) {
    printf ("Connected to the cmlogServer\n");

    printf ("Type Help to find out all available commands\n");
    while (1) {
      printf ("cmlogAdmin >");
      fgets (line, sizeof (line), stdin);

      status = sscanf (line, "%s %s", command, attr);
      if (status >= 1) {
	if (::strcasecmp (command, "help") == 0) {
	  printf ("Commands are: \n");
	  printf ("              get (attribute) \n");
	  printf ("              shutdown        \n");
	  printf ("              quit            \n");
	  printf ("              help            \n");
	}
	else if (::strcasecmp (command, "shutdown") == 0) {
	  if (admin.shutdownServer () == CMLOG_SUCCESS)
	    printf ("cmlogServer has been shutdown\n");
	  else
	    printf ("Cannot send out shutdown request\n");
	}
	else if (::strcasecmp (command, "get") == 0) {
	  if (status == 1) {
	    printf ("Enter server attribute name\n");
	    scanf ("%s", attr);
	  }
	  if (admin.getCallback (attr, getAttrCallback, (void *)0) != CMLOG_SUCCESS)
	    printf ("Cannot send out get attribute %s request\n");
	  else {
	    if (admin.pendIO (4.0) != CMLOG_SUCCESS) 
	      printf ("Network request for attribute %s error\n");
	  }
	}
	else if (::strcasecmp (command, "quit") == 0)
	  break;
      }
    }
  }
  else
    printf ("Cannot connect to cmlog server, try use setenv CMLOG_PORT to a right value\n");
  
}
