/******************************************************************************
*
*  envGetRemote.c  -  Retrieve Environment Variable from Boot host and 
*                     set locally.
*
*  Author: David Abbott 
*          Jefferson Lab Data Acquisition Group
*          May 2001
*
*  Revision  1.0 - Initial Revision
*
*  Inputs:       envName  -   Environment variable name (required)
*                remHost  -   Remote host to query (default - VxWorks boot host)
*                remUser  -   Remote user (default - VxWorks boot user)
*
*  Returns:    OK or ERROR if env variable not set
*/

#include <vxWorks.h>
#include <stdio.h>
#include <string.h>
#include <bootLib.h>
#include <ioLib.h>
#include <remLib.h>
#include <envLib.h>
#include <errnoLib.h>

/* Max Environment variable length (bytes) */ 
#define EGR_MAX_ENV_LENGTH     1024

/* Default port for cmd deamon on remote host (/etc/services) */
#define EGR_HOST_DEFAULT_PORT   514

/* External declarations */
extern void printErrno(int);
extern BOOT_PARAMS sysBootParams;

STATUS
envGetRemote( char *envName, char *remHost, char *remUser)
{

  int rfd, stat;
  int envlen, nbytes = 0;

  char rhost[BOOT_HOST_LEN], ruser[BOOT_USR_LEN];
  char rstring[128];
  char res[EGR_MAX_ENV_LENGTH];
  char envS[EGR_MAX_ENV_LENGTH];

  /* Get remote host info */
  if(remHost == NULL) {
    strcpy(rhost,sysBootParams.hostName);
  }else{
    strcpy(rhost,remHost);
  }

  /* Get remote user info */
  if(remUser == NULL) {
    strcpy(ruser,sysBootParams.usr);
  }else{
    strcpy(ruser,remUser);
  }

  /* Construct the remote command */
  sprintf(rstring,"echo $%s",envName);
  envlen = strlen(envName);
  /* Set local user to the same as remote user*/
  iam(ruser,NULL);

/*  printf("host = %s  user = %s     cmd = %s\n",rhost,ruser,rstring); */

  /* Issue the remote command */
  rfd = rcmd(rhost,EGR_HOST_DEFAULT_PORT,ruser,ruser,rstring,0);

  if(rfd > 0) {  /* Read the result */

    nbytes = read(rfd,res,EGR_MAX_ENV_LENGTH);
    /* Make sure result string is not too long */
    if(nbytes >= (EGR_MAX_ENV_LENGTH-envlen)) {
      printf("envGetRemote: ERROR: env result string too long\n");
      close(rfd);
      return(ERROR);
    } else {
      res[nbytes-1] = 0;  /* remove newline */
      res[nbytes] = 0;
    }

    /* Check for undefined variable */
    if(strstr(res,"Undefined") != NULL) {
      printf("%s\n",res);
      close(rfd);
      return(ERROR);
    }

    /* Create string for putenv and set */
    sprintf(envS,"%s=%s",envName,res); 
    stat = putenv(envS);
    if(stat) {
      printf("envGetRemote: ERROR in putenv()\n");
      close(rfd);
      return(ERROR);
    } else {
      printf("Set %s to %s\n",envName,res);
    }

  } else {  /* Print the error */
    printf("envGetRemote: ERROR: ");
    printErrno(errno);
    return(ERROR);
  }

  /* Close file descriptor */
  close(rfd);

  return(OK);

}
