/**************************************************************************
 * Copyright (c)   2001    Southeastern Universities Research Association,
 *                         Thomas Jefferson National Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
 **************************************************************************
 *
 * Description:
 *      Selection Message Dialog Box
 *
 * Author:  
 *      Jie Chen
 *      Jefferson Lab HPC Group
 *
 * Revision History:
 *   $Log: SelectionMsgDialog.java,v $
 *   Revision 1.1  2001/10/18 18:45:43  chen
 *   First version of Jcmlog
 *
 *
 *
 */
package cmlog.gui;

import java.io.*;
import java.net.*;
import java.text.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.table.*;

import cmlog.*;

public class SelectionMsgDialog extends JDialog implements ActionListener
{
    /**
     * And/OR operator
     */
    public static final int AND_OP = 0;
    public static final int OR_OP  = 1;

    public static final String[] OP_REP = {"and", "or"};
    public static final String[] OP_MSG = {"&&",  "||"};

    public static final String[] LOGIC_REP = {"=", "!=", "has",
					      "<", ">", "<=", ">="};
    public static final String[] LOGIC_MSG = {"==", "!=", "like",
					      "<", ">", "<=", ">="};

    /**
     * Internal single selection message for a tag
     */
    class SingleSelMsg
    {
	/**
	 * Title
	 */
	private String title_;

	/**
	 * Tag
	 */
	private String tag_;


	/**
	 * Option for operator between messages (AND/OR)
	 */
	private JComboBox op_ = null;

	/**
	 * Option for logic operators within a message.
	 */
	private JComboBox logic_ = null;

	/**
	 * Check box for this message.
	 */
	private JCheckBox active_ = null;

	/**
	 * Text field for value.
	 */
	private JTextField input_ = null;

	/**
	 * JPanel holding all the above.
	 */
	private JPanel panel_ = null;

	
	public SingleSelMsg (String title, String tag,
			     int i, JPanel tpanel, 
			     GridBagLayout bl,
			     GridBagConstraints c)
	{
	    title_ = title;
	    tag_ = tag;

	    // create check box
	    c.fill = GridBagConstraints.BOTH;
	    c.gridx = 0;
	    c.gridy = 2*i;
	    c.ipadx = 4;
	    c.gridwidth = 1;
	    c.gridheight = 1;
	    c.weightx = 0;
	    c.weighty = 0;
	    active_ = new JCheckBox ();
	    tpanel.add (active_);
	    bl.setConstraints (active_, c);


	    // create Title label
	    c.gridx = 1;
	    c.gridwidth = 2;
	    c.gridheight = 1;
	    JLabel tlabel = new JLabel (title_);
	    tpanel.add (tlabel);
	    bl.setConstraints (tlabel, c);

	    // create combo box for logic operators.
	    c.gridx = 4;
	    c.gridwidth = 1;
	    c.gridheight = 1;
	    logic_ = new JComboBox (SelectionMsgDialog.LOGIC_REP);
	    logic_.setEditable (false);
	    tpanel.add (logic_);
	    bl.setConstraints (logic_, c); 

	    // create text field
	    c.gridx = 5;
	    c.gridwidth = GridBagConstraints.REMAINDER;
	    c.gridheight = 1;
	    input_ = new JTextField (24);
	    tpanel.add (input_);
	    bl.setConstraints (input_, c);


	    // add operator
	    c.gridx = 1;
	    c.gridy = 2*i + 1;
	    c.gridwidth =  1;
	    c.gridheight = 1;
	    op_ = new JComboBox (SelectionMsgDialog.OP_REP);
	    op_.setEditable (false);
	    tpanel.add (op_);
	    bl.setConstraints (op_, c);
	}

	/**
	 * Return panel
	 */
	public JPanel getPanel ()
	{
	    return panel_;
	}

	/**
	 * Return whether this single message is selected.
	 */
	public boolean isSelected ()
	{
	    return active_.isSelected ();
	}

	/**
	 * Check whether input has value in it.
	 */
	public boolean hasInput ()
	{
	    String val = input_.getText ();
	    
	    if (val == null)
		return false;

	    val = val.trim ();
	    if (val.length() == 0)
		return false;

	    return true;
	}

	/**
	 * Return single message.
	 */
	public String message ()
	{
	    if (active_.isSelected () == false)
		return null;

	    StringBuffer buf = new StringBuffer ();

	    // tag
	    buf.append (tag_);
	    buf.append (" ");

	    // get logic operator
	    int    idx = logic_.getSelectedIndex();
	    String lop = SelectionMsgDialog.LOGIC_MSG[idx];
	    buf.append (lop);
	    buf.append (" ");

	    // get text field
	    String val = input_.getText ();
	    boolean valnum = true;
	    if (val != null) {
		val = val.trim ();
		if (val.length() > 0) {
		    // check whether this input is a number
		    DecimalFormat fm = new DecimalFormat ();
		    try {
			fm.parse (val);
		    }catch (Exception e) {
			valnum = false;
		    }

		    if (valnum == true)
			buf.append (val);
		    else
			buf.append ("'" + val + "'");
		}
	    }
	    
	    return new String(buf);
	}

	/**
	 * Return and/or operator.
	 */
	public String intraOperator ()
	{
	    int idx = op_.getSelectedIndex ();
	    return SelectionMsgDialog.OP_MSG[idx];
	}
    }

    /**
     * Internal ok button.
     */
    private JButton ok_;

    /**
     * Internal cancel button.
     */
    private JButton cancel_;

    /**
     * List of single messages.
     */
    private SingleSelMsg[] smsgs_ = null;

    /**
     * Display for this dialog.
     */
    private Disp disp_ = null;

    /**
     * Text field to display the final selection message.
     */
    private JTextField output_ = null;

    /**
     * Constructor.
     */
    public SelectionMsgDialog (Disp disp, JTextField output)
    {
	super (disp.getParentWindow(),
	       "CMLOG Selection Message", true);

	int i;

	disp_ = disp;
	output_ = output;

	JPanel manager = new JPanel ();
	manager.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));
	manager.setLayout(new BorderLayout(10, 10));

	Config configobj = disp_.getConfigObject ();
	int numcol = configobj.getNumberOfColumns ();
	String[] titles = configobj.getColumnTitles ();
	String[] tags = configobj.getColumnTags ();

	// create top panel holding all single messages
	GridBagLayout bl = new GridBagLayout();
	GridBagConstraints c = new GridBagConstraints();
	JPanel tpanel = new JPanel (bl);
	tpanel.setBorder(BorderFactory.createEmptyBorder(10,10,10,10));

	smsgs_ = new SingleSelMsg[numcol];
	for (i = 0; i < numcol; i++) 
	    smsgs_[i] = new SingleSelMsg (titles[i], tags[i], i, 
					  tpanel, bl, c);

	manager.add (tpanel, BorderLayout.CENTER);

	// create button panel
	GridLayout gl = new GridLayout (1, 2, 40, 0);
	JPanel bpanel = new JPanel (gl);
	
	// create ok button
	ok_ = new JButton ("Ok");
	bpanel.add (ok_);

	// create cancel button 
	cancel_ = new JButton ("Cancel");
	bpanel.add (cancel_);

	// add action listener
	ok_.addActionListener (this);
	cancel_.addActionListener (this);

	manager.add (bpanel, BorderLayout.SOUTH);

	// put panel into content pane
	Container cpn = getContentPane ();
	cpn.add (manager);

	getRootPane().setDefaultButton(ok_);
    }


    /**
     * Implements action listener for two buttons.
     */
    public void actionPerformed (ActionEvent e)
    {
	Object  source = null;
	int     i, j;
	int     prevsel = -1;
	StringBuffer msg = null;

	if ((source = e.getSource()) == ok_) {
	    for (i = 0; i < smsgs_.length; i++) {
		if (smsgs_[i].isSelected() == true &&
		    smsgs_[i].hasInput() == true) {
		    if (prevsel == -1) 
			msg = new StringBuffer (smsgs_[i].message());
		    else {
			// find out option dialog box associated with 
			// a previous selection kept in index prevsel
			msg.append (smsgs_[prevsel].intraOperator());
			msg.append (smsgs_[i].message());
		    }
		    prevsel = i;
		}
	    }
	    if (msg != null)
		output_.setText (new String (msg));
	}			
	hide ();
    }
}
