/**************************************************************************
 * Copyright (c)   2001    Southeastern Universities Research Association,
 *                         Thomas Jefferson National Accelerator Facility
 *
 * This software was developed under a United States Government license
 * described in the NOTICE file included as part of this distribution.
 *
 * Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
 **************************************************************************
 *
 * Description:
 *      Data Handler for cmlog browser java display
 *
 * Author:  
 *      Jie Chen
 *      Jefferson Lab HPC Group
 *
 * Revision History:
 *   $Log: DataHandler.java,v $
 *   Revision 1.1  2001/10/18 18:45:41  chen
 *   First version of Jcmlog
 *
 *
 *
 */
package cmlog.gui;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.table.*;

import cmlog.*;

public class DataHandler extends AbstractTableModel implements BrowserDisconnectEventListener, BrowserMonitorEventListener, BrowserQueryEventListener
{
    /**
     * Property or environment variable to use to connect to the server.
     */
    private static final String hostEnv_ = "CMLOG_HOST";
    private static final String portEnv_ = "CMLOG_PORT";
    /**
     * Empty String for initial display.
     */
    private static final String empty_ = " ";

    /**
     * Type of this data model.
     */
    private int    type_ = Disp.IDLE;

    /**
     * Current data poll of cdevData.
     */
    private java.util.Vector   data_ = null;

    /**
     * Current beginning of data poll.
     */
    private int    first_ = 0;

    /**
     * Maximum number of data to keep.
     * If bufsize_ == 0, keep everything
     */
    private int    bufsize_ = 1024;

    /**
     * Number of column.
     */
    private int    numcol_ = 0;

    /**
     * Initial number of rows.
     */
    private int    initnumrow_ = 0;


    /**
     * Titles of this table.
     */
    private String[] titles_ = null;

    /**
     * Tags of this table.
     */
    private String[] tags_ = null;

    /**
     * Network Handler of cmlog.
     */
    Browser          netHandle_ = null;

    /**
     * Display associated with this data handler.
     */
    Disp             disp_ = null;

    /**
     * Number of messages found so far for a query process.
     */
    int              numfound_ = 0;


    /**
     * Query event.
     */
    BrowserEvent     qevent_ = null;

    /**
     * Current query argument.
     */
    QueryArgument    qarg_ = null;

    /**
     * Time stamp of last data received by query.
     */
    double           timeStamp_ = (double)0.0;


    /**
     * Date Format string.
     */
    private static final String formatString_ = "EEE MMM d hh:mm:ss yyyy";

    /**
     * Formatter for a Date object.
     */
    private SimpleDateFormat formatter_ = new SimpleDateFormat(formatString_);


    /**
     * A date object to convert time stamp.
     */
    private Date         dobj_ = new Date ();
    

    /**
     * Constructor.
     *
     * @param numcol number of columns of the table.
     * @param ininumrow initial number of rows of the table
     * @param titles titles of this table.
     * @param tags   correspong tags of this table.
     * @param bufsize how many data to keep: -1 keep all
     */
    public DataHandler (int numcol, int initnumrow,
			String[] titles, String[] tags,
			int bufsize, int type,
			Disp disp)
    {
	super ();
	int i;

	numcol_ = numcol;
	initnumrow_ = initnumrow;
	bufsize_ = bufsize;
	type_ = type;
	disp_ = disp;
	
	data_ = new Vector (bufsize_);

	titles_ = new String[titles.length];
	tags_ = new String[tags.length];

	for (i = 0; i < titles.length; i++)
	    titles_[i] = titles[i];

	for (i = 0; i < tags.length; i++)
	    tags_[i] = tags[i];

	netHandle_ = new Browser ();

    }


    /**
     * Implement getRowCount function defined by the parent class.
     */
    public int getRowCount()
    {
	if (data_.size() - first_ <= initnumrow_)
	    return initnumrow_;
	return data_.size () - first_;
    }


    /**
     * Implement getColumnCount function defined by the parent class.
     */
    public int getColumnCount()
    {
	return numcol_;
    }

    /**
     * Implement getValueAt function defined by the parent class.
     */
    public Object getValueAt(int row, int column)
    {
	if (row >= data_.size() - first_)
	    return DataHandler.empty_;

	// get cdevData at this row
	Vector   poll = (Vector)data_;
	cdevData rdata = (cdevData)poll.elementAt (row + first_);

	// retrieve a cdevDataEntry
	cdevDataEntry entry = null;
	String        ret   = null;

	try {
	    entry = rdata.get (tags_[column]);
	}catch (Exception e) {
	    ret = "N/A";
	    return ret;
	}
	
	if (entry != null) {
	    if (tags_[column].equals ("cmlogTime") == false)
		ret = entry.stringValue();
	    else {
		double ts = entry.doubleValue ();
		dobj_.setTime ((long)(ts*1000));
		ret = formatter_.format (dobj_);
	    }
	}		
	else
	    ret = "N/A";

	return ret;
    }

    /**
     * Implementation of getColumnName defined by the parent class.
     */
    public String getColumnName(int col) 
    {
	return titles_[col];
    }

    /**
     * Add a single cdevData into the pool.
     */
    public void addUpdatingData (cdevData data)
    {
	int row, i;

	if (bufsize_ == 0) {
	    // keep everything
	    row = data_.size ();
	    data_.add (data);
	    if (row < initnumrow_) // we are in the beginning
		fireTableRowsUpdated (row, row);
	    else
		fireTableRowsInserted(row, row);
	}
	else {
	    if (data_.size () - first_ < bufsize_) {
		row = data_.size () - first_;
		data_.add (data);
		if (row < initnumrow_) // we are in the beginning
		    fireTableRowsUpdated (row, row);
		else
		    fireTableRowsInserted(row, row);
	    }
	    else {
		first_ ++;
		row = data_.size () - first_;
		data_.add (data);


		if (first_ == bufsize_ - 1) {
		    // shrink the vector to bufsize_
		    // could not use sublist due to Java error
		    Vector newdata = new Vector (2*bufsize_);
		    for (i = 0; i < bufsize_ - 1; i++)
			newdata.add (i, data_.elementAt(bufsize_ + i));
		    
		    data_ = newdata;
		    first_ = 0;
		}

		fireTableRowsDeleted(0, 0);
		fireTableRowsInserted(row, row);

	    }
	}
	disp_.scrollTableToBottom (row);
    }


    /**
     * Add a set of messages into the poll
     */
    public void addQueryData (cdevMessage[] msgs, int num)
    {
	cdevData data = null;
	cdevDataEntry de = null;
	int row, i;

	row = data_.size ();
	for (i = 0; i < num; i++) {
	    numfound_ ++;
	    data = msgs[i].getData ();
	    
	    // get key value 
	    de = data.get (cmlog.Config.KEY_TAG);
	    if (de != null)
		timeStamp_ = de.doubleValue();
	    
	    data_.add (data);
	}

	if (row < initnumrow_)  {
	    if (row + num < initnumrow_)
		fireTableRowsUpdated (row, row + num - 1);
	    else {
		fireTableRowsUpdated (row, initnumrow_ - 1);		
		fireTableRowsInserted(initnumrow_, row + num - 1);
	    }
	}
	else
	    fireTableRowsInserted(row, row + num - 1);
    }

    /**
     * Get network handler (Browser)
     */
    public Browser getCmlogBrowser ()
    {
	return netHandle_;
    }

    /**
     * Connect to a cmlog server specified by two environment
     * Variables.
     */
    public void connect (String host, int port) 
	throws UnknownHostException, IOException
    {
	netHandle_.connect (host, port);

	netHandle_.addDisconnectEventListener (this);
    }

    /**
     * Start monitoring incoming messages
     */
    public boolean startUpdating (String selection)
    {
	boolean status = true;
	cdevData outbound = null;
	
	if (selection != null) {
	    outbound = new cdevData ();
	    outbound.insert (cmlog.Config.QUERYMSG_TAG, selection);
	}
	
	status = netHandle_.addMonitorEventListener (outbound, this);
	return status;
    }

    /**
     * Stop monitoring incoming messages
     */
    public boolean stopUpdating ()
    {
	boolean status = true;
	
	status = netHandle_.removeMonitorEventListener (null, this);
	return status;
    }

    /**
     * Close connection.
     */
    public void close () throws IOException
    {
	if (netHandle_.connected () == true)
	    netHandle_.disconnect ();
    }

    /**
     * Implementation of disconnection event listener
     */
    public void handleDisconnectEvent (BrowserEvent event)
    {
	disp_.popupWarningDialog ("The cmlog server closed this connection due to wrong protocol version or network failure.");
	disp_.getParentWindow().setDisconnectedState ();
    }

    /**
     * Handle monitor event from server.
     */
    public void handleMonitorEvent (BrowserEvent event)
    {
	cdevData data;
	int i;
	int nmsgs = 0;
	cdevDataEntry de;
	
	if (event.status == cmlog.Config.CMLOG_SUCCESS) {
	    cdevMessage[] msgs = event.packet.messages();
	    nmsgs = event.packet.numberOfMessages ();
	    for (i = 0; i < nmsgs; i++) {
		data = msgs[i].getData ();
		addUpdatingData (data);
	    }
	}
	else if (event.status == cmlog.Config.CMLOG_CBK_FINISHED)
	    ;
    }

    /**
     * Handle query event from the server
     */
    public void handleQueryEvent (BrowserEvent event)
    {
	cdevData data;
	int i;
	int nmsgs = 0;
	cdevDataEntry de;
	boolean querydone = false;

	if (event.status == cmlog.Config.CMLOG_ERROR ||
	    event.status == cmlog.Config.CMLOG_NOTFOUND) {
	    // finished with error or not found anything
	    disp_.setType (Disp.IDLE);

	    if (numfound_ > 0) 
		disp_.popupErrorDialog ("Found no more messages: " + String.valueOf (numfound_) + " messages have been found.");

	    else 
		disp_.popupErrorDialog ("Cannot find anything !!!");
	
	    numfound_ = 0;
	    querydone = true;
	}
	else if (event.status == cmlog.Config.CMLOG_INCOMPLETE) 
	    addQueryData (event.packet.messages(), 
			  event.packet.numberOfMessages ());
	else if (event.status == cmlog.Config.CMLOG_PAUSED) {
	    // reset query in progress flag
	    disp_.setType (Disp.IDLE);

	    // get time stamp information
	    getQueryMiscInfo (event.packet.messages(), 
			      event.packet.numberOfMessages());
	    
	    String msg = null;
	    if (numfound_ > 0)
		msg = "Found " + String.valueOf (numfound_) + " items, do you want to continue?";
	    else
		msg = "Found nothing so far, do you want to continue?";

	    // set end time to the display window
	    Date end = new Date();
	    end.setTime ((long)timeStamp_*1000);
	    disp_.getParentWindow().setEndTime (end);
	    
	    querydone = true;

	    // popup continuing dialog
	    disp_.popupQueryProgDialog (msg, timeStamp_, qarg_);
	}
	else if (event.status == cmlog.Config.CMLOG_QUERYMSG_ERR) {
	    disp_.setType (Disp.IDLE);

	    numfound_ = 0;
	    querydone = true;

	    disp_.popupErrorDialog ("Query Message Syntax Error.");
	}
	else {
	    disp_.setType (Disp.IDLE);

	    String msg;
	    if (numfound_ > 0)
		msg = "Query Process finished: found " + String.valueOf (numfound_) + " items.";
	    else
		msg = "Query Process finished: found nothing";

	    numfound_ = 0;
	    querydone = true;

	    // set end time to the display window
	    Date end = new Date();
	    end.setTime ((long)timeStamp_*1000);
	    disp_.getParentWindow().setEndTime (end);

	    disp_.popupInfoDialog (msg);
	}
	
	if (querydone == true)
	    disp_.getParentWindow().disableStopButton();
    }

    /**
     * get time stamp information
     */
    public void getQueryMiscInfo (cdevMessage[] msgs,
				  int num)
    {
	cdevData data = msgs[0].getData();
	cdevDataEntry de = data.get (cmlog.Config.KEY_TAG);

	if (de != null)
	    timeStamp_ = de.doubleValue();
    }
	

    /**
     * Remove all elements.
     */
    public void removeAll ()
    {
	first_ = 0;
	data_.clear();
	fireTableDataChanged();
    }

    /**
     * Start a querying process.
     */
    public boolean doQuery (Date from, Date to,
			    int  numitems, boolean getall,
			    String qstr)
    {
	// create query data containing start and end time
	int      ssec = (int)(from.getTime()/1000);
	int      esec = (int)(to.getTime()/1000);

	cdevData qdata = new cdevData();

	// insert start/end time
	qdata.insert (cmlog.Config.START_TAG, ssec);
	qdata.insert (cmlog.Config.END_TAG,   esec);

	// number of items
	if (getall == false)
	    qdata.insert (cmlog.Config.NUMITEMS_TAG, numitems);
	
	if (qstr != null)
	    qdata.insert (cmlog.Config.QUERYMSG_TAG, qstr);

	try {
	    qevent_ = netHandle_.query (qdata, this);
	}catch (IllegalArgumentException e) {
	    System.err.println (e.getMessage());
	    return false;
	}catch (IOException ie) {
	    System.err.println (ie.getMessage());
	    return false;
	}

	// update query argument we are using now
	qarg_ = new QueryArgument (from, to, numitems, getall, qstr);

	// if get all is true, enable stop button
	disp_.getParentWindow().enableStopButton();

	return true;
    }

    /**
     * Stop a query 
     */
    public void stopQuery ()
    {
	if (qevent_ != null) {
	    try {
		netHandle_.stopQuery (null, qevent_);
	    }catch (Exception e) {
		disp_.popupErrorDialog ("Could not stop current query process.");
	    }
	    qevent_ = null;
	}
    }
}

		
    

	
	








