//-----------------------------------------------------------------------------
// Copyright (c) 1994,1995 Southeastern Universities Research Association,
//                         Continuous Electron Beam Accelerator Facility
//
// This software was developed under a United States Government license
// described in the NOTICE file included as part of this distribution.
//
// Jefferson Lab HPC Group, 12000 Jefferson Ave., Newport News, VA 23606
//-----------------------------------------------------------------------------
//
// Description:
//      Generic CDEV Message Protocol
//
// Author:  
//      Jie Chen
//      Jefferson Lab HPC Group
//
// Revision History:
//   $Log: cdevMessage.java,v $
//   Revision 1.1  1999/12/14 15:31:38  chen
//   initial java implementation
//
//
//
package cmlog;

import java.io.*;
import java.math.*;

public class cdevMessage
{
    private short    clientID_ = -1;
    private int      transIndex_ = 0;
    private int      cancelTransIndex_ = 0;
    private int      localDataIndex_ = 0;
    private int      foreignDataIndex_ = 0;
    private int      operationCode_ = 0;
    private int      deviceCount_ = 0;
    private String[] deviceList_= null;
    private String   message_ = null;
    private cdevData data_ = null;
    private cdevData context_ = null;
    private cdevData tagMap_ = null;

    /*
     * Integer mask to denote which part of message is set
     */
    private static final int _CLIP_VERSION          = 0xffff0000;
    private static final int _CLIP_IDSET            = 0x00008000;
    private static final int _CLIP_TRINDEXSET       = 0x00004000;
    private static final int _CLIP_CTRINDEXSET      = 0x00002000;
    private static final int _CLIP_LOCALDINDEXSET   = 0x00001000;
    private static final int _CLIP_FOREIGNDINDEXSET = 0x00000800;
    private static final int _CLIP_OPCODESET        = 0x00000400;
    private static final int _CLIP_DEVLISTSET       = 0x00000200;
    private static final int _CLIP_MSGSET           = 0x00000100;
    private static final int _CLIP_DATASET          = 0x00000080;
    private static final int _CLIP_CONTEXTSET       = 0x00000040;
    private static final int _CLIP_TAGMAPSET        = 0x00000020;

    /* Version of this set of protocol */
    private static final int CLIP_CURRENT_VERSION   = 0x00010000; 


    /**
     * Default construct for cdevMessage
     */
    public cdevMessage ()
    {
	clientID_ = -1;
	transIndex_ = 0;
	cancelTransIndex_ = 0;
	localDataIndex_ = 0;
	foreignDataIndex_ = 0;
	operationCode_ = 0;
	deviceCount_ = 0;
	deviceList_ = null;
	message_ = null;
	data_ = null;
	context_ = null;
	tagMap_ = null;
    }

    /**
     * Construct a cdevMessage from variaous parameters
     */
    public cdevMessage (short ClientID, 
			int TransIndex,
			int CancelTransIndex,
			int LocalDataIndex,
			int ForeignDataIndex,
			int OperationCode,
			String[] DeviceList,
			String Message,
			cdevData Data,
			cdevData Context,
			cdevData TagMap)
    {
	clientID_ = ClientID;
	transIndex_ = TransIndex;
	cancelTransIndex_ = CancelTransIndex;
	localDataIndex_ = LocalDataIndex;
	foreignDataIndex_ = ForeignDataIndex;
	operationCode_ = OperationCode;
	
	if (DeviceList != null && DeviceList.length > 0) {
	    deviceCount_ = DeviceList.length;
	    deviceList_ = new String[deviceCount_];
	    for (int i = 0; i < deviceCount_; i++)
		deviceList_[i] = DeviceList[i];
	}
	else {
	    deviceList_ = null;
	    deviceCount_ = 0;
	}
	
	message_ = Message;
	data_ = Data;
	context_ = Context;
	tagMap_ = TagMap;
    }

    /**
     * Create a new message with an existing message.
     */
    public cdevMessage (cdevMessage msg)
    {
	clientID_ = msg.clientID_;
	transIndex_ = msg.transIndex_;
	cancelTransIndex_ = msg.cancelTransIndex_;
	localDataIndex_ = msg.localDataIndex_;
	foreignDataIndex_ = msg.foreignDataIndex_;
	operationCode_ = msg.operationCode_;

	deviceCount_ = msg.deviceCount_;	
	if (msg.deviceCount_ > 0) {
	    deviceList_ = new String[deviceCount_];
	    for (int i = 0; i < deviceCount_; i++) 
		deviceList_[i] = new String(msg.deviceList_[i]);
	}
	else {
	    deviceList_ = null;
	}

	if (msg.message_ != null)
	    message_ = new String (msg.message_);
	else
	    message_ = null;

	if (msg.data_ != null)
	    data_ = new cdevData (msg.data_);
	else
	    data_ = null;

	if (msg.context_ != null)
	    context_ = new cdevData (msg.context_);
	else
	    context_ = null;

	if (msg.tagMap_ != null)
	    tagMap_ = new cdevData (msg.tagMap_);
	else
	    tagMap_ = null;
    }
	    
    /**
     * Set message data members into default values
     */
    public void clear ()
    {
	clientID_ = -1;
	transIndex_ = 0;
	cancelTransIndex_ = 0;
	localDataIndex_ = 0;
	foreignDataIndex_ = 0;
	operationCode_ = 0;
	deviceCount_ = 0;
	deviceList_ = null;
	message_ = null;
	data_ = null;
	context_ = null;
	tagMap_ = null;
	/* No memort recollection :-( */
    } 
    
    /**
     * Return binary stream size of this message
     */
    public int streamSize ()
    {
	int datasize = 0;
	int map = 0;
	int tmp = 0;
	int dsize = 0;
	int cxtsize = 0;
	int tagmapsize = 0;

	map |= CLIP_CURRENT_VERSION;

	if(clientID_   > 0)                    map |= _CLIP_IDSET;
	if(transIndex_ > 0)                    map |= _CLIP_TRINDEXSET;
	if(cancelTransIndex_ > 0)              map |= _CLIP_CTRINDEXSET;
	if(localDataIndex_ > 0)                map |= _CLIP_LOCALDINDEXSET;
	if(foreignDataIndex_ > 0)              map |= _CLIP_FOREIGNDINDEXSET;
	if(operationCode_ > 0)                 map |= _CLIP_OPCODESET;
	if(deviceCount_>0 && deviceList_!=null)map |= _CLIP_DEVLISTSET;
	if(message_!=null)                     map |= _CLIP_MSGSET;
	if(data_ != null &&  data_.numValidEntries() > 0) {
	    map |= _CLIP_DATASET;
	    dsize = data_.streamSize ();
	}
	if (context_ != null && context_.numValidEntries () > 0) {
	    map |= _CLIP_CONTEXTSET;
	    cxtsize = context_.streamSize ();
	}
	if (tagMap_ != null && tagMap_.numValidEntries () > 0){
	    map |= _CLIP_TAGMAPSET;
	    tagmapsize = tagMap_.streamSize ();
	}
	
	// integer bit mask
	datasize += cdevDataOutputStream.streamSize (map);

	if ((tmp = map & _CLIP_IDSET) != 0) 
	    datasize += cdevDataOutputStream.streamSize (clientID_);
	if ((tmp = map & _CLIP_TRINDEXSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (transIndex_);
	if ((tmp = map & _CLIP_CTRINDEXSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (cancelTransIndex_);
	if ((tmp = map & _CLIP_LOCALDINDEXSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (localDataIndex_);
	if ((tmp = map & _CLIP_FOREIGNDINDEXSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (foreignDataIndex_);
	if ((tmp = map & _CLIP_OPCODESET) != 0)
	    datasize += cdevDataOutputStream.streamSize (operationCode_);
	if ((tmp = map & _CLIP_DEVLISTSET) != 0) {
	    datasize += cdevDataOutputStream.streamSize (deviceCount_);
	    for (int i = 0; i < deviceCount_; i++) 
		datasize += cdevDataOutputStream.streamSize (deviceList_[i]);
	}
	if ((tmp = map & _CLIP_MSGSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (message_);
	if ((tmp = map & _CLIP_DATASET) != 0)
	    datasize += (cdevDataOutputStream._CDEV_STREAM_BYTE_UNIT + dsize);
	if ((tmp = map & _CLIP_CONTEXTSET) != 0)
	    datasize += (cdevDataOutputStream._CDEV_STREAM_BYTE_UNIT+cxtsize);
	if ((tmp = map & _CLIP_TAGMAPSET) != 0)
	    datasize +=(cdevDataOutputStream._CDEV_STREAM_BYTE_UNIT+tagmapsize);
	return datasize;
    }

    /**
     * Convert a message into a binary output stream
     */
    public void streamOut (OutputStream output) throws IOException
    {
	cdevDataOutputStream writer = new cdevDataOutputStream (output);

	try {
	    streamOut (writer);
	}catch (IOException e) {
	    throw e;
	}
    }

	    
    /**
     * Convert a message into a binary data stream
     */
    public void streamOut (cdevDataOutputStream writer) throws IOException
    {
	int datasize = 0;
	int map = 0;
	int tmp = 0;
	int dsize = 0;
	int cxtsize = 0;
	int tagmapsize = 0;

	map |= CLIP_CURRENT_VERSION;

	if(clientID_   > 0)                    map |= _CLIP_IDSET;
	if(transIndex_ > 0)                    map |= _CLIP_TRINDEXSET;
	if(cancelTransIndex_ > 0)              map |= _CLIP_CTRINDEXSET;
	if(localDataIndex_ > 0)                map |= _CLIP_LOCALDINDEXSET;
	if(foreignDataIndex_ > 0)              map |= _CLIP_FOREIGNDINDEXSET;
	if(operationCode_ > 0)                 map |= _CLIP_OPCODESET;
	if(deviceCount_>0 && deviceList_!=null)map |= _CLIP_DEVLISTSET;
	if(message_!=null)                     map |= _CLIP_MSGSET;
	if(data_ != null &&  data_.numValidEntries() > 0) {
	    map |= _CLIP_DATASET;
	    dsize = data_.streamSize ();
	}
	if (context_ != null && context_.numValidEntries () > 0) {
	    map |= _CLIP_CONTEXTSET;
	    cxtsize = context_.streamSize ();
	}
	if (tagMap_ != null && tagMap_.numValidEntries () > 0) {
	    map |= _CLIP_TAGMAPSET;
	    tagmapsize = tagMap_.streamSize ();
	}

	// get binary stream size first
	// integer bit mask
	datasize += cdevDataOutputStream.streamSize (map);

	if ((tmp = map & _CLIP_IDSET) != 0) 
	    datasize += cdevDataOutputStream.streamSize (clientID_);
	if ((tmp = map & _CLIP_TRINDEXSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (transIndex_);
	if ((tmp = map & _CLIP_CTRINDEXSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (cancelTransIndex_);
	if ((tmp = map & _CLIP_LOCALDINDEXSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (localDataIndex_);
	if ((tmp = map & _CLIP_FOREIGNDINDEXSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (foreignDataIndex_);
	if ((tmp = map & _CLIP_OPCODESET) != 0)
	    datasize += cdevDataOutputStream.streamSize (operationCode_);
	if ((tmp = map & _CLIP_DEVLISTSET) != 0) {
	    datasize += cdevDataOutputStream.streamSize (deviceCount_);
	    for (int i = 0; i < deviceCount_; i++) 
		datasize += cdevDataOutputStream.streamSize (deviceList_[i]);
	}
	if ((tmp = map & _CLIP_MSGSET) != 0)
	    datasize += cdevDataOutputStream.streamSize (message_);
	if ((tmp = map & _CLIP_DATASET) != 0)
	    datasize += (cdevDataOutputStream._CDEV_STREAM_BYTE_UNIT 
			 + dsize);
	if ((tmp = map & _CLIP_CONTEXTSET) != 0)
	    datasize += (cdevDataOutputStream._CDEV_STREAM_BYTE_UNIT 
			 + cxtsize);
	if ((tmp = map & _CLIP_TAGMAPSET) != 0)
	    datasize += (cdevDataOutputStream._CDEV_STREAM_BYTE_UNIT 
			 + tagmapsize);


	// write out individual piece
	// write out map information first
	try {
	    writer.write (map);
	}catch (IOException e) {
	    throw e;
	}

	// write out client id information
	if ((tmp = map & _CLIP_IDSET) != 0) {
	    try {
		writer.write (clientID_);
	    }catch (IOException e) {
		throw e;
	    }
	}
	
	// write out transaction index information
	if ((tmp = map & _CLIP_TRINDEXSET) != 0) {
	    try {
		writer.write (transIndex_);
	    }catch (IOException e) {
		throw e;
	    }
	}

	// write out cancel transaction index information
	if ((tmp = map & _CLIP_CTRINDEXSET) != 0) {
	    try {
		writer.write (cancelTransIndex_);
	    }catch (IOException e) {
		throw e;
	    }
	}

	// write out local data index information
	if ((tmp = map & _CLIP_LOCALDINDEXSET) != 0) {
	    try {
		writer.write (localDataIndex_);
	    } catch (IOException e){
		throw e;
	    }
	}

	// write out foreign data index information
	if ((tmp = map & _CLIP_FOREIGNDINDEXSET) != 0) {
	    try {
		writer.write (foreignDataIndex_);
	    }catch (IOException e) {
		throw e;
	    }
	}

	// write out opcode information
	if ((tmp = map & _CLIP_OPCODESET) != 0) {
	    try {
		writer.write (operationCode_);
	    }catch (IOException e) {
		throw e;
	    }
	}

	// write out device list information
	if ((tmp = map & _CLIP_DEVLISTSET) != 0) {
	    try {
		writer.write (deviceCount_);
	    }catch (IOException e) {
		throw e;
	    }

	    for (int i = 0; i < deviceCount_; i++) {
		try {
		    writer.write (deviceList_[i]);
		}catch (IOException e) {
		    throw e;
		}
	    }
	}

	// write out message
	if ((tmp = map & _CLIP_MSGSET) != 0) {
	    try {
		writer.write (message_);
	    }catch (IOException e) {
		throw e;
	    }
	}

	// write data
	if ((tmp = map & _CLIP_DATASET) != 0) {
	    // writes out data size first
	    try {
		writer.write (dsize);
	    }catch (IOException e) {
		throw e;
	    }
	    try {
		data_.streamOut (writer);
	    }catch (IOException e) {
		throw e;
	    }
	}

	// write context
	if ((tmp = map & _CLIP_CONTEXTSET) != 0) {
	    // writes out data size first
	    try {
		writer.write (cxtsize);
	    }catch (IOException e) {
		throw e;
	    }
	    try {
		context_.streamOut (writer);
	    }catch (IOException e) {
		throw e;
	    }
	}

	// write tagmap
	if ((tmp = map & _CLIP_TAGMAPSET) != 0) {
	    // writes out data size first
	    try {
		writer.write (tagmapsize);
	    }catch (IOException e) {
		throw e;
	    }
	    try {
		tagMap_.streamOut (writer);
	    }catch (IOException e) {
		throw e;
	    }
	}
	
    }

    /**
     * Convert input stream into a cdevMessage after the datasize has 
     * been determined
     */
    public void streamIn (InputStream stream) throws IOException
    {
	// clean out this object first
	clear ();

	// attach input stream
	cdevDataInputStream input = new cdevDataInputStream (stream);

	try {
	    streamIn (input);
	}catch (IOException e) {
	    throw e;
	}
    }

    /**
     * Convert input stream into a cdevMessage after the datasize has 
     * been determined
     */
    public void streamIn (cdevDataInputStream input) throws IOException
    {
	int tmp = 0;
	int map = 0;
	// clear out old information
	clear ();

	try {
	    map = input.readInt ();
	}catch (IOException e) {
	    throw e;
	}

	int version = versionNum (map);
	if (version != 1) {
	    clear ();
	    throw new IOException ("Wrong clip version number: " + String.valueOf (version));
	}

	// get client id
	if ((tmp = map & _CLIP_IDSET) != 0) {
	    try {
		clientID_ = input.readShort ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}

	// get transaction index
	if ((tmp = map & _CLIP_TRINDEXSET) != 0) {
	    try {
		transIndex_ = input.readInt ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}
	
	// get cancel transaction index
	if ((tmp = map & _CLIP_CTRINDEXSET) != 0) {
	    try {
		cancelTransIndex_ = input.readInt ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}
	
	// get local data index
	if ((tmp = map & _CLIP_LOCALDINDEXSET) != 0) {
	    try {
		localDataIndex_ = input.readInt ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}

	// get foreign data index
	if ((tmp = map & _CLIP_FOREIGNDINDEXSET) != 0) {
	    try {
		foreignDataIndex_ = input.readInt ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}

	// get operation code
	if ((tmp = map & _CLIP_OPCODESET) != 0) {
	    try {
		operationCode_ = input.readInt ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}

	// get device list
	if ((tmp = map & _CLIP_DEVLISTSET) != 0) {
	    try {
		deviceCount_ = input.readInt();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	    
	    if (deviceCount_ <= 0) {
		int tmpcount = deviceCount_;
		clear ();
		throw new IOException ("Device Count <" + String.valueOf(tmpcount) + "> is incorrect");
	    }
	    
	    deviceList_ = new String[deviceCount_];
	    for (int i = 0; i < deviceCount_; i++) {
		try {
		    deviceList_[i] = input.readString ();
		}catch (IOException e) {
		    clear ();
		    throw e;
		}
	    }
	}

	// get message
	if ((tmp = map & _CLIP_MSGSET) != 0) {
	    try {
		message_ = input.readString ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}

	// get data
	if ((tmp = map & _CLIP_DATASET) != 0) {
	    // read data length in
	    int datalen;
	    try {
		datalen = input.readInt ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	    // read data
	    data_ = new cdevData();
	    try {
		data_.streamIn (input);
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}

	// get context data
	if ((tmp = map & _CLIP_CONTEXTSET) != 0) {
	    // read data length in
	    int datalen;
	    try {
		datalen = input.readInt ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	    // read context
	    context_ = new cdevData();
	    try {
		context_.streamIn (input);
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}

	// get tagmap
	if ((tmp = map & _CLIP_TAGMAPSET) != 0) {
	    // read data length in
	    int datalen;
	    try {
		datalen = input.readInt ();
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	    // read tagmap
	    tagMap_ = new cdevData();
	    try {
		tagMap_.streamIn (input);
	    }catch (IOException e) {
		clear ();
		throw e;
	    }
	}
    }

	    
    /**
     * Get Client ID of this message 
     */
    public short getClientID () 
    {
	return clientID_;
    }

    /**
     * Get transaction index of this message 
     */
    public int getTransIndex ()
    {
	return transIndex_;
    }
		
    /**
     * Get cancel transaction index of this message 
     */    
    public int getCancelTransIndex ()
    {
	return cancelTransIndex_;
    }

    /**
     * Get local data index of this message 
     */    
    public int getLocalDataIndex ()
    {
	return localDataIndex_;
    }

    /**
     * Get foreign data index of this message 
     */
    public int getForeignDataIndex ()
    {
	return foreignDataIndex_;
    }

    /**
     * Get operation code of this message 
     */
    public int getOperationCode ()
    {
	return operationCode_;
    }

    /**
     * Get message
     */
    public String getMessage ()
    {
	return message_;
    }

    /**
     * Get number of devices of this message 
     */
    public int getDeviceCount ()
    {
	return deviceCount_;
    }

    /**
     * Get list of devices of this message 
     */
    public String[] getDeviceList ()
    {
	return deviceList_;
    }

    /**
     * Get internal data of this message 
     */
    public cdevData getData ()
    {
	return data_;
    }

    /**
     * Get context of this message 
     */
    public cdevData getContext ()
    {
	return context_;
    }

    /**
     * Get tag map of this message 
     */
    public cdevData getTagMap ()
    {
	return tagMap_;
    }
    

    /**
     * Set Client ID with an input parameter
     */
    public void setClientID (short ClientID)
    {
	clientID_ = ClientID;
    }

    /**
     * Set transaction index with an input parameter
     */
    public void setTransIndex (int TransIndex)
    {
	transIndex_ = TransIndex;
    }

    /**
     * Set cancel transaction index with an input parameter
     */
    public void setCancelTransIndex (int CancelTransIndex)
    {
	cancelTransIndex_ = CancelTransIndex;
    }

    /**
     * Set local data index with an input parameter
     */
    public void setLocalDataIndex (int LocalDataIndex)
    {
	localDataIndex_ = LocalDataIndex;
    }

    /**
     * Set foreign data index with an input parameter
     */
    public void setForeignDataIndex (int fi)
    {
	foreignDataIndex_ = fi;
    }

    /**
     * Set operation code with an input parameter
     */
    public void setOperationCode (int opcode)
    {
	operationCode_ = opcode;
    }


    /**
     * Set devices list with an input parameter
     */
    public void setDeviceList (String[] devices)
    {
	if (devices != null && devices.length > 0) {
	    deviceCount_ = devices.length;
	    deviceList_ = new String[deviceCount_];
	    for (int i = 0; i < deviceCount_; i++) 
		deviceList_[i] = devices[i];
	}
	else {
	    deviceCount_ = 0;
	    deviceList_ = null;
	}
    }

    /**
     * Set message with an input parameter
     */
    public void setMessage (String Message)
    {
	message_ = Message;
    }

    /**
     * Set data with an input parameter
     */
    public void setData (cdevData data)
    {
	data_ = data;
    }

    /**
     * Set context with an input parameter
     */
    public void setContext (cdevData Context)
    {
	context_ = Context;
    }

    /**
     * Set tag map with an input parameter
     */
    public void setTagMap (cdevData tagmap)
    {
	tagMap_ = tagmap;
    }

    // retrieve clip version number
    private int versionNum (int a) 
    {
	int version = ((a & _CLIP_VERSION) >> 16) & 0x0000ffff;
	return version;
    }

}	    
    

